#include <QMaemo5ValueButton>
#include <QMaemo5ListPickSelector>
#include <QStandardItemModel>
#include <QPushButton>
#include <QListWidget>
#include <QLineEdit>
#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QFile>
#include <QAction>
#include <QMenu>
#include <QDebug>

#include "infoaccess.h"
#include "settings.h"
#include "region.h"
#include "favourite.h"

const int CHANNEL = 1;
const int SEARCH  = 2;

FavouriteWindow::FavouriteWindow(QMainWindow *parent) : SettingsWindow(parent)
{
	setWindowTitle("FAVOURITES");
	_mainwidget = new QWidget(this);

	loadSettings(); // load from the file what is there currently

	_regionListModel = new QStandardItemModel(this);
	_regionSelectorList = new QMaemo5ListPickSelector(this);
	_regionSelectorList->setModel(_regionListModel);

	_regionSelector = new QMaemo5ValueButton("PICK REGION", this);
	_regionSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
	_regionSelector->setPickSelector(_regionSelectorList);
	_regionSelector->setValueText(getRegion());
	_region = getRegion();

	_channelListModel = new QStandardItemModel(this);

	_channelSelector = new QMaemo5ValueButton("PICK CHANNELS", this);
	_channelSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);

	_searchListEdit = new QLineEdit(this);
	_searchListEdit->hide();

	_selectedListWidget = new QListWidget(this);	
	_searchListWidget = new QListWidget(this);	
	_searchListWidget->hide(); // shown only when search title setting is chosen for channel using context menu

	_backButton = new QPushButton("FAV CHANNEL LIST", this);
	_backButton->hide(); // shown only when search title setting is chosen for channel using context menu
	_cancelButton = new QPushButton("CANCEL", this);
	_saveButton = new QPushButton("SAVE", this);

	_h1layout = new QHBoxLayout; // layout for channel selectors
	_h2layout = new QHBoxLayout; // layout for the list
	_h3layout = new QHBoxLayout; // layout for the cancel/save button
	_v1layout = new QVBoxLayout; // layout for main widget

	_h1layout->addWidget(_regionSelector);
	_h1layout->addWidget(_channelSelector);
	_h1layout->addWidget(_searchListEdit);
	_h2layout->addWidget(_selectedListWidget);
	_h2layout->addWidget(_searchListWidget);
	_h3layout->addWidget(_backButton);
	_h3layout->addWidget(_cancelButton);
	_h3layout->addWidget(_saveButton);

	_v1layout->addLayout(_h1layout);
	_v1layout->addLayout(_h2layout);
	_v1layout->addLayout(_h3layout);
	
	_mainwidget->setLayout(_v1layout);

	setCentralWidget(_mainwidget);

	_a1 = new QAction(QString("delete channel"), this);
	_a2 = new QAction(QString("search title window"), this);

	initRegionListModel();

	connect(_regionSelectorList, SIGNAL(selected(QString)), this, SLOT(selectRegion(QString)));
	connect(_searchListEdit, SIGNAL(returnPressed()), this, SLOT(addSearchTitle()));
	connect(_backButton, SIGNAL(clicked()), this, SLOT(showSelectedListWidget()));
	connect(_cancelButton, SIGNAL(clicked()), this, SLOT(doCancel()));
	connect(_saveButton, SIGNAL(clicked()), this, SLOT(doSave()));
	//connect(_selectedListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(removeChannel(QListWidgetItem*)));
	connect(_selectedListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(showMenu(QListWidgetItem*)));
	connect(_searchListWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(removeSearchTitle(QListWidgetItem*)));
	connect(_a1, SIGNAL(triggered()), this, SLOT(removeSelectedChannel()));
	connect(_a2, SIGNAL(triggered()), this, SLOT(showSearchTitles()));

	selectRegion(_region);
}

void FavouriteWindow::showMenu(QListWidgetItem* item)
{
	_channel = item->text();

	QList<QAction*> actions;
	actions.append(_a1);
	actions.append(_a2);
	QMenu::exec(actions, QCursor::pos());
}

void FavouriteWindow::initRegionListModel()
{
	_regionListModel->clear();
	getSettingsMap().clear(); 

	RegionWindow *r = new RegionWindow(this);
	QStringList regionList = r->getRegionList();

	//////////////////////////////////////////////////////
	// REGION SELECTOR LIST 
	//////////////////////////////////////////////////////
	foreach(QString region, regionList)
	{
		QStandardItem *regionitem = new QStandardItem(region);
		regionitem->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
		regionitem->setEditable(false); // prevent editing of the item
		_regionListModel->appendRow(regionitem);
		
	}
}

void FavouriteWindow::initChannelListModel()
{
	//////////////////////////////////////////////////////
	// CHANNEL SELECTOR LIST 
	//////////////////////////////////////////////////////
	_channelListModel->clear();

	_channelSelectorList = new QMaemo5ListPickSelector(this);
	_channelSelectorList->setModel(_channelListModel);
	_channelSelector->setPickSelector(_channelSelectorList);
	connect(_channelSelectorList, SIGNAL(selected(QString)), this, SLOT(selectChannel(QString)));

	QStringList channelList = _infoapi->getChannelList();

	foreach(QString channel, channelList)
	{
		QStandardItem *item = new QStandardItem(channel);
		item->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
		item->setEditable(false); // prevent editing of the item
		_channelListModel->appendRow(item);
	}
}

void FavouriteWindow::selectRegion(QString region)
{
	_infoapi = new InfoAccess(this, region);
	_region = region;
	initChannelListModel();  // init the channel list for the selected region
	showSelectedListWidget(); // load the current favourites of the region selected
}

void FavouriteWindow::selectChannel(QString channel)
{
	//qDebug() << "Adding " << channel << " as your favourite";

	if ( getFavourites(_region).contains(channel) )
	{
		//qDebug() << channel << " is already a favourite";
		return;
	}
	else
		setFavourites(_region, channel);

	//qDebug() << getSettingsMap();

	_selectedListWidget->addItem(channel);
}

void FavouriteWindow::addSearchTitle()
{
	//qDebug() << "Adding " << _searchListEdit->text()<< " to your list";

	foreach(QString searchTitle, getSearchTitles(_region, _channel))
	{
		if ( searchTitle == _searchListEdit->text() )
		{
			//qDebug() << _searchListEdit->text() << " is already in the list";
			return;
		}
	}

	setSearchTitles(_region, _channel, _searchListEdit->text());

	//qDebug() << getSettingsMap();

	_searchListWidget->addItem(_searchListEdit->text());
	_searchListEdit->clear();
}

void FavouriteWindow::showSelectedListWidget()
{
	toggleList(CHANNEL);

	foreach(QString channel, getFavourites(_region))
		_selectedListWidget->addItem(channel);
}

void FavouriteWindow::removeSelectedChannel()
{
	//qDebug() << "Removing " << item->text() << " from favourites";

	removeFromFavourite(_region, _channel);
	_selectedListWidget->removeItemWidget(_selectedListWidget->currentItem());

	//qDebug() << getSettingsMap();

	showSelectedListWidget();
}

void FavouriteWindow::removeSearchTitle(QListWidgetItem *item)
{
	//qDebug() << "Removing " << item->text() << " from your list";

	removeFromSearchTitles(_region, _channel, item->text());
	_searchListWidget->removeItemWidget(item);

	//qDebug() << getSettingsMap();

	showSearchTitles();
}

void FavouriteWindow::showSearchTitles()
{
	toggleList(SEARCH);

	//qDebug() << getSearchTitles(_region, _channel);

	foreach(QString searchTitles, getSearchTitles(_region, _channel))
		_searchListWidget->addItem(searchTitles);
}

void FavouriteWindow::doSave()
{
	updateSettings();
	emit clicked(); // CLOSE THE WINDOW
}

void FavouriteWindow::doCancel()
{
	emit clicked(); // CLOSE THE WINDOW
}

void FavouriteWindow::toggleList(int listType)
{
	switch(listType)
	{
		case CHANNEL:
			_selectedListWidget->clear();
			_regionSelector->show();
			_channelSelector->show();
			_selectedListWidget->show();
			_searchListEdit->hide();
			_searchListWidget->hide();
			_backButton->hide();
			break;

		case SEARCH:
			_searchListWidget->clear();
			_searchListWidget->show();
			_searchListEdit->show();
			_backButton->show();
			_regionSelector->hide();
			_channelSelector->hide();
			_selectedListWidget->hide();
			break;

	}
}

FavouriteWindow::~FavouriteWindow()
{
}
