#include <QObject>
#include <QFile>
#include <QDir>
#include <QMaemo5InformationBox>
#include <QTextStream>
#include <QProcess>
#include <QDebug>
#include <QXmlStreamReader>

#include "datadownloader.h"

DataDownloader::DataDownloader(QObject *parent) : QObject(parent)
{
}

DataDownloader::DataDownloader(QObject *parent, const char* datapath) : QObject(parent), DATAPATH(datapath)
{
}

void DataDownloader::initSettings()
{
	_settingsFile->open(QIODevice::WriteOnly | QIODevice::ReadOnly);

	QTextStream stream(_settingsFile);

	// create the map of channel and date until which data is available
	while ( !stream.atEnd() )
	{
		QString line = stream.readLine();

		QString channel = line.section('#',1,1);	
		// dates are in this format ==>  #bbc1#01022010#02022010#05022010
		QStringList datelist = line.section('#',2).split('#');	

		_channelDateMap[channel] = QSet<QString>::fromList(datelist);
	}

	_settingsFile->close();
}

QString DataDownloader::finishedDownload(QString channel, QIODevice *source)
{
	return (QString());
}

QString DataDownloader::saveFile(QString channel, QIODevice *source)
{
	//qDebug() << "File has been downloaded for : " << channel;

	QString filedate;
	QString channelname;
	channelname = _channelCodeMap[channel];
 	if ( isSingleXMLTVFile() ) 
	{
		filedate = QDate::currentDate().toString("yyyyMMdd"); // store the download date for single file channels
	}
	else
	{
		filedate = _requestDate.toString("yyyyMMdd"); // store the request date for per day channels
	}

	QString filename = QString(QString(DATAPATH) + "%1--%2--.xml").arg(channelname).arg(filedate);
	QFile xmlFile(filename);
	xmlFile.open(QIODevice::WriteOnly);

	xmlFile.write(source->readAll());
	xmlFile.close();

	return filename;
}

QString DataDownloader::saveZipFile(QString channel, QIODevice *source)
{
	//qDebug() << "File has been downloaded for : " << channel;

	QString filedate;
 	if ( isSingleXMLTVFile() ) filedate = QDate::currentDate().toString("yyyyMMdd"); // store the download date for single file channels
	else					filedate = _requestDate.toString("yyyyMMdd"); // store the request date for per day channels

	QString channelname;
	channelname = _channelCodeMap[channel];

	QFile::remove(QString(QString(DATAPATH) + "%1--%2--.xml.gz").arg(channelname).arg(filedate));
	QFile::remove(QString(QString(DATAPATH) + "%1--%2--.xml").arg(channelname).arg(filedate));

	QString zipfile = QString(QString(DATAPATH) + "%1--%2--.xml.gz").arg(channelname).arg(filedate);
	QString filename = QString(QString(DATAPATH) + "%1--%2--.xml").arg(channelname).arg(filedate);

	QFile gzipFile(zipfile);
	gzipFile.open(QIODevice::WriteOnly);

	gzipFile.write(source->readAll());
	gzipFile.close();

	QProcess gzip;
	QString cmd("gzip -f -d " + gzipFile.fileName());

	gzip.start(cmd);
	gzip.waitForFinished();

	return filename;
}

DataDownloader::~DataDownloader()
{
}

void DataDownloader::checkDataStatus()
{
	// IF (
	// 		( per day channel file && if file exists for the request date ) 
	//		OR
	//		( single channel file && if request date <= max date of datafile)
	// THEN dont do anything and use local file
	if ( (isSingleXMLTVFile() && 
		_channelDateMap.find(_requestChannel) != _channelDateMap.end() && _channelDateMap[_requestChannel].toList().first() >= _requestDate.toString("yyyyMMdd") ) 
			||
			(!isSingleXMLTVFile() && _channelDateMap[_requestChannel].contains(_requestDate.toString("yyyyMMdd")) )
		)
	{
		// file exists for today ( current date ) so no need to download again
		//qDebug() << "data for " << _requestChannel << " for date " << _requestDate << " exists";
		_useLocal = true;
	}
	else
	{
		// download the requested channel for the requested date
		_useLocal = false;

		// if its a single file per channel/region delete existing files, 
		// at anytime we need only data for the week(or future) starting today
		if ( isSingleXMLTVFile() )
		{
			QDir xmltvPath(QString(DATAPATH), QString("*.xml"));
			foreach(QString filename, xmltvPath.entryList())
				QFile::remove(filename);
		}

		// FOR SINGLE XMLTV file channels
		// init the settings map with todays date for each channel
		// this will later get updated as its read in updateSettings.
		// we do this so that in case we download a channel with no data or no file then updateSettings wont be called
		// and we will keep re-downloading all the files till we get a channel with data
		// int his way atleast for the day we dont re-download on the same day if a channel doesnt have data so the user can keep trying
		// to see other channels with the same data
		if ( isSingleXMLTVFile() )
		{
			foreach(QString eachChannel, _channelCodeMap.keys())
			{
				_channelDateMap[eachChannel].clear();
		   		_channelDateMap[eachChannel].insert(QDate::currentDate().toString("yyyyMMdd"));
			}
			updateSettingsFile(QDate::currentDate().toString("yyyyMMdd"));
		}
	}
}

void DataDownloader::updateSettingsFile(QString maxDataDate, bool updateInd)
{
	// if indicator is false just return as it means nothing to update
	if ( !updateInd )
		return;

	QFile::remove(_settingsFile->fileName());

	if(isSingleXMLTVFile())
	{
		foreach(QString eachChannel, _channelCodeMap.keys())
		{
			_channelDateMap[eachChannel].clear();
		   _channelDateMap[eachChannel].insert(maxDataDate);
		}
	}
	else
	{
		_channelDateMap[_requestChannel].insert(_requestDate.toString("yyyyMMdd"));
	}

	_settingsFile->open(QIODevice::WriteOnly | QIODevice::ReadOnly);

	QTextStream s(_settingsFile);

	foreach(QString key, _channelDateMap.keys())	
	{
		s << "#" << key;
		foreach(QString date, _channelDateMap[key])
			s << '#' << date; 

		s << endl;
	}

	_settingsFile->close();
}

QStringList DataDownloader::getChannelList()
{
	QFile file(_channelFile);
	QStringList channellist;

	if (file.open(QIODevice::ReadOnly | QIODevice::Text))
	{
		QTextStream textStream(&file);
		
		while ( !textStream.atEnd() )
		{
			QStringList line = textStream.readLine().split('#');
			
			QString channelname = line[0];
			QString channelcode = line[1];

			_channelCodeMap[channelname] = channelcode;
		}

		return _channelCodeMap.keys();
	}
	
	return channellist;
}

QString DataDownloader::getChannelDisplayName(QString channelId)
{
	return _channelCodeMap.key(channelId);
}

QStringList DataDownloader::getFilelist()
{
	QString channelId = _channelCodeMap[_requestChannel];
	QDir xmltvPath(QString(DATAPATH), QString(channelId +  "*.xml") );
	
	QStringList filelist(xmltvPath.entryList());

	QStringList fullPathFileList;
	foreach(QString filename , filelist)
	{
		QString filepath = QString(DATAPATH)+filename;

		//////////////////////////////////////////////////////////
		// DELETE ALL OLD FILES, the ones that are 2 days or older
		// only per day channel files need to be deleted this way
		// single XLTV files always there is only 1 at any time for
		// a channel
		//////////////////////////////////////////////////////////
		QString filedatestr = filepath.section("--",1,1);
		QDate filedate = QDate::fromString(filedatestr, "yyyyMMdd");
		if ( QDate::currentDate() > filedate.addDays(2) )
		{
			//qDebug() << "Deleting old file " << filename;
			if ( !isSingleXMLTVFile()) 
			{
				QFile::remove(filepath);
				_channelDateMap[_requestChannel].remove(filedatestr);
				//qDebug() << "removing " << filepath;
				continue;
			}
		}

		fullPathFileList << filepath;
	}

	//qDebug() << fullPathFileList;

	return fullPathFileList;
}

void DataDownloader::setInput(DownloadInput d)
{
	_downloadIndicator = d.downloadIndicator;
	_requestChannel = d.requestChannel;
	_requestDate = d.requestDate;
}

void DataDownloader::splitXMLTVFile(QString filename)
{
	QFile file(filename);
	QFile out;
    file.open(QIODevice::ReadOnly);

	QString currChannel("");
	QString channel;
	while( !file.atEnd() )
    {
		QByteArray ba = file.readLine();
		QString line(ba);
		if ( line.contains("<programme") )
		{
			QXmlStreamReader reader( ba );
			QXmlStreamReader::TokenType tokenType = reader.readNext();
			tokenType = reader.readNext();
			QXmlStreamAttributes attributes = reader.attributes();
			QString channel = (attributes.value("channel")).toString();
			//qDebug() << line << "channel = " << channel;

			if ( channel != currChannel )
			{
				if ( currChannel != "" )
				{
					out.write("</tv>");
					out.close();
				}
				QString outfilename = QString(QString(DATAPATH) + "%1--%2--.xml").arg(channel).arg(QDate::currentDate().toString("yyyyMMdd"));
				out.setFileName(outfilename);
    			out.open(QIODevice::WriteOnly);
				currChannel = channel;
				out.write("<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>");  // to understand the western european latin characters
				out.write("<tv>\n");
			}

			out.write(ba);

			ba = file.readLine();
			line = ba;
			//qDebug() << line ;
			while ( !file.atEnd() && !line.contains("programme") )
			{
				out.write(ba);
				ba = file.readLine();
				line = ba;
				//qDebug() << line;
			}
			out.write(ba);
		}
	}

	// remove the huge xml file
	QFile::remove(filename);
}
