#include <QLabel>
#include <QPushButton>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QTextBrowser>
#include <QMenu>
#include <QAction>
#include <QtGui>
#include <QMaemo5ValueButton>
#include <QMaemo5ListPickSelector>
#include <QMaemo5InformationBox>
#include <QMessageBox>
#include <QSizePolicy>
#include <QtMaemo5>
#include <QListWidget>
#include <QListWidgetItem>
#include <QScrollArea>
#include <QPixmap>
#include <QDBusConnection>
#include <QDBusMessage>

#include <mce/mode-names.h>
#include <mce/dbus-names.h>

#include "mainwindow.h"
#include "misc.h"
#include "region.h"
#include "favourite.h"
#include "reminder.h"

//          	LAYOUT DIAGRAM
// --------------------------------------
// |           h1layout         |       |
// | 	     <--------->	    |action |
// | 	 Selection Panel	    |panel  |
// |----------------------------|       |
// | 							|       |
// | 	^						|       |
// | 	|             			|  v2   |
// | v1 | prog detail panel     |       | 
// | 	v						|       |
// | 							|       |
// | 							|       |
// |___________________________ |_______|
//   <---------- layout--------------->
//
//
const int NAVIGATION_BUTTON_SIZE = 70;
const int ACTION_BUTTON_SIZE = 70;

MainWindow::MainWindow()
    : QMainWindow(), _channelIndex(0), _landscapeOrientation(true)
{

	//////////////////////////////////////////////////////////////////	
	// this is the class that inits and updates the settings file
	// and also provides getters for the settings
	//////////////////////////////////////////////////////////////////	
	_settingsWindow = new SettingsWindow(this);

	//////////////////////////////////////////////////////////////////	
	//#### inits the APP settings
	//////////////////////////////////////////////////////////////////	
	initAppSettings();
	//////////////////////////////////////////////////////////////////	

	// create different widget for different orientations
	_portraitMainWidget = new QWidget(this);
	_landscapeMainWidget = new QWidget(this);

	// this is the class that provides the channel information from XMLTV downloads
	_infoapi = new InfoAccess(this,_settingsWindow->getRegion());

	/////////////////////////////////////////////
	/////     CALENDER & ALARM OBJECT      //////
	/////////////////////////////////////////////
	initCalendar();
	//////////////// END OF CALENDER INIT ////////////////////////////////////
			
	/////////////////////////////////////////////
	// the list of widgets for the selection panel ( see layout diagram above )
	/////////////////////////////////////////////

	// date selector
	_dateListModel = new QStandardItemModel(this);
	_dateSelectorList = new QMaemo5ListPickSelector(this);
	_dateSelectorList->setModel(_dateListModel);
	_dateSelector = new QMaemo5ValueButton("PICK DATE", this);
	_dateSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
	_dateSelector->setPickSelector(_dateSelectorList);
	_dateSelector->setValueText(QDate::currentDate().toString());

	// channel selector
	_channelListModel = new QStandardItemModel(this);
	_channelSelectorList = new QMaemo5ListPickSelector(this);
	_channelSelectorList->setModel(_channelListModel);
	_channelSelector = new QMaemo5ValueButton("PICK CHANNELS", this);
	_channelSelector->setPickSelector(_channelSelectorList);
	_channelSelector->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);

	// record and remind action buttons
	_recordButton = new QPushButton(this);
	_remindButton = new QPushButton(this);
	_recordButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_remindButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_recordButton->setIcon(QIcon(_settingsWindow->getIconPath() + "record.png"));
	_remindButton->setIcon(QIcon(_settingsWindow->getIconPath() + "remind.png"));
	
	/////////////////////////////////////////////
	// the list of widgets for the prog detail panel ( see diagram above for layout )
	/////////////////////////////////////////////

	// to list the programmes and their time
	_programmeList = new QListWidget(this);
	//_programmeList->setSelectionMode(QAbstractItemView::MultiSelection);
	_infoLabel = new QLabel(QString("<b>HIT GO !<\b>") + QString("\n\n*note data may be unavailable for some channels for some/all dates"), this);
	_infoLabel->setStyleSheet("QLabel { color : red }");
	_infoLabel->setWordWrap(true);
	_infoLabel->hide();

	// to show details of selected programmes
	_programmeDetail = new QTextBrowser(this);
	_programmeDetailLabel = new QLabel("YOUR CHOSEN PROGRAMMES", this);
	_programmeDetailLabel->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	_programmeDetailLabel->setStyleSheet("QLabel { background-color : lightgreen;  border-width: 5px; border-style: solid; border-color: green; color : black }");
	_programmeDetail->hide();
	_programmeDetailLabel->hide();

	// to get back to programme list
	_backToProgrammeListButton = new QPushButton(this);
	_backToProgrammeListButton->hide();
	_backToProgrammeListButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_backToProgrammeListButton->setIcon(QIcon(_settingsWindow->getIconPath() + "back.png"));

	// to clear selected programmes
	_clearProgButton = new QPushButton(this);
	_clearProgButton->setMaximumSize(ACTION_BUTTON_SIZE, ACTION_BUTTON_SIZE);
	_clearProgButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));
	_clearProgButton->hide();



	/////////////////////////////////////////////
	// the list of widgets for the action panel ( see above diagram for layout detail )
	/////////////////////////////////////////////

	// button to fetch data if not available in local memory
	_goButton = new QPushButton(this);
	_goButton->setMinimumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_goButton->setIcon(QIcon(_settingsWindow->getIconPath() + "go.png"));

	// button to clear the selected channels and all local data
	_clearButton = new QPushButton(this);
	_clearButton->setMaximumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_clearButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));

	// button to display selected programme details window
	_programmeDetailButton = new QPushButton(this);
	_programmeDetailButton->setMaximumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_programmeDetailButton->setIcon(QIcon(_settingsWindow->getIconPath() + "detail.png"));

	// buttons to navigate between selected channels for the chosen date
	_nextChannelButton = new QPushButton(this);
	_prevChannelButton = new QPushButton(this);
	_nextChannelButton->setMinimumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_prevChannelButton->setMinimumSize(NAVIGATION_BUTTON_SIZE, NAVIGATION_BUTTON_SIZE);
	_nextChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "next.png"));
	_prevChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "prev.png"));
	// only display when multi channel selection
	_nextChannelButton->setDisabled(true);
	_prevChannelButton->setDisabled(true);

	/////////////////////////////////////////////////////////////////
	// init the layouts for both orientation modes
	initLandscape();
	initPortrait();
	/////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////	
	//#### inits the models used by the channel and date selector ####
	//////////////////////////////////////////////////////////////////	
	initListModels();
	//////////////////////////////////////////////////////////////////	
   
	//////////////////////////////////////////////////////////////////	
	//#### create the menus
	//////////////////////////////////////////////////////////////////	
	createMenus();
	//////////////////////////////////////////////////////////////////	

	//////////////////////////////////////////////////////////////////	
	//#### init(listen to) and set the default orientation 
	//////////////////////////////////////////////////////////////////	
	initOrientation();
   //////////////////////////////////////////////////////////////////	

    // CONNECT ALL BUTTONS AND WIDGET TO THEIR SLOTS
	connect(_infoapi, SIGNAL(finished(bool)), this, SLOT(finished(bool)));
	connect(_dateSelectorList, SIGNAL(selected(QString)), this, SLOT(selectDate(QString)));
	connect(_channelSelectorList, SIGNAL(selected(QString)), this, SLOT(selectChannel(QString)));
	connect(_goButton, SIGNAL(clicked()), this, SLOT(go()));
	connect(_backToProgrammeListButton, SIGNAL(clicked()), this, SLOT(showProgrammeListWidget()));
	connect(_clearProgButton, SIGNAL(clicked()), this, SLOT(clearProg()));
	connect(_clearButton, SIGNAL(clicked()), this, SLOT(clearAll()));
	connect(_nextChannelButton, SIGNAL(clicked()), this, SLOT(showNextChannel()));
	connect(_prevChannelButton, SIGNAL(clicked()), this, SLOT(showPrevChannel()));
	connect(_programmeList, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(addProgrammeActionList(QListWidgetItem*)));
	connect(_programmeDetailButton, SIGNAL(clicked()), this, SLOT(showProgrammeDetailWidget()));
	connect(_recordButton, SIGNAL(clicked()), this, SLOT(showRecordMessage()));
	connect(_remindButton, SIGNAL(clicked()), this, SLOT(showRemindMessage()));
}

// LOADS THE APP SETTINGS TO MEMORY
void MainWindow::initAppSettings()
{
	_settingsWindow->loadSettings();
}

// INITS THE SHOWTIME CALENDAR
void MainWindow::initCalendar()
{
	_reminderManager = new Reminder;
	bool calexists = _reminderManager->checkCalendarNameExists("SHOWTIME");

	if ( !calexists )
	{
		//qDebug() << "SHOWTIME calendar does not exist, creating one";
		_reminderManager->addCalendar(QString("SHOWTIME"));
	}
	else
	{
		//qDebug() << "SHOWTIME calendar exists";
	}
}

// INITS THE DATE AND CHANNEL LISTS
void MainWindow::initListModels()
{
	//////////////////////////////////////////////////////
	// CHANNEL SELECTOR LIST 
	//////////////////////////////////////////////////////

	_dateListModel->clear();
	_channelListModel->clear();

	QStringList channelList = _infoapi->getChannelList();

	QStandardItem *item = new QStandardItem("YOUR FAVOURITES");
	item->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
	item->setEditable(false); // prevent editing of the item
	_channelListModel->appendRow(item);

	foreach(QString channel, channelList)
	{
		QStandardItem *item = new QStandardItem(channel);
		item->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
		item->setEditable(false); // prevent editing of the item
		_channelListModel->appendRow(item);
	}

	//////////////////////////////////////////////////////
	// DATE SELECTOR LIST 
	//////////////////////////////////////////////////////
	for(int i = -1; i < 7; ++i )
	{
		QStandardItem *dateitem = new QStandardItem(QDate::currentDate().addDays(i).toString());
		dateitem->setTextAlignment(Qt::AlignCenter); // the Maemo 5 design spec recommends this.
		dateitem->setEditable(false); // prevent editing of the item
		_dateListModel->appendRow(dateitem);
		
	}
}

// CREATES THE MENU
void MainWindow::createMenus()
{
	regionAct = new QAction(tr("&REGION"), this);
	favAct = new QAction(tr("&FAVOURITES"), this);
	setAct = new QAction(tr("&SETTINGS"), this);
	//openxmlAct = new QAction(tr("&LOAD XMLTV"), this);
	helpAct = new QAction(tr("&HELP"), this);
	donateAct = new QAction(tr("&DONATE"), this);

	connect(regionAct, SIGNAL(triggered()), this , SLOT(showRegion()));
	connect(favAct, SIGNAL(triggered()), this , SLOT(showFavourite()));
	connect(setAct, SIGNAL(triggered()), this , SLOT(showSettings()));
	//connect(openxmlAct, SIGNAL(triggered()), this , SLOT(loadXMLTV()));
	connect(helpAct, SIGNAL(triggered()), this , SLOT(showHelp()));
	connect(donateAct, SIGNAL(triggered()), this , SLOT(showDonate()));

	menu = menuBar()->addMenu(tr(""));
	menu->addAction(regionAct);
	menu->addAction(favAct);
	menu->addAction(setAct);
	//menu->addAction(openxmlAct);
	menu->addAction(helpAct);
	menu->addAction(donateAct);
}

// CONNECTS THE ACCELEROMETER USING DBUS to listen to orientation changes in AUTO mode only
void MainWindow::initOrientation()
{
	if ( _settingsWindow->getOrientation() == "AUTO" )
	{
		QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
							MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ));

		QDBusConnection::systemBus().connect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
			                MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString)));
		setLandscape(); // default
	}
    else if (_settingsWindow->getOrientation() == "LANDSCAPE") 
	{
		QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
							MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ));
		setLandscape();
	}
    else if (_settingsWindow->getOrientation() == "PORTRAIT")
	{
		QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
							MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ));
		setPortrait();
	}
}

// SLOT when orientation change is detected
void MainWindow::orientationChanged(const QString &newOrientation)
{
	if (newOrientation == QLatin1String(MCE_ORIENTATION_PORTRAIT))
		setPortrait();
	else
		setLandscape();
}

// The widgets that need to be hidden/show when the programme list is displayed
void MainWindow::showProgrammeListWidgetButtons()
{
	_goButton->show();  
	_programmeDetail->hide();
	_programmeDetailLabel->hide();
	_clearProgButton->hide();
	_programmeDetailButton->show();
	_backToProgrammeListButton->hide();
	_programmeList->show();
	_channelSelector->show();
	_dateSelector->show();
	_clearButton->show();
	_nextChannelButton->show();
	_prevChannelButton->show();

	showNextPrevChannelButtons();
}
		
// The method that calls the infoaccess class to download/read xmltv info into local memory
void MainWindow::go()
{
	if (_channelSelector->valueText() == "" || _dateSelector->valueText() == "")
		return;

	//qDebug() << "CHANNEL SELECTED : " << _channelSelector->valueText();

	_currentDate = QDate::fromString(_dateSelector->valueText(), Qt::TextDate);

	showProgrammeListWidgetButtons();

	DownloadInput d;
	d.requestDate = QDate::fromString(_dateSelector->valueText(), Qt::TextDate);
	d.requestChannel = _currentChannel;

	bool dataDownloaded = _infoapi->getChannelInfo(d);

	toggleBusy(true);

	// Call the callback slot finished explicitly here as its not called if data isnot downloaded
	if (!dataDownloaded)
		finished(false);
}

// simple message to guide the user to hit the go button when data doesnt exist in local memory
// for selected channel and date
void MainWindow::showInfoLabel(QString info)
{
	_programmeList->clear();
	_infoLabel->setText(info);
	if(_channelSelector->valueText() != "")
		_infoLabel->show();
}

// SLOT when date is selected
void MainWindow::selectDate(QString date)
{
	//qDebug() << "Current Date " << _dateSelector->valueText();
	if ( _dateSelector->valueText() != "" )
	{
		_currentDate = QDate::fromString(_dateSelector->valueText(), Qt::TextDate);
	}
    
    // SHOW the prog list if in memory already
	showProgrammeListWidget();
}

// SLOT when channel is selected
void MainWindow::selectChannel(QString channel)
{
	// IF YOU HAVE SELECTED FAVOURITES SIMPLY LOAD THE FAVOURITES AND RETURN
	if ( channel == "YOUR FAVOURITES" )
	{
		loadFavourites();
		return;
	}

	//qDebug() << "Current channel " << _channelSelector->valueText();
	if ( _channelSelector->valueText() != "" )
		_currentChannel = _channelSelector->valueText();

	// if the channel does not exist in the selected list then add it 
	// and set the index to current channel
	if ( ! _channelsSelected.contains(_currentChannel) )
	{
		_channelsSelected << _currentChannel;
		_channelIndex = _channelsSelected.count() - 1;
	}
	// set the index to the selected channel that exists
	else
	{
		_channelIndex = _channelsSelected.indexOf(_currentChannel);
	}

    // SHOW the prog list if in memory already
	showProgrammeListWidget();
}


// CLEAR all info and start afresh
void MainWindow::clearAll()
{
	_channelsSelected.clear();
	_channelIndex = 0;
	_channelDataMap.clear();
	_currentChannel.clear();
	_programmeList->clear();
	_programmeDetail->clear();
	_programmeSetForAction.clear();
	_nextChannelButton->setDisabled(true);
	_prevChannelButton->setDisabled(true);
	_infoapi->clear();

	_channelSelector->setValueText("");
	_dateSelector->setValueText("");
	_channelSelector->setText("PICK CHANNELS");
	_dateSelector->setValueText(QDate::currentDate().toString());
}

// CLEAR selected programmes list
void MainWindow::clearProg()
{
	_programmeDetail->clear();
	_programmeSetForAction.clear();
}

// Callback SLOT when the XMLTV info required is downloaded/read
void MainWindow::finished(bool dataRequested)
{
	//qDebug() << "In finished";

	toggleBusy(false);
	_infoLabel->hide();

	// we pass the dataRequested here to indicate that we are calling the below method
	// after requesting data since it wasnt available in the local channelDataMap
	return showProgrammeListWidget(true);
}


// dataRequested indicates that this is being called by finished via a callback after requesting data
// if this is false it means that its being called by usual navigation buttons 
// and in that case if there is no data we display HitGo
void MainWindow::showProgrammeListWidget(bool dataRequested)
{
	_programmeList->clear();
	_infoLabel->hide();
	showProgrammeListWidgetButtons();
	
    QList<ChannelData> completeData = (_infoapi->results())[_currentChannel];

	//qDebug() << _infoapi->results()[_currentChannel];

	bool dataAvailable = false;

	foreach(ChannelData data, completeData)
	{
		QString title = data.title;
		QString stime = data.stime;
		stime.truncate(5);
		QString etime = data.etime;
		etime.truncate(5);
		QString time = stime + " - " + etime;
		// have 2 keys here as in the display we dont want to display date
		// but on clicking the item in the list we want the key to have the date also just in case the user selects the same program 
		// for multiple days
		QString keyForDisplay;
		QString keyForInsert;
		if ( _landscapeOrientation )
		{
			keyForDisplay = title + "\n\t\t\t\t\t\t" + time;
			//keyForInsert = title + "\n\t\t\t\t\t\t" + time + "\t" + data.sdate;
			keyForInsert = title + " " + time + " " + data.sdate;
		}
		else // in portrait mode we need less tabs to prevent horizontal scrolling
		{
			keyForDisplay = title + "\n\t\t\t" + time;
			keyForInsert = title + " " + time + " " + data.sdate;
		}

		if ( data.sdate == _currentDate.toString(Qt::TextDate) )
		{
			_programmeList->addItem(keyForDisplay);
			_channelDataMap[keyForInsert] = data;
			dataAvailable = true;
		}
	}
	// this means that data is not available locally in channelDataMap and also it hasnt been
	// requested so it means you have to hit GO :-) !
	if ( !dataRequested && !dataAvailable )
		showInfoLabel( QString("<b>HIT GO !<\b>\n") + QString("\n\n*Note data may be unavailable for some channels for some/all dates") );
	else if ( dataRequested && !dataAvailable )
		showInfoLabel( QString("<b>NO DATA AVAILABLE FOR THIS CHANNEL !<\b>") );

}

// SLOT to show the selected programme details like description etc
void MainWindow::showProgrammeDetailWidget()
{
	_clearButton->hide();
	_programmeDetailButton->hide();
	_goButton->hide();
	_channelSelector->hide();
	_dateSelector->hide();
	// clear previous programme selection data
	_programmeDetail->clear();
	_prevChannelButton->hide();
	_nextChannelButton->hide();

	_programmeList->hide();  // hide list

	_programmeDetailLabel->show();
	_clearProgButton->show();
	_programmeDetail->show();  // show detail
	_backToProgrammeListButton->show();

	foreach(QString key, _programmeSetForAction)
	{
		_programmeDetail->setTextColor(QColor(Qt::darkGreen));
		_programmeDetail->append(_channelDataMap[key].channelId.toUpper());
		_programmeDetail->setTextColor(QColor(Qt::blue));
		_programmeDetail->append(_channelDataMap[key].title);
		_programmeDetail->setTextColor(QColor(Qt::red));
		_programmeDetail->append(_channelDataMap[key].sdate + "  ::  " + _channelDataMap[key].stime.left(5) +  "-" + _channelDataMap[key].etime.left(5));
		_programmeDetail->setTextColor(QColor(Qt::black));
		_programmeDetail->append(_channelDataMap[key].desc);
		_programmeDetail->append("");
		_programmeDetail->moveCursor(QTextCursor::Start);
	}
}

// show next selected channel
void MainWindow::showNextChannel()
{
	_prevChannelButton->show();

	++_channelIndex;
	showNextPrevChannelButtons();

	// shows the data for the channel if available for selected date
	showProgrammeListWidget();
}

// show previous selected channel
void MainWindow::showPrevChannel()
{

	_nextChannelButton->show();

	--_channelIndex;
	showNextPrevChannelButtons();

	// shows the data for the channel if available for selected date
	showProgrammeListWidget();
}

// make sure we are showing the next/prev arrows correctly
// also set the channel drop down button text to current selected channel
void MainWindow::showNextPrevChannelButtons()
{
	// change the text on the selector depending on the programme list you are viewing
	if ( _channelsSelected.count() != 0 )
	{
		_channelSelector->setValueText(_channelsSelected[_channelIndex]);
		_currentChannel = _channelSelector->valueText();
	}

    // show the arrows only when we have more than 1 channel
	if ( _channelsSelected.count() > 1 )
	{
		_prevChannelButton->setDisabled(false);
		_nextChannelButton->setDisabled(false);
	}

    // disable next when we have reached the last channel
	if( _channelIndex == _channelsSelected.count()-1 )
	{
		_nextChannelButton->setDisabled(true);
	}

    // disable prev when we have reached the first channel
	if( _channelIndex == 0 )
	{
		_prevChannelButton->setDisabled(true);
	}
}

// this adds the selected programs to the action list
// action list is used when applying action ( remind/record )
void MainWindow::addProgrammeActionList(QListWidgetItem* programme)
{
	QString key;
	// add the date to the listwidgetitem as in the list we dont display the date
	if ( _dateSelector->valueText() != "" )
		key = (programme->text() + "\t" + _dateSelector->valueText()).simplified();

	// on clicking this slot is called so
	// if the program exists then remove it ( as user deselected) else add it(user selected)
	if(_programmeSetForAction.find(key) == _programmeSetForAction.end()) 
	{
		//qDebug() << "Selecting " << key;
		_programmeSetForAction.insert(key);
	}
	else
	{
		//qDebug() << "De Selecting " << key;
		_programmeSetForAction.remove(key);
	}
}

// show the record action we are doing to the user
// currently only available to UK
void MainWindow::showRecordMessage()
{
    if ( _settingsWindow->getRegion() != "UK" )
    {
		QMaemo5InformationBox::information(this, "SORRY THIS FUNCTIONALITY IS ONLY SUPPORTED BY UK AT THE MOMENT, if you are aware of any support for your region, please let me know either in my thread talk.maemo or send me an email" , QMaemo5InformationBox::NoTimeout);

        return;
    }

	QString info1 = "Sent below message to the SMS app for recipient 61759 to remote record\n";
	QString info2 = "You will be charged as per SKY UK rates, Please check their website for details\n";
	QString info3 = "\nEnsure you send that SMS and then click DONE here, else the next message(if any) will overwrite it in the SMS app\n\n";

	QMessageBox msgBox;
	foreach(QString progKey, _programmeSetForAction)
	{
		QString sdate = QDate::fromString(_channelDataMap[progKey].sdate, Qt::TextDate).toString("dd/MM");
		//qDebug() << _channelDataMap[progKey].stime;
		QString stime = QTime::fromString(_channelDataMap[progKey].stime, "HH:mm:ss").toString("HH:mm");
		QString recordMsg = _channelDataMap[progKey].title + "." +_channelDataMap[progKey].channelId + "." + sdate + "." + stime;

		QStringList dataToSender;
		dataToSender << recordMsg << "61759";

		//qDebug() << "no. of programmes = " << _programmeSetForAction.count();
		
		_infoapi->sendRecordMsg(dataToSender);

		QMessageBox::StandardButton reply;
		reply = QMessageBox::information(this, "PLEASE READ THIS WARNING", info1+info2+info3+recordMsg);
	}

	clearProg();  // clear selected programmes
}

// SLOT to update the calendar DB and set the reminder
void MainWindow::showRemindMessage()
{

	foreach(QString progKey, _programmeSetForAction)
	{
		QDateTime progSTime = QDateTime::fromString(_channelDataMap[progKey].sdate + _channelDataMap[progKey].stime, "ddd MMM d yyyyHH:mm:ss");
		QDateTime progETime = QDateTime::fromString(_channelDataMap[progKey].edate + _channelDataMap[progKey].etime, "ddd MMM d yyyyHH:mm:ss");

		//qDebug() << "Alarm Start Time : " << progSTime << " Alarm End time : " << progETime;

		QString remindTitle = _channelDataMap[progKey].title + " on channel " + _channelDataMap[progKey].channelId;
		QString remindMessage = "This is a reminder of your favourite show from the SHOWTIME APP on the Nokia N900";

		if ( _settingsWindow->getRemindType() == "EVENT" )	
		{
			_reminderManager->addEvent("SHOWTIME", progSTime, progETime, remindTitle, remindMessage, 1);
		}
        else if ( _settingsWindow->getRemindType() == "ALARM" )	
		{
			_reminderManager->addAlarmEvent("SHOWTIME", progSTime, remindTitle, progSTime, progETime, remindMessage);
		}

	}

	QMaemo5InformationBox::information(this, "<b>YOUR SHOWTIME CALENDAR HAS BEEN UPDATED WITH YOUR SELECTED PROGRAMMES<\b>" , QMaemo5InformationBox::NoTimeout);

	clearProg();  // clear selected programmes
}

// set the app widgets to landscape mode
void MainWindow::setLandscape()
{
	initLandscape();
	setAttribute(Qt::WA_Maemo5ForceLandscapeOrientation, true);
	setCentralWidget(_landscapeMainWidget);

	_landscapeOrientation = true;
	_programmeList->clear(); // have to do this as we are doing some manual formatting to display the programmes and times
}

// set the app widgets to portrait mode
void MainWindow::setPortrait()
{
	initPortrait();
	setAttribute(Qt::WA_Maemo5ForcePortraitOrientation, true);
	setCentralWidget(_portraitMainWidget);

	_landscapeOrientation = false;
	_programmeList->clear(); // have to do this as we are doing some manual formatting to display the programmes and times
}

// init the landscape layout
void MainWindow::initLandscape()
{
	/////////////////////////////////////////////
	// the layout definitions
	/////////////////////////////////////////////
	_lh1layout = new QHBoxLayout;
	_lv1layout = new QVBoxLayout;
	_lv2layout = new QVBoxLayout;
	_landscapeLayout = new QHBoxLayout;
	_landscapeMainWidget = new QWidget(this);
	
	// define the selection panel layout	
	_lh1layout->addWidget(_programmeDetailLabel);
	_lh1layout->addWidget(_dateSelector);
	_lh1layout->addWidget(_recordButton);
	_lh1layout->addWidget(_remindButton);
	_lh1layout->addWidget(_channelSelector);
	_lh1layout->addWidget(_clearProgButton);
	_lh1layout->addWidget(_backToProgrammeListButton);

	// define the action panel layout	
	_lv2layout->addWidget(_goButton);
	_lv2layout->addWidget(_programmeDetailButton);
	_lv2layout->addWidget(_clearButton);
	_lv2layout->addWidget(_nextChannelButton);
	_lv2layout->addWidget(_prevChannelButton);

	// define the data panel layout	
	_lv1layout->addLayout(_lh1layout);
	_lv1layout->addWidget(_programmeDetail);
	_lv1layout->addWidget(_programmeList);
	_lv1layout->addWidget(_infoLabel);

	// define the main layout	
	_landscapeLayout->addLayout(_lv1layout);
	_landscapeLayout->addLayout(_lv2layout);

	_landscapeMainWidget->setLayout(_landscapeLayout);
}

// init the portrait layout
void MainWindow::initPortrait()
{
	/////////////////////////////////////////////
	// the layout definitions
	/////////////////////////////////////////////
	_ph1layout = new QHBoxLayout;
	_ph2layout = new QHBoxLayout;
	_ph3layout = new QHBoxLayout;
	_ph4layout = new QHBoxLayout;
	_pv1layout = new QVBoxLayout;
	_portraitLayout = new QHBoxLayout;
	_portraitMainWidget = new QWidget(this);
	
	// channel selector with record
	_ph1layout->addWidget(_programmeDetailLabel); // this is available in programme detail mode only
	_ph1layout->addWidget(_channelSelector);
	_ph1layout->addWidget(_recordButton);

	// date selector with remind
	_ph2layout->addWidget(_dateSelector);
	_ph2layout->addWidget(_remindButton);
	_ph2layout->addWidget(_clearProgButton); // this is available in programme detail mode only
	_ph2layout->addWidget(_backToProgrammeListButton);// this is available in programme detail mode only

	// define the data panel layout	
	_ph3layout->addWidget(_infoLabel);
	_ph3layout->addWidget(_programmeDetail);// this is available in programme detail mode only
	_ph3layout->addWidget(_programmeList);

	// define the action panel layout	
	_ph4layout->addWidget(_clearButton);
	_ph4layout->addWidget(_programmeDetailButton);
	_ph4layout->addWidget(_goButton);
	_ph4layout->addWidget(_prevChannelButton);
	_ph4layout->addWidget(_nextChannelButton);

	_pv1layout->addLayout(_ph1layout);
	_pv1layout->addLayout(_ph2layout);
	_pv1layout->addLayout(_ph3layout);
	_pv1layout->addLayout(_ph4layout);

	// define the main layout	
	_portraitLayout->addLayout(_pv1layout);

	_portraitMainWidget->setLayout(_portraitLayout);
}


MainWindow::~MainWindow()
{
}

// slot to load your own xmltv file
void MainWindow::loadXMLTV()
{
	QString filename(_settingsWindow->getBinPath() + "xmltvfile.xml");
	_infoapi->processXMLTV(filename);
}

// slot to encourage me more ;-)
void MainWindow::showDonate()
{
	 QDesktopServices::openUrl(QUrl("https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=rkmaemo@gmx.com&lc=GB&item_name=SHOWTIME Maemo&currency_code=GBP&bn=PP%2dDonationsBF%3abtn_donateCC_LG%2egif%3aNonHosted"));
}

// slot to set/display your favourite channel selection
void MainWindow::showFavourite()
{
	_favouriteWindow = new FavouriteWindow(this);
	_favouriteWindow->setAttribute(Qt::WA_Maemo5StackedWindow);

	connect(_favouriteWindow, SIGNAL(clicked()), this, SLOT(reInitFavouriteSettings()));

	_favouriteWindow->show();
}

// slot to set/display your region
// resets all data to start
void MainWindow::showRegion()
{
	_regionWindow = new RegionWindow(this);
	_regionWindow->setAttribute(Qt::WA_Maemo5StackedWindow);

	connect(_regionWindow, SIGNAL(clicked()), this, SLOT(reInitInfoAccess()));

	_regionWindow->show();

}

// slot to set/display settings 
void MainWindow::showSettings()
{
	_miscWindow = new MiscSettingsWindow(this);
	_miscWindow->setAttribute(Qt::WA_Maemo5StackedWindow);

	connect(_miscWindow, SIGNAL(clicked()), this, SLOT(reInitMiscSettings()));

	_miscWindow->show();
}

// SLOT::reloads the favourite settings
void MainWindow::reInitFavouriteSettings()
{
	initAppSettings();
	_favouriteWindow->close();
}

// SLOT::reloads the settings , clears all data
// and starts afresh with newly selected region
void MainWindow::reInitInfoAccess()
{
	initAppSettings();

	//qDebug() << "Region has been set to " << _settingsWindow->getRegion();
	clearAll();
	// disconnect the old slot
	disconnect(_infoapi, SIGNAL(finished(bool)), this, SLOT(finished(bool)));
	_infoapi = new InfoAccess(this, _settingsWindow->getRegion());
	// connect the new slot
	connect(_infoapi, SIGNAL(finished(bool)), this, SLOT(finished(bool)));
	initListModels();

	_regionWindow->close();
}

// SLOT::reloads the misc settings
void MainWindow::reInitMiscSettings()
{
	initAppSettings(); // in case ALARM type is changed
	initOrientation(); // in case Orientation is changed

	// IN CASE ICON SET IS UPDATED
	_recordButton->setIcon(QIcon(_settingsWindow->getIconPath() + "record.png"));
	_remindButton->setIcon(QIcon(_settingsWindow->getIconPath() + "remind.png"));
	_backToProgrammeListButton->setIcon(QIcon(_settingsWindow->getIconPath() + "back.png"));
	_clearProgButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));
	_goButton->setIcon(QIcon(_settingsWindow->getIconPath() + "go.png"));
	_clearButton->setIcon(QIcon(_settingsWindow->getIconPath() + "clear.png"));
	_programmeDetailButton->setIcon(QIcon(_settingsWindow->getIconPath() + "detail.png"));
	_nextChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "next.png"));
	_prevChannelButton->setIcon(QIcon(_settingsWindow->getIconPath() + "prev.png"));

	_miscWindow->close();
}

// SLOT::this function is called by the selectChannel slot when selected channel is FAVOURITES
void MainWindow::loadFavourites()
{
	QStringList favourites = _settingsWindow->getFavourites(_settingsWindow->getRegion());

	if (favourites.isEmpty())
	{
		//qDebug() << "No favourites present for the selected region";
		_channelSelector->setValueText(_currentChannel);
		return;
	}

	foreach(_currentChannel, favourites)
	{
		// if the channel does not exist in the selected list then add it 
		// and set the index to current channel
		if ( ! _channelsSelected.contains(_currentChannel) )
		{
			//qDebug() << "Adding Favourite Channel to list" << _currentChannel;
			_channelsSelected << _currentChannel;
			_channelSelector->setValueText(_currentChannel);
			_channelIndex = _channelsSelected.count() - 1;
		}
	}

    // SHOW the prog list if in memory already
	showProgrammeListWidget();
}

void MainWindow::toggleBusy(bool isBusy)
{
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, isBusy);
}

void MainWindow::showHelp()
{
	QLabel *imageLabel = new QLabel;
	QImage image("/opt/showtime/data/helpfiles/showtime_help.gif");
	imageLabel->setPixmap(QPixmap::fromImage(image));

	QMainWindow *_helpWindow = new QMainWindow ( this );
	_helpWindow->setAttribute(Qt::WA_Maemo5StackedWindow);
	_helpWindow->setAttribute(Qt::WA_DeleteOnClose);
	_helpWindow->setWindowTitle("HELP WINDOW (scroll to view)");

	QScrollArea *_scroll= new QScrollArea;
	_scroll->setWidget(imageLabel);

	_helpWindow->setCentralWidget(_scroll);
	_helpWindow->show();
}


//////////////////////////////////////////############## END OF SHOWTIM ###############////////////////////////////////////////////////
