#include <glib.h>
#include <sharing-plugin-interface.h>
#include <sharing-transfer.h>
#include <conicconnection.h>
#include <osso-log.h>
#include <hildon/hildon.h>
#include <string.h>

#include "mobilblogg.h"

guint sharing_plugin_interface_init (gboolean* dead_mans_switch);
guint sharing_plugin_interface_uninit (gboolean* dead_mans_switch);
SharingPluginInterfaceSendResult sharing_plugin_interface_send (
		SharingTransfer* transfer, ConIcConnection* con,
		gboolean* dead_mans_switch);
SharingPluginInterfaceAccountSetupResult
sharing_plugin_interface_account_setup (
		GtkWindow* parent,
		SharingService* service, SharingAccount** worked_on,
		osso_context_t* osso);
SharingPluginInterfaceEditAccountResult
sharing_plugin_interface_edit_account (
		GtkWindow* parent,
		SharingAccount* account,
		ConIcConnection* con,
		gboolean* dead_mans_switch);
SharingPluginInterfaceAccountValidateResult
sharing_plugin_interface_account_validate (
		SharingAccount* account, ConIcConnection* con,
		gboolean *cont, gboolean* dead_mans_switch);
		
SharingPluginInterfaceAccountSetupResult
create_ui (GtkWindow* parent,
           SharingAccount* worked_on,
           gboolean isEdit);

guint
sharing_plugin_interface_init (gboolean *dead_mans_switch)
{
    *dead_mans_switch = FALSE;
    return 0;
}

guint
sharing_plugin_interface_uninit (gboolean *dead_mans_switch)
{
    *dead_mans_switch = FALSE;
    ULOG_INFO_L("Cleaning up mobilblogg");
    return 0;
}


SharingPluginInterfaceAccountSetupResult
create_ui (GtkWindow* parent,
           SharingAccount* worked_on,
           gboolean isEdit)
{
    
    GtkWidget *dlg = 0;
    
    ULOG_INFO_L("Starting Mobilblogg UI");
    
    if (isEdit) {
        dlg = gtk_dialog_new_with_buttons ("Edit account - MobilBlogg", parent,
				                           GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				                           GTK_STOCK_SAVE, GTK_RESPONSE_YES,
                                           GTK_STOCK_DELETE, GTK_RESPONSE_NO,
                                           NULL);
    } else {
        dlg = gtk_dialog_new_with_buttons ("Account setup - MobilBlogg", parent,
				                           GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				                           GTK_STOCK_OK, GTK_RESPONSE_YES,
                                           NULL);
    }
    
    GtkWidget *dlg_content = gtk_dialog_get_content_area (GTK_DIALOG (dlg));
    GtkWidget *table = gtk_table_new (4, 2, FALSE);
    gtk_container_add (GTK_CONTAINER (dlg_content), table);
    
    GtkWidget *username, *password, *secretword;

    GtkWidget *label = gtk_label_new ("Username");
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL | GTK_EXPAND, HILDON_MARGIN_DOUBLE, 0);
    
    username = hildon_entry_new (HILDON_SIZE_AUTO);
    hildon_entry_set_placeholder (HILDON_ENTRY (username), "User name");
    gtk_table_attach_defaults (GTK_TABLE (table), username, 1, 2, 0, 1);
    if (isEdit) {
        gtk_entry_set_text (GTK_ENTRY (username), sharing_account_get_username (worked_on));
    }

    label = gtk_label_new ("Password");
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL | GTK_EXPAND, HILDON_MARGIN_DOUBLE, 0);

    password = hildon_entry_new (HILDON_SIZE_AUTO);
    hildon_entry_set_placeholder (HILDON_ENTRY (password), "Password");
    hildon_gtk_entry_set_input_mode (GTK_ENTRY (password), HILDON_GTK_INPUT_MODE_FULL | HILDON_GTK_INPUT_MODE_INVISIBLE);
    gtk_table_attach_defaults (GTK_TABLE (table), password, 1, 2, 1, 2);
    if (isEdit) {
        gtk_entry_set_text (GTK_ENTRY (password), sharing_account_get_password (worked_on));
    }

    label = gtk_label_new ("Secret word");
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL | GTK_EXPAND, HILDON_MARGIN_DOUBLE, 0);

    secretword = hildon_entry_new (HILDON_SIZE_AUTO);
    hildon_entry_set_placeholder (HILDON_ENTRY (secretword), "Secret word");
    hildon_gtk_entry_set_input_mode (GTK_ENTRY (secretword), HILDON_GTK_INPUT_MODE_FULL | HILDON_GTK_INPUT_MODE_INVISIBLE);
    gtk_table_attach_defaults (GTK_TABLE (table), secretword, 1, 2, 2, 3);
    if (isEdit) {
        gtk_entry_set_text (GTK_ENTRY (secretword), sharing_account_get_param (worked_on, "secretword"));
    }

    gtk_widget_show_all (GTK_WIDGET (dlg));
    gint result = gtk_dialog_run (GTK_DIALOG (dlg));
    
    if (result == GTK_RESPONSE_YES) {
        ULOG_INFO_F("Here we go: %s %s %s", gtk_entry_get_text (GTK_ENTRY (username)),gtk_entry_get_text (GTK_ENTRY (password)), gtk_entry_get_text (GTK_ENTRY (secretword)));
        sharing_account_set_param (worked_on, "username", gtk_entry_get_text (GTK_ENTRY (username)));
        sharing_account_set_param (worked_on, "password", gtk_entry_get_text (GTK_ENTRY (password)));
        sharing_account_set_param (worked_on, "secretword", gtk_entry_get_text (GTK_ENTRY (secretword)));
        
        gtk_widget_destroy (dlg);
        
        return SHARING_EDIT_ACCOUNT_SUCCESS;
    } else if (result == GTK_RESPONSE_NO){
        gtk_widget_destroy (dlg);
        return SHARING_EDIT_ACCOUNT_DELETE;
    } else {
        gtk_widget_destroy (dlg);
        return SHARING_EDIT_ACCOUNT_CANCELLED;
    }
}

SharingPluginInterfaceAccountValidateResult
sharing_plugin_interface_account_validate (SharingAccount* account,
                                           ConIcConnection* con,
                                           gboolean *cont,
                                           gboolean* dead_mans_switch)
{
    gchar *user = sharing_account_get_username (account);
	gchar *pass = sharing_account_get_password (account);
	gchar *secretword = sharing_account_get_param (account, "secretword");
	
    if (!user || !pass || !secretword)
        return SHARING_ACCOUNT_VALIDATE_NOT_STARTED;
	
    ULOG_INFO_L("At least we have everything we need, let's parse it");
	
	if (!strlen (user))
        return SHARING_ACCOUNT_VALIDATE_NOT_STARTED;
	if (!strlen (pass))
        return SHARING_ACCOUNT_VALIDATE_NOT_STARTED;
	if (!strlen (secretword))
        return SHARING_ACCOUNT_VALIDATE_NOT_STARTED;

    ULOG_INFO_L("Running mobilblogg_login");
    mobilblogg_t *mob = mobilblogg_init (con, user, pass);
    mobilblogg_error_code res = mobilblogg_login (mob);
    mobilblogg_destroy (mob);
    ULOG_INFO_F("mobilblogg_login -> %d", res);

    if (res == MobilBloggNoError) {
        return SHARING_ACCOUNT_VALIDATE_SUCCESS;
    } else if (res == MobilBloggConnectionError) {
        return SHARING_ACCOUNT_VALIDATE_ERROR_CONNECTION;
    } else if (res == MobilBloggCredentialsError) {
        return SHARING_ACCOUNT_VALIDATE_FAILED;
    } else {
        return SHARING_ACCOUNT_VALIDATE_ERROR_UNKNOWN;
    }
}

SharingPluginInterfaceAccountSetupResult
sharing_plugin_interface_account_setup (
		GtkWindow* parent,
		SharingService* service, SharingAccount** worked_on,
		osso_context_t* osso)
{
    return create_ui (parent, *worked_on, FALSE);
}

SharingPluginInterfaceEditAccountResult
sharing_plugin_interface_edit_account (GtkWindow* parent,
                                       SharingAccount* account,
                                       ConIcConnection* con,
                                       gboolean* dead_mans_switch)                                   
{
    return create_ui (parent, account, TRUE);
}

SharingPluginInterfaceSendResult
sharing_plugin_interface_send (SharingTransfer* transfer,
                               ConIcConnection* con,
                               gboolean* dead_mans_switch)
{
    SharingEntry *entry = sharing_transfer_get_entry (transfer);
    SharingAccount *account = sharing_entry_get_account (entry);
    SharingPluginInterfaceSendResult ret = SHARING_SEND_ERROR_CONNECTION;
    
    const gchar *right = sharing_entry_get_option (entry, "privacy");
    if (g_ascii_strcasecmp (right, "all") == 0) {
        right = "";
    }
    
    ULOG_INFO_F("privacy: %s", right);
    
    sharing_transfer_set_progress (transfer, 0.0);
    *dead_mans_switch = FALSE;
    
    mobilblogg_t *mob = mobilblogg_init (con, sharing_account_get_username (account), sharing_account_get_password (account));
    mobilblogg_set_secretword (mob, sharing_account_get_param (account, "secretword"));
    mobilblogg_error_code res = mobilblogg_login (mob);
    if (res != MobilBloggNoError || !sharing_transfer_continue (transfer)) {
        ULOG_INFO_L("Failed login");
        mobilblogg_destroy (mob);
        if (!sharing_transfer_continue (transfer)) {
            return SHARING_SEND_CANCELLED;
        } else if (res == MobilBloggCredentialsError) {
            return SHARING_SEND_ERROR_AUTH;
        } else {
            return SHARING_SEND_ERROR_CONNECTION;
        }
    }
    
    ULOG_INFO_L("Login done");
    
    sharing_transfer_set_progress (transfer, 0.1);
    *dead_mans_switch = FALSE;
    
    guint64 total_bytes = 0;
    for (GSList *p = sharing_entry_get_media (entry); p; p = g_slist_next (p)) {
        SharingEntryMedia *media = p->data;
        if (!sharing_entry_media_get_sent (media)) {
            total_bytes += sharing_entry_media_get_size (media);
            ULOG_INFO_F("total_bytes = %llu", total_bytes);
        }
    }
    
    if (!total_bytes) total_bytes = 1;
    guint64 sent_bytes = 0;
    
    for (GSList* p = sharing_entry_get_media (entry); p; p = g_slist_next (p)) {
            SharingEntryMedia* media = p->data;
            
            ULOG_INFO_L("sending %s", sharing_entry_media_get_title (media));
            
            if (sharing_entry_media_get_sent (media)) {
                ULOG_INFO_F("get_sent %d", sharing_entry_media_get_sent (media));
                continue;
            }

            if (!sharing_transfer_continue (transfer)) {
                ret = SHARING_SEND_CANCELLED;
                break;
            }

            guint64 media_size = sharing_entry_media_get_size (media);

            gdouble progress_start = 1.0 - 0.9*((total_bytes - sent_bytes) / (gdouble)total_bytes);
            gdouble progress_end = 1.0 - 0.9*((total_bytes - (sent_bytes + media_size)) / (gdouble)total_bytes);

            sharing_transfer_set_progress (transfer, progress_start);
            *dead_mans_switch = FALSE;
            
            mobilblogg_error_code res = mobilblogg_send (mob, transfer, 
                                                         progress_start, progress_end,
                                                         dead_mans_switch, right, media);

            if (res == MobilBloggNoError) {
                sharing_entry_media_set_sent (media, TRUE);
                ret = SHARING_SEND_SUCCESS;
            } else if (res == MobilBloggErrorCancelled) {
                ret = SHARING_SEND_CANCELLED;
            } else if (res == MobilBloggCredentialsError) {
                ret = SHARING_SEND_ERROR_AUTH;
            } else {
                ret = SHARING_SEND_ERROR_CONNECTION;
            }
    }
    
    sharing_transfer_set_progress (transfer, 1.0);
    *dead_mans_switch = FALSE;

    return ret;
    
}
