/*
 * This file is part of Command-Line Sharing Plugin
 *
 * Copyright (C) 2010 Tuomas Kulve.
 * Copyright (C) 2008-2009 Nokia Corporation. All rights reserved.
 *
 * This maemo code example is licensed under a MIT-style license,
 * that can be found in the file called "COPYING" in the root
 * directory.
 *
 */

#include <stdio.h>
#include <glib.h>
#include <osso-log.h>
#include <sharing-http.h>
#include "send.h"


#ifdef ULOG_DEBUG_L
#undef ULOG_DEBUG_L
#endif


#define ULOG_DEBUG_L(FMT, ARG...) {FILE *f = fopen("/tmp/cli.log", "a"); fprintf(f, "%s:%d: " FMT, __FILE__, __LINE__, ## ARG); fprintf(f, "\n"); fclose(f);}


/**
 * send:
 * @account: #SharingTransfer to be send
 * @con: Connection used
 * @dead_mans_switch: Turn to %FALSE at least every 30 seconds.
 *
 * Sends #SharingTransfer to service.
 *
 * Returns: #SharingPluginInterfaceSendResult
 */
SharingPluginInterfaceSendResult cli_send (SharingTransfer* transfer,
    ConIcConnection* con, gboolean* dead_mans_switch)
{
    SharingEntry *entry;
	SharingAccount *account;

	entry = sharing_transfer_get_entry(transfer);
	account = sharing_entry_get_account(entry);

	ULOG_DEBUG_L("Sending");

    for (GSList* p = sharing_entry_get_media (entry);
		 p != NULL;
		 p = g_slist_next(p)) {

      SharingEntryMedia* media = p->data;

      /* Process media */
      if (!sharing_entry_media_get_sent (media)) {
		gchar cmd[4096];
		gint exit_status;
		GError *error = NULL;
		gboolean retval;
		const gchar *local_file;
		gchar* command_line;

		local_file = sharing_entry_media_get_localpath(media);

		ULOG_DEBUG_L("Local file: %s", local_file);

		if (local_file == NULL) {
		  ULOG_DEBUG_L("No local file: %s", error->message);
		  return SHARING_SEND_ERROR_UNKNOWN;
		}


		sharing_transfer_set_progress(transfer, 0.3);

        command_line = sharing_account_get_param (account, "command_line");

		ULOG_DEBUG_L("service command line: %s", command_line);

		g_snprintf(cmd, 4096,
				   command_line,
				   local_file, 
				   sharing_entry_media_get_filename(media));

		ULOG_DEBUG_L("Executing: %s", cmd);

		retval = g_spawn_command_line_sync(cmd,
										NULL, NULL,
										&exit_status,
										&error);
		if (!retval) {
		  ULOG_DEBUG_L("Failed to scp: %s", error->message);
		  g_error_free(error);
		  return SHARING_SEND_ERROR_UNKNOWN;
		}

		sharing_transfer_set_progress(transfer, 1.0);
		sharing_entry_media_set_sent (media, TRUE);

      }
    }

    return SHARING_SEND_SUCCESS;
}

