/*
 *  scout Maemo 5 calendar, contact and conversations search tool
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "calendar-todo-editor.h"
#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <CMulticalendar.h>
#include <CCalendar.h>
#include <CTodo.h>
#include <CAlarm.h>
#include <clockd/libtime.h>
#include "calendar-event-preview.h"

typedef struct todo_ui_data_t
{
  GtkWidget* window;
  GtkWidget* todo_title_entry;
  GtkWidget* todo_done_check_button;
  GtkWidget* start_date_button;
  GtkWidget* alarm_check_button;
  GtkWidget* alarm_date_button;
  GtkWidget* alarm_time_button;
  CCalendar* calendar;
  CTodo* todo;
} todo_ui_data_t;

static void
_delete_todo_ui_data(GtkWidget* widget,
		      gpointer user_data)
{
  todo_ui_data_t* todo_ui_data = (todo_ui_data_t*)user_data;
  delete todo_ui_data->calendar;
  delete todo_ui_data->todo;
  g_free(todo_ui_data);
}

static void
_save_todo_entry(GtkWidget* button, gpointer user_data)
{
  todo_ui_data_t* todo_ui_data = (todo_ui_data_t*)user_data;
  const gchar* title = hildon_entry_get_text(HILDON_ENTRY(todo_ui_data->todo_title_entry));
  if(!title || strlen(title)==0)
  {
    hildon_banner_show_information(NULL, NULL, dgettext("calendar", "cal_ib_add_title"));
    return;
  }
  string update_text(title);
  todo_ui_data->todo->setSummary(update_text);
  if(hildon_check_button_get_active(HILDON_CHECK_BUTTON(todo_ui_data->todo_done_check_button)))
    todo_ui_data->todo->setStatus(1);
  else
    todo_ui_data->todo->setStatus(0);

  struct tm start_time_stamp;
  start_time_stamp.tm_sec = start_time_stamp.tm_min = start_time_stamp.tm_hour = 0;
  
  guint year, month, day;
  hildon_date_button_get_date(HILDON_DATE_BUTTON(todo_ui_data->start_date_button),
			      &year,
			      &month,
			      &day);
  start_time_stamp.tm_year = year-1900;
  start_time_stamp.tm_mon = month;
  start_time_stamp.tm_mday = day;
  todo_ui_data->todo->setDateStart(mktime(&start_time_stamp));

  if(hildon_check_button_get_active(HILDON_CHECK_BUTTON(todo_ui_data->alarm_check_button)))
  {
    if(!todo_ui_data->todo->getAlarm())
    {
      todo_ui_data->todo->setAlarmBefore(0);
    }
    struct tm now;
    now.tm_sec = now.tm_min = now.tm_hour = 0;
    guint year, month, day, hour, min;
    
    hildon_date_button_get_date(HILDON_DATE_BUTTON(todo_ui_data->alarm_date_button),
				&year,
				&month,
				&day);
    hildon_time_button_get_time(HILDON_TIME_BUTTON(todo_ui_data->alarm_time_button),
				&hour,
				&min);

    now.tm_year = year-1900;
    now.tm_mon = month;
    now.tm_mday = day;
    now.tm_hour = hour;
    now.tm_min = min;
    now.tm_sec = 0;
    todo_ui_data->todo->getAlarm()->setTrigger(mktime(&now));
  }
  else
  {
    if(todo_ui_data->todo->getAlarm())
    {
      todo_ui_data->todo->removeAlarm();
    }
  }

  int error = 0;
  todo_ui_data->calendar->modifyTodo(todo_ui_data->todo, error);
  gtk_widget_hide(GTK_WIDGET(todo_ui_data->window));
}

static void
_show_hide_alarm_control(HildonCheckButton* button,
			 gpointer user_data)
{
  if(hildon_check_button_get_active(button))
  {
    gtk_widget_show(GTK_WIDGET(user_data));
  }
  else
  {
    gtk_widget_hide(GTK_WIDGET(user_data));
  }
}


void
open_calendar_todo_window(osso_context_t* osso, int calendar_id, const gchar* event_id)
{
  int error = 0;  
  CCalendar* calendar = CMulticalendar::MCInstance()->getCalendarById(calendar_id, error);
  if(calendar)
  {
    CTodo* todo = calendar->getTodo(event_id, error);
    if(todo)
    {
      GtkWidget* window;
      GtkWidget* todo_title_entry;
      GtkWidget* todo_done_check_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
      GtkWidget* save_entry_button;
      GtkWidget* alarm_check_button;
      GtkWidget* alarm_date_button;
      GtkWidget* alarm_time_button;
      GtkWidget* start_date_button;

      window = hildon_stackable_window_new();

      todo_title_entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);

      gtk_button_set_label(GTK_BUTTON(todo_done_check_button), dgettext("calendar", "cal_fi_done"));
      hildon_check_button_set_active(HILDON_CHECK_BUTTON(todo_done_check_button),
				     todo->getStatus()==1);


      alarm_check_button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
      alarm_date_button = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
      alarm_time_button = hildon_time_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
      start_date_button = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);

      
      time_t start_time_stamp = todo->getDateStart();
      struct tm* start_time_tm = localtime(&start_time_stamp);
      hildon_date_button_set_date(HILDON_DATE_BUTTON(start_date_button),
				  start_time_tm->tm_year + 1900,
				  start_time_tm->tm_mon,
				  start_time_tm->tm_mday);
      
      CAlarm* alarm = todo->getAlarm();
      if(alarm)
      {
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(alarm_check_button),
				       TRUE);
	time_t alarm_time_stamp = alarm->getTrigger();
	struct tm* alarm_time_tm = localtime(&alarm_time_stamp);
	hildon_date_button_set_date(HILDON_DATE_BUTTON(alarm_date_button),
				    alarm_time_tm->tm_year + 1900,
				    alarm_time_tm->tm_mon,
				    alarm_time_tm->tm_mday);
	hildon_time_button_set_time(HILDON_TIME_BUTTON(alarm_time_button),
				    alarm_time_tm->tm_hour,
				    alarm_time_tm->tm_min);
      }
      else
      {
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(alarm_check_button),
				       FALSE);
	time_t start_time = todo->getDateStart();
	struct tm* alarm_time_tm = localtime(&start_time);
	hildon_date_button_set_date(HILDON_DATE_BUTTON(alarm_date_button),
				    alarm_time_tm->tm_year + 1900,
				    alarm_time_tm->tm_mon,
				    alarm_time_tm->tm_mday-1);
	
	hildon_time_button_set_time(HILDON_TIME_BUTTON(alarm_time_button),
				    12,
				    0);
      }
      
      hildon_button_set_title(HILDON_BUTTON(alarm_date_button),
			      dgettext("calendar","cal_fi_alarm"));

      todo_ui_data_t* todo_ui_data = g_new0(todo_ui_data_t, 1);
      todo_ui_data->window = window;
      todo_ui_data->todo_title_entry = todo_title_entry;
      todo_ui_data->todo_done_check_button = todo_done_check_button;
      todo_ui_data->calendar = calendar;
      todo_ui_data->todo = todo;
      todo_ui_data->alarm_check_button = alarm_check_button;
      todo_ui_data->alarm_date_button = alarm_date_button;
      todo_ui_data->alarm_time_button = alarm_time_button;
      todo_ui_data->start_date_button = start_date_button;
      
      save_entry_button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
						    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      dgettext("hildon-libs", "wdgt_bd_save"), 
						    NULL);

      calendar_add_open_calendar_button_menu(window, osso, todo->getDateStart());
      g_signal_connect(save_entry_button, "clicked", G_CALLBACK(_save_todo_entry), todo_ui_data);
      g_signal_connect(window, "hide", G_CALLBACK(_delete_todo_ui_data), todo_ui_data);

      gtk_button_set_label(GTK_BUTTON(alarm_check_button), dgettext("calendar", "cal_fi_alarm"));
      GtkWidget* vbox = gtk_vbox_new(FALSE, HILDON_MARGIN_HALF);
      hildon_entry_set_text(HILDON_ENTRY(todo_title_entry),
			    todo->getSummary().c_str());

      gtk_box_pack_start(GTK_BOX(vbox), todo_title_entry, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), todo_done_check_button, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), start_date_button, TRUE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), alarm_check_button, TRUE, FALSE, 0);
      GtkWidget* alarm_time_stamp_box = gtk_hbox_new(FALSE, HILDON_MARGIN_HALF);
      g_signal_connect(alarm_check_button, "toggled", G_CALLBACK(_show_hide_alarm_control), alarm_time_stamp_box);
      gtk_box_pack_start(GTK_BOX(alarm_time_stamp_box), alarm_date_button, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(alarm_time_stamp_box), alarm_time_button, FALSE, FALSE, 0);
      gtk_box_pack_start(GTK_BOX(vbox), alarm_time_stamp_box, FALSE, FALSE, 0);
      GtkWidget* vbox2 = gtk_vbox_new(FALSE, HILDON_MARGIN_HALF);
      GtkWidget* pan = hildon_pannable_area_new();
      GtkWidget* alignment = gtk_alignment_new(0, 0, 1, 0);
      gtk_container_add(GTK_CONTAINER(alignment), vbox);
      hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(pan), alignment);
      gtk_box_pack_start(GTK_BOX(vbox2), pan, TRUE, TRUE, 0);
      gtk_box_pack_start(GTK_BOX(vbox2), save_entry_button, FALSE, FALSE, 0);
      gtk_container_add(GTK_CONTAINER(window), vbox2);
      gtk_window_set_title(GTK_WINDOW(window), dgettext("calendar", "cal_ti_edit_task"));
      gtk_widget_show_all(window);
      if(!todo->getAlarm())
      {
	gtk_widget_hide(alarm_time_stamp_box);
      }
    }
  }
}
