// CommonInputMethod.cpp
//
#include "CommonInputMethod.h"
#include "UnicodeStringList.h"
#include "PriorityList.h"


bool CCommonInputMethod::IsRoot(const wchar_t in_character)
{
	if(m_pRoots == NULL) return false;

	for(size_t i=0; i<m_nRoots && (m_pRoots[i] != 0x0000 || m_pRoots[i] != 0x000D || m_pRoots[i] != 0x000A); i++)
	{
		if(in_character == m_pRoots[i])
		{
			return true;
		}
	}

	return false;
}

#if defined(__WINDOWS__)
bool CCommonInputMethod::ReadFile(wchar_t* in_path)
{
	FILE*   stream;
	wchar_t	buffer[LINE_BUFFER_SIZE];

	errno_t err;
	if ((err = _wfopen_s(&stream, in_path, L"rt,ccs=UNICODE")) != 0)
	{
#ifdef OUTPUT_DEBUG_STRING
		OutputDebugString(L"_wfopen_s() failed!\n");
#endif
		return false;
	}


	// The first row of text file.
	fgetws(buffer, LINE_BUFFER_SIZE, stream);

	m_nRoots = ::GetLength(buffer);

	if(m_nRoots > 0)
	{
		m_pRoots = new wchar_t[m_nRoots+1];
		memset(m_pRoots, 0, (m_nRoots+1)*sizeof(wchar_t));
		memcpy(m_pRoots, buffer, m_nRoots*sizeof(wchar_t));
	}

	const wchar_t ucEqual = 0x003D;		// '='
	const wchar_t ucSepsrator = 0x0020;	// ' '

	while(fgetws(buffer, LINE_BUFFER_SIZE, stream) != NULL)
	{
		size_t				iEqual = 0;
		size_t				iEnd = 0;
		size_t				iWordStart = 0;
		size_t				iWordEnd = 0;
		wchar_t*			pRootString = NULL;
		CCandidateIndex*	pCI;
		bool				bAddCI = true;

		pCI = new CCandidateIndex();
		if(pCI == NULL) continue;

		for(size_t i=0; i<LINE_BUFFER_SIZE; i++)
		{
			if( (buffer[i] == 0x0000) || 
				(buffer[i] == 0x000D) || 
				(buffer[i] == 0x000A) )
			{
				iEnd = i;
				break;
			}
		}

		if(iEnd <= 0 || iEnd >= LINE_BUFFER_SIZE) continue;

		for(size_t i=0; i<iEnd; i++)
		{
			if(buffer[i] == ucEqual)
			{
				iEqual = i;
				break;
			}
		}

		if(iEqual <= 0) continue;

		pRootString = new wchar_t[iEqual+1];
		if(pRootString != NULL)
		{
			size_t ir = 0;
			memset(pRootString, 0, (iEqual+1)*sizeof(wchar_t));
			for(size_t i=0; i <iEqual; i++)
			{
				if(IsRoot(buffer[i]))
				{
					pRootString[ir] = buffer[i];
					ir++;
				}
			}

			if(ir == 0)
			{
				delete [] pRootString;
				pRootString = NULL;
				continue;
			}
			else
			{
				pCI->Root.SetString(pRootString);
				delete [] pRootString;
				pRootString = NULL;
			}
		}
		else
		{
			continue;
		}

		iWordStart = iEqual+1;
		iWordEnd = 0;
		bAddCI = true;
		while(iWordStart < iEnd)
		{

			//if(iWordStart >= iEnd) break;

			iWordEnd = iWordStart;
			for(size_t i=iWordStart; i <= iEnd; i++)
			{
				if( (buffer[i] == ucSepsrator) ||
					(buffer[i] == 0x0000) ||
					(buffer[i] == 0x000D) ||
					(buffer[i] == 0x000A) )
				{
					iWordEnd = i;
					break;
				}
			}

			if(iWordEnd > iEnd)
			{
				bAddCI = false;
				break;
			}

			if(iWordEnd == iWordStart)
			{
				iWordStart++;
			}
			else if(iWordEnd > iWordStart)
			{
				size_t nWordString = iWordEnd - iWordStart;

				wchar_t* pWordString = new wchar_t[nWordString+1];
				CCandidateWord* pCW = new CCandidateWord();
				if(pWordString != NULL && pCW != NULL)
				{

					memset(pWordString, 0, (nWordString+1)*sizeof(wchar_t));
					memcpy(pWordString, &(buffer[iWordStart]), nWordString*sizeof(wchar_t));

					pCW->Word.SetString(pWordString);
					pCW->SetPriority(0);
					delete [] pWordString;
					pWordString = NULL;
					nWordString = 0;

					pCI->CandidateWords.Add(pCW);

					pCW = NULL;
				}

				iWordStart = iWordEnd+1;
			}
		}

		if(bAddCI)
		{
			m_Indexes.Add(pCI);
			pCI = NULL;
		}
	}


	fclose(stream);

	return true;
}
#endif


#if defined(__LINUX__) || defined(__MACOSX__)
bool CCommonInputMethod::ReadFile(char* in_path)
{
	FILE*   stream;
	
	wchar_t	buffer[LINE_BUFFER_SIZE];
	memset(buffer, 0, LINE_BUFFER_SIZE*sizeof(wchar_t));
    
			    
	if ((stream = fopen(in_path, "rb")) == NULL)
	{
#ifdef OUTPUT_DEBUG_STRING
        OutputDebugString(L"fopen() failed!\r\n");
#endif
		return false;
	}
	
	BOM_TYPE type = hasBOM(stream);
	if(type == BOM_UTF8 || type== BOM_ERROR)
	{
        fclose(stream);
		return false;	
	}
	
        
	// The first row of text file.

	if(myFGetWS(buffer, LINE_BUFFER_SIZE, stream) == NULL)
    {
#ifdef OUTPUT_DEBUG_STRING
        OutputDebugString(L"fgetws() failed!\n");
#endif
		fclose(stream);
        return false;
    }
	
       

	m_nRoots = ::GetLength(buffer);

	if(m_nRoots > 0)
	{
		m_pRoots = new wchar_t[m_nRoots+1];
		memset(m_pRoots, 0, (m_nRoots+1)*sizeof(wchar_t));
		memcpy(m_pRoots, buffer, m_nRoots*sizeof(wchar_t));
	}

	const wchar_t ucEqual = 0x003D;		// '='
	const wchar_t ucSepsrator = 0x0020;	// ' '

	while(myFGetWS(buffer, LINE_BUFFER_SIZE, stream) != NULL)
	{
		size_t				iEqual = 0;
		size_t				iEnd = 0;
		size_t				iWordStart = 0;
		size_t				iWordEnd = 0;
		wchar_t*			pRootString = NULL;
		CCandidateIndex*	pCI;
		bool				bAddCI = true;

		pCI = new CCandidateIndex();
		if(pCI == NULL) continue;

		for(size_t i=0; i<LINE_BUFFER_SIZE; i++)
		{
			if( (buffer[i] == 0x0000) || 
				(buffer[i] == 0x000D) || 
				(buffer[i] == 0x000A) )
			{
				iEnd = i;
				break;
			}
		}

		if(iEnd <= 0 || iEnd >= LINE_BUFFER_SIZE) continue;

		for(size_t i=0; i<iEnd; i++)
		{
			if(buffer[i] == ucEqual)
			{
				iEqual = i;
				break;
			}
		}

		if(iEqual <= 0) continue;

		pRootString = new wchar_t[iEqual+1];
		if(pRootString != NULL)
		{
			size_t ir = 0;
			memset(pRootString, 0, (iEqual+1)*sizeof(wchar_t));
			for(size_t i=0; i <iEqual; i++)
			{
				if(IsRoot(buffer[i]))
				{
					pRootString[ir] = buffer[i];
					ir++;
				}
			}

			if(ir == 0)
			{
				delete [] pRootString;
				pRootString = NULL;
				continue;
			}
			else
			{
				pCI->Root.SetString(pRootString);
				delete [] pRootString;
				pRootString = NULL;
			}
		}
		else
		{
			continue;
		}

		iWordStart = iEqual+1;
		iWordEnd = 0;
		bAddCI = true;
		while(iWordStart < iEnd)
		{

			//if(iWordStart >= iEnd) break;

			iWordEnd = iWordStart;
			for(size_t i=iWordStart; i <= iEnd; i++)
			{
				if( (buffer[i] == ucSepsrator) ||
					(buffer[i] == 0x0000) ||
					(buffer[i] == 0x000D) ||
					(buffer[i] == 0x000A) )
				{
					iWordEnd = i;
					break;
				}
			}

			if(iWordEnd > iEnd)
			{
				bAddCI = false;
				break;
			}

			if(iWordEnd == iWordStart)
			{
				iWordStart++;
			}
			else if(iWordEnd > iWordStart)
			{
				size_t nWordString = iWordEnd - iWordStart;

				wchar_t* pWordString = new wchar_t[nWordString+1];
				CCandidateWord* pCW = new CCandidateWord();
				if(pWordString != NULL && pCW != NULL)
				{

					memset(pWordString, 0, (nWordString+1)*sizeof(wchar_t));
					memcpy(pWordString, &(buffer[iWordStart]), nWordString*sizeof(wchar_t));

					pCW->Word.SetString(pWordString);
					pCW->SetPriority(0);
					delete [] pWordString;
					pWordString = NULL;
					nWordString = 0;

					pCI->CandidateWords.Add(pCW);

					pCW = NULL;
				}

				iWordStart = iWordEnd+1;
			}
		}

		if(bAddCI)
		{
			m_Indexes.Add(pCI);
			pCI = NULL;
		}
	}


	fclose(stream);

	return true;
}
#endif



// Listing all roots and candidate words.
bool CCommonInputMethod::Dump()
{
#ifdef OUTPUT_DEBUG_STRING
	OutputDebugString(L"CCommonInputMethod::Dump()\r\n");

	ListNode*			pNode = NULL;
	CCandidateIndex*	pData = NULL;
	wchar_t			buffer[LINE_BUFFER_SIZE];
	wchar_t			string_buffer[10];
	size_t length = 0;

	while((pData = m_Indexes.DumpData(&pNode)) != NULL)
	{
		memset(buffer, 0, LINE_BUFFER_SIZE*sizeof(wchar_t));
		swprintf(buffer, LINE_BUFFER_SIZE, L"%u:", pData->GetID());
		OutputDebugString(buffer);

		memset(buffer, 0, LINE_BUFFER_SIZE*sizeof(wchar_t));
		if(pData->Root.GetString(buffer, LINE_BUFFER_SIZE, &length))
		{
			OutputDebugString(buffer);
			OutputDebugString(L" = ");


			ListNode*			pSubNode = NULL;
			CCandidateWord*	pSubData = NULL;

			memset(buffer, 0, LINE_BUFFER_SIZE*sizeof(wchar_t));
			while((pSubData = pData->CandidateWords.DumpData(&pSubNode)) != NULL)
			{
				memset(string_buffer, 0, 10*sizeof(wchar_t));
				if(pSubData->Word.GetString(string_buffer, 10, &length))
				{
					memset(buffer, 0, LINE_BUFFER_SIZE*sizeof(wchar_t));
					swprintf(buffer, LINE_BUFFER_SIZE, L"%u:", pSubData->GetID());
					OutputDebugString(buffer);
					
					OutputDebugString(string_buffer);
					
					swprintf(buffer, LINE_BUFFER_SIZE, L"(%d), ", pSubData->GetPriority());
					OutputDebugString(buffer);
				}
			}

			OutputDebugString(L"\r\n");
		}
	}

#endif
	return true;
}


// Getting candidate word list by root(zero ending unicode string).
bool CCommonInputMethod::GetCandidateWordsByRoot(
	wchar_t* pRoots,					// in
	bool bIgnoreCase,					// in
	CCandidateIndex** ppCandidateIndex)	// out
{
	if(ppCandidateIndex == NULL) return false;
	if(pRoots == NULL) return false;
	
	CCandidateIndex ciSearchPattern;

	bool result = false;

	LLEqualOption nOption = (bIgnoreCase == true) ? BY_STRING_IGNORECASE : BY_STRING;


	*ppCandidateIndex = NULL;

	ciSearchPattern.Root.SetString(pRoots);

	*ppCandidateIndex = m_Indexes.FindCandidateIndex(&ciSearchPattern, nOption);


	if(*ppCandidateIndex == NULL)
		result = false;
	else
		result = true;

	return result;
}


bool CCommonInputMethod::DumpCandidateWords(
	CCandidateIndex* pCandidateIndex)	// in
{
#ifdef OUTPUT_DEBUG_STRING
	OutputDebugString(L"CCommonInputMethod::DumpCandidateWords()\r\n");

	if(pCandidateIndex != NULL)
	{
		ListNode*				pNode = NULL;
		CCandidateWord*		pData = NULL;
		wchar_t				buffer[LINE_BUFFER_SIZE];
		wchar_t				string_buffer[CANDIDATEWORD_BUFFER_SIZE];
		CCandidateWordList* pWordList = &(pCandidateIndex->CandidateWords);
		size_t length = 0;

		
		swprintf(buffer, LINE_BUFFER_SIZE, L"%u:", pCandidateIndex->GetID());
		OutputDebugString(buffer);
		pCandidateIndex->Root.ShowDebugString();
		OutputDebugString(L" = ");

		while((pData = pWordList->DumpData(&pNode)) != NULL)
		{
			memset(string_buffer, 0, CANDIDATEWORD_BUFFER_SIZE*sizeof(wchar_t));
			if(pData->Word.GetString(string_buffer, CANDIDATEWORD_BUFFER_SIZE, &length))
			{
				memset(buffer, 0, LINE_BUFFER_SIZE*sizeof(wchar_t));
				swprintf(buffer, LINE_BUFFER_SIZE, L"%u:", pData->GetID());
				OutputDebugString(buffer);
				
				OutputDebugString(string_buffer);
				
				swprintf(buffer, LINE_BUFFER_SIZE, L"(%d), ", pData->GetPriority());
				OutputDebugString(buffer);
			}
		}
		OutputDebugString(L"\r\n");
		return true;
	}
	else
	{
		OutputDebugString(L"No candidate word.\r\n");
		return false;
	}
#endif

	return true;
}



size_t CCommonInputMethod::GetCountOfCandidateWords(CCandidateIndex* pCandidateIndex)
{
	if(pCandidateIndex == NULL)
		return 0;
	else
		return pCandidateIndex->CandidateWords.GetCount();
}


bool CCommonInputMethod::GetCandidateWordsArray(CCandidateIndex* pCandidateIndex, CandidateWordBuffer* pCWBuffer, size_t uBufferSize)
{
	if(pCWBuffer == NULL) return false;

	if(pCandidateIndex != NULL)
	{
		CCandidateWordList* pWordList = &(pCandidateIndex->CandidateWords);
		ListNode*				pNode = NULL;
		CCandidateWord*		pData = NULL;
		size_t length = 0;
		size_t i = 0;


		if(pWordList->GetCount() > uBufferSize) return false;
		if(pWordList->GetCount() == 0) return true;
		
		while((pData = pWordList->DumpData(&pNode)) != NULL)
		{
			memset(pCWBuffer[i].CandidateWord, 0, CANDIDATEWORD_BUFFER_SIZE*sizeof(wchar_t));

			pData->Word.GetString(pCWBuffer[i].CandidateWord, CANDIDATEWORD_BUFFER_SIZE, &length);
			pCWBuffer[i].ID = pData->GetID();
			pCWBuffer[i].Priority = pData->GetPriority();

			i++;
		}

		return true;
	}
	else
	{
#ifdef OUTPUT_DEBUG_STRING
		OutputDebugString(L"No candidate word.\r\n");
#endif
		return false;
	}
}


bool CCommonInputMethod::SelectCandidateWord(
	CCandidateIndex* pCandidateIndex,		// in
	unsigned int uID,						// in
	CandidateWordBuffer* pCandidateWord)	// out
{
#ifdef OUTPUT_DEBUG_STRING
			OutputDebugString(L"CCommonInputMethod::SelectCandidateWord()\r\n");
#endif

	if(pCandidateIndex == NULL) return false;

	CCandidateWord* pData = pCandidateIndex->CandidateWords.SelectCandidateWord(&m_Priorities, pCandidateIndex->GetID(), uID);
	
	if(pData != NULL)
	{
		if(pCandidateWord != NULL)
		{
			size_t length = 0; 
			pCandidateWord->ID = uID;
			pData->Word.GetString(pCandidateWord->CandidateWord, CANDIDATEWORD_BUFFER_SIZE, &length);
			pCandidateWord->Priority = pData->GetPriority();
			
#ifdef OUTPUT_DEBUG_STRING
			if(length == 0)
			OutputDebugString(L"pCandidateWord->CandidateWord : out of buffer.\r\n");
#endif
			
		}
	
#ifdef OUTPUT_DEBUG_STRING
		//pData->Word.ShowDebugString();
		//OutputDebugString(L"\r\n");
#endif
		return true;
	}
	else
	{
		return false;
	}
}


bool CCommonInputMethod::SelectCandidateWord(
	unsigned int uCandidateIndexID,			// in
	unsigned int uCandidateWordID,			// in
	CandidateWordBuffer* pCandidateWord)	// out
{
	CCandidateIndex* pCI = NULL;

	pCI = FindCandidateIndexByID(uCandidateIndexID);

	return this->SelectCandidateWord(pCI, uCandidateWordID, pCandidateWord);
}


// parameter - LLCompareOption nOption
//		DES_BY_ID : Descending by ID. Ex: 6, 6, 5, 2, 1, 0, ...
//		ASC_BY_ID : Ascending by ID. Ex: 0, 0, ...., 1, 2, 5, 6, 6
//		DES_BY_PRIORITY : Descending by priority. Ex: 5, 4, 3, 2, 1
//		ASC_BY_PRIORITY : Ascending by priority. Ex: 1, 2, 3, 4, 5
bool CCommonInputMethod::SortCandidateWords(CCandidateIndex* pCandidateIndex, LLCompareOption nOption)
{
	if(pCandidateIndex == NULL) return false;

	return pCandidateIndex->CandidateWords.Sort(nOption);
}
//-----------------------------------------------------------------------------------


bool CCommonInputMethod::GetAvailableRoots(
	wchar_t* pInitialCharacters,		// in
	bool bIgnoreCase,					// in
	CUnicodeStringList* pStringList)	// out
{
	if(pStringList == NULL)
		return false;

	ListNode* pNode = NULL;
	CCandidateIndex* pData = NULL;
	wchar_t ucRoot[2];
	CUnicodeString* pString = NULL;


	if(pStringList->GetCount() > 0)
	{
		pStringList->Clear();
	}

#ifdef OUTPUT_DEBUG_STRING
	OutputDebugString(pInitialCharacters);
	OutputDebugString(L" => ");
#endif

	while((pData = m_Indexes.DumpData(&pNode))!= NULL)
	{
		if(pData->Root.StartsWith(pInitialCharacters, bIgnoreCase))
		{
			memset(ucRoot, 0, 2*sizeof(wchar_t));
			size_t length = ::GetLength(pInitialCharacters);

			if(pData->Root.GetWChar(length, &ucRoot[0]))
			{
				pString = new CUnicodeString();
				if(pString != NULL)
				{
					pString->SetString(ucRoot);

					if(pStringList->Contains(pString, bIgnoreCase) == false)
					{
						pStringList->Add(pString);
#ifdef OUTPUT_DEBUG_STRING
						OutputDebugString(ucRoot);
						OutputDebugString(L", ");
#endif
					}

					pString = NULL;
				}
			}
			else
			{}
		}
		else
		{}
	}


	if(pStringList->GetCount() > 0)
	{
#ifdef OUTPUT_DEBUG_STRING
		OutputDebugString(L"\r\n");
#endif
		return true;
	}
	else
	{
		pStringList->Clear();
#ifdef OUTPUT_DEBUG_STRING
		OutputDebugString(L"No roots.\r\n");
#endif

		return false;
	}
}

size_t CCommonInputMethod::GetCountOfAvailableRoots(
	CUnicodeStringList* pStringList)	// in
{
	if(pStringList == NULL)
		return 0;
	else
		return pStringList->GetCount();
}

bool CCommonInputMethod::GetAvailableRootsArray(
	CUnicodeStringList* pStringList,	// in
	wchar_t* pBuffer,					// out
	size_t uBufferSize)					// in
{
	if(pStringList == NULL) return false;
	if(pBuffer == NULL) return false;

	if(pStringList->GetCount() > uBufferSize) return false;

	memset(pBuffer, 0, sizeof(wchar_t)*uBufferSize);


	if(pStringList->GetCount() == 0) return true;

	size_t i = 0;

	ListNode* pNode = NULL;
	CUnicodeString* pData = NULL;

	while((pData = pStringList->DumpData(&pNode)) != NULL)
	{
		if(pData->GetWChar(0, &(pBuffer[i])) == false)	return false;
		i++;
	}

	return true;
}


//-----------------------------------------------------------------------------------


// parameter - LLCompareOption nOption
//		DES_BY_STRING : Descending by string.
//		ASC_BY_STRING : Ascending by string.
//		DES_BY_STRING_IGNORECASE : Descending by string (ignore case).
//		ASC_BY_STRING_IGNORECASE : Ascending by string (ignore case).
//		DES_BY_ID : Descending by ID. Ex: 6, 6, 5, 2, 1, 0, ...
//		ASC_BY_ID : Ascending by ID. Ex: 0, 0, ...., 1, 2, 5, 6, 6
void CCommonInputMethod::SortIndexes(LLCompareOption nOption)
{
	m_Indexes.Sort(nOption);
}

//-----------------------------------------------------------------------------------


// Listing Priority List.
bool CCommonInputMethod::DumpPriorityList()
{
#ifdef OUTPUT_DEBUG_STRING
	OutputDebugString(L"CCommonInputMethod::DumpPriorityList()\r\n");

	ListNode*			pNode = NULL;
	PriorityData*	pData = NULL;

	wchar_t			buffer[LINE_BUFFER_SIZE];

	OutputDebugString(L"(CandidateIndexID, CandidateWordID) = Priority\r\n");
	while((pData = m_Priorities.DumpData(&pNode)) != NULL)
	{
		memset(buffer, 0, LINE_BUFFER_SIZE*sizeof(wchar_t));
		swprintf(buffer, LINE_BUFFER_SIZE, L"(%u, %u) = %d\r\n", 
			pData->CandidateIndexID,
			pData->CandidateWordID,
			pData->Priority);
		OutputDebugString(buffer);
	}

#endif
	return true;
}

// Find candidate index by ID.
CCandidateIndex* CCommonInputMethod::FindCandidateIndexByID(unsigned int uCandidateIndexID)
{
	ListNode*				pNode = NULL;
	CCandidateIndex*	pData = NULL;

	while((pData = m_Indexes.DumpData(&pNode)) != NULL)
	{
		if(pData->GetID() == uCandidateIndexID)
		{
			return pData;	
		}
	}

	return NULL;
}

CCandidateWord* CCommonInputMethod::FindCandidateWordByIDs(
	unsigned int uCandidateIndexID,
	unsigned int uCandidateWordID)
{
	CCandidateIndex* pCI = NULL;
	CCandidateWord* pCW = NULL;

	pCI = FindCandidateIndexByID(uCandidateIndexID);

	if(pCI != NULL)
	{
		pCW = pCI->CandidateWords.FindCandidateWordByID(uCandidateWordID);

		return pCW;
	}

	return NULL;
}

//-----------------------------------------------------------------------------------

#define CTP_FILETYPE	"CTP"
#define CTP_VERSION		1.0

#define CTP_MASK_SIZE	0x10041902
#define CTP_MASK_DATA	0x12110539

// Character Table Priority File:
// +-----------+
// | CTPHeader |
// +-----------+
// | CTPData   |
// +-----------+
// | CTPData   |
// +-----------+
// | ...       |
// +-----------+

struct CTPHeader
{
	unsigned int	StructureSize;
	char			FileType[4];
	float			Version;	
	size_t			DataCount;
};

struct CTPData
{
	unsigned int	StructureSize;
	unsigned int	CandidateIndexID;
	unsigned int	CandidateWordID;
	int				Priority;
};

bool CCommonInputMethod::LoadPriorities(const char* pFilename)
{
	FILE* stream = NULL;

	if(pFilename == NULL) return false;

	if((stream = fopen(pFilename, "rb")) != NULL)
	{
		CTPHeader header;
		CTPData data;

		fread(&header, sizeof(CTPHeader), 1, stream);
		
		// decoding values.
		header.StructureSize ^= CTP_MASK_SIZE;
		header.DataCount ^= CTP_MASK_DATA;	

		if( sizeof(CTPHeader) == 0x10 &&  // different compiler maybe define different size value type.
			sizeof(CTPData) == 0x10 && // different compiler maybe define different size value type.
			strcmp(header.FileType, CTP_FILETYPE) == 0 &&
			header.Version == CTP_VERSION &&
			header.DataCount > 0 )
		{
			m_Priorities.Clear();

			for(size_t i=0; i<header.DataCount; i++)
			{
				fread(&data, sizeof(CTPData), 1, stream);

				// decoding values.
				data.StructureSize ^= CTP_MASK_SIZE;
				data.CandidateIndexID ^= CTP_MASK_DATA;
				data.CandidateWordID ^= CTP_MASK_DATA;
				data.Priority ^= CTP_MASK_DATA;


				// add new data to priority list.
				PriorityData* pNewData = new PriorityData;
				if(pNewData != NULL)
				{
					pNewData->CandidateIndexID = data.CandidateIndexID;
					pNewData->CandidateWordID = data.CandidateWordID;
					pNewData->Priority = data.Priority;

					m_Priorities.Add(pNewData);

					// hook new data to candidate index list.
					CCandidateWord* pData = FindCandidateWordByIDs(data.CandidateIndexID, data.CandidateWordID);

					if(pData != NULL)
					{
						pData->PriorityDataPtr = pNewData;
						pData->SetPriority(data.Priority);
					}
				}
			}
		
		}

		fclose(stream);
		return true;
	}

	return false;
}


bool CCommonInputMethod::SavePriorities(const char * pFilename)
{
	FILE* stream = NULL;

	if(pFilename == NULL) return false;

	if ((stream = fopen(pFilename, "wb")) != NULL)
	{
		CTPHeader header;
		CTPData data;

		header.StructureSize = sizeof(CTPHeader);
		strcpy(header.FileType, CTP_FILETYPE);
		header.Version = CTP_VERSION;
		header.DataCount = this->m_Priorities.GetCount();
		
		// encoding values.
		header.StructureSize ^= CTP_MASK_SIZE;
		header.DataCount ^= CTP_MASK_DATA;

		fwrite(&header, sizeof(CTPHeader), 1, stream);

		// decoding values.
		header.StructureSize ^= CTP_MASK_SIZE;
		header.DataCount ^= CTP_MASK_DATA;

		if(header.DataCount > 0)
		{
			ListNode*			pNode = NULL;
			PriorityData*	pData = NULL;

			data.StructureSize = sizeof(CTPData);
			while((pData = m_Priorities.DumpData(&pNode)) != NULL)
			{
				data.CandidateIndexID = pData->CandidateIndexID;
				data.CandidateWordID = pData->CandidateWordID;
				data.Priority = pData->Priority;


				// encoding values.
				data.StructureSize ^= CTP_MASK_SIZE;
				data.CandidateIndexID ^= CTP_MASK_DATA;
				data.CandidateWordID ^= CTP_MASK_DATA;
				data.Priority ^= CTP_MASK_DATA;

				fwrite(&data, sizeof(CTPData), 1, stream);

				// decoding values.
				data.StructureSize ^= CTP_MASK_SIZE;
				data.CandidateIndexID ^= CTP_MASK_DATA;
				data.CandidateWordID ^= CTP_MASK_DATA;
				data.Priority ^= CTP_MASK_DATA;

			}
		}

		fclose(stream);
		return true;
	}


	return false;
}


//-----------------------------------------------------------------------------------

#define CTB_FILETYPE	"CTB"
#define CTB_VERSION		1.0

#define CTB_MASK_SIZE	0x10041902
#define CTB_MASK_DATA	0x12141404
#define CTB_MASK_BYTE	0x63


// Character Table Binary File
// +-----------------------------+
// |          CTBHeader          |
// +.............................+
// |      CTBRootsArrayData      |
// +-----------------------------+
// |    CTBCandidateIndexData    |
// +.............................+
// |          CTBRootData        |
// +-----------------------------+
// | +-------------------------+ |
// | |   CTBCandidateWordData  | |
// | +.........................+ |
// | |       CTBWordData       | |
// | +-------------------------+ |
// | |   CTBCandidateWordData  | |
// | +.........................+ |
// | |       CTBWordData       | |
// | +-------------------------+ |
// | |          ......         | |
// | +-------------------------+ |
// +-----------------------------+
// |    CTBCandidateIndexData    |
// +.............................+
// |          CTBRootData        |
// +-----------------------------+
// | +-------------------------+ |
// | |   CTBCandidateWordData  | |
// | +.........................+ |
// | |       CTBWordData       | |
// | +-------------------------+ |
// | |   CTBCandidateWordData  | |
// | +.........................+ |
// | |       CTBWordData       | |
// | +-------------------------+ |
// | |          ......         | |
// | +-------------------------+ |
// +-----------------------------+
// |            ......           |
// +-----------------------------+

struct CTBHeader
{
	unsigned int	StructureSize;
	char			FileType[4];
	float			Version;

//	CCommonInputMethod
	size_t			m_nRoots;	// m_nRoots+1 == size of CTBRootsArrayData

//	CCandidateIndexList
	size_t			CandidateIndexCount;	// count of CTBCandidateIndexData
	unsigned int	m_CandidateIndexID_Counter;

};

//UTF32Bytes*	CTBRootsArrayData;	// Zero ending unicode string.

struct CTBCandidateIndexData
{
	unsigned int	StructureSize;

//	CCandidateIndex
	unsigned int	CandidateIndexID;
	size_t			RootLength;		// RootLength+1 == size of CTBRootData;

//	CCandidateWordList
	unsigned int	m_CandidateWordID_Counter;
	size_t			CandidateWordCount;		// count of CTBCandidateWordData
};

//UTF32Bytes*	CTBRootData;	// Zero ending unicode string.


struct CTBCandidateWordData
{
	unsigned int	StructureSize;

//	CCandidateWord
	unsigned int	m_ID;

	size_t			WordLength;	// WordLength+1 == size of CTBWordData
};

//UTF32Bytes*	CTBWordData;	// Zero ending unicode string.


bool CCommonInputMethod::SaveCTB(const char * pFilename)
{
#ifdef OUTPUT_DEBUG_STRING
	OutputDebugString(L"CCommonInputMethod::SaveCTB()\r\n");
#endif

	FILE* stream = NULL;

	if(pFilename == NULL) return false;

	if ((stream = fopen(pFilename, "wb")) != NULL)
	{
//-------------------------------------------------------
		CTBHeader header;
		UTF32Bytes* CTBRootsArrayData = NULL; // Zero ending unicode string.

		header.StructureSize = sizeof(CTBHeader);
		strcpy(header.FileType, CTB_FILETYPE);
		header.Version = CTB_VERSION;

		header.m_nRoots = this->m_nRoots;
		if(this->m_pRoots == NULL)
		{
			fclose(stream);
			return false;
		}

		header.CandidateIndexCount = m_Indexes.GetCount();
		header.m_CandidateIndexID_Counter = m_Indexes.GetCandidateIndexIDCounter();


		if(header.m_nRoots > 0)
		{
			CTBRootsArrayData = new UTF32Bytes[header.m_nRoots+1];
			if(CTBRootsArrayData != NULL)
			{
				memset(CTBRootsArrayData, 0, sizeof(UTF32Bytes)*(header.m_nRoots+1));

				for(size_t i=0; i<header.m_nRoots; i++)
				{
					if(::wchar_tToUTF32Bytes(&(CTBRootsArrayData[i]), &(this->m_pRoots[i])) == false)
					{
						header.m_nRoots = 0;			
						break;
					}
				}
			}
			else
			{
				header.m_nRoots = 0;			
			}
		}


		// encoding values
		header.StructureSize ^= CTB_MASK_SIZE;
		header.m_nRoots ^= CTB_MASK_DATA;
		header.m_CandidateIndexID_Counter ^= CTB_MASK_DATA;

// CTBHeader
		fwrite(&header, sizeof(CTBHeader), 1, stream);

		// decoding values
		header.StructureSize ^= CTB_MASK_SIZE;
		header.m_nRoots ^= CTB_MASK_DATA;
		header.m_CandidateIndexID_Counter ^= CTB_MASK_DATA;

		if(header.m_nRoots > 0)
		{
			// encoding values, except last zero value byte.
			for(size_t i=0; i<header.m_nRoots; i++)
			{
				CTBRootsArrayData[i].byte0 ^= CTB_MASK_BYTE;
				CTBRootsArrayData[i].byte1 ^= CTB_MASK_BYTE;
			}

// CTBRootsArrayData
			fwrite(CTBRootsArrayData, sizeof(UTF32Bytes), header.m_nRoots+1, stream);
		}

		if(CTBRootsArrayData != NULL)
		{
			delete [] CTBRootsArrayData;
			CTBRootsArrayData = NULL;
		}

//-------------------------------------------------------
		CTBCandidateIndexData ci;
		UTF32Bytes* CTBRootData = NULL; // Zero ending unicode string.
		CTBCandidateWordData cw;
		UTF32Bytes* CTBWordData = NULL; // Zero ending unicode string.

		if(header.CandidateIndexCount > 0)
		{
			ListNode*				pNode = NULL;
			CCandidateIndex*	pData = NULL;

			ci.StructureSize = sizeof(CTBCandidateIndexData);
			while((pData = m_Indexes.DumpData(&pNode)) != NULL)
			{
				//	CCandidateIndex
				ci.CandidateIndexID = pData->GetID();
				ci.RootLength = pData->Root.GetLength();

				//	CCandidateWordList
				ci.m_CandidateWordID_Counter = pData->CandidateWords.GetCandidateWordIDCounter();
				ci.CandidateWordCount = pData->CandidateWords.GetCount();


				if(ci.RootLength > 0)
				{
					CTBRootData = new UTF32Bytes[ci.RootLength + 1];
					if(CTBRootData != NULL)
					{
						if(pData->Root.ConvertToUTF32Bytes(CTBRootData, ci.RootLength + 1) == false)
						{
							ci.RootLength = 0;
						}
					}
				}


				// encoding values
				ci.StructureSize ^= CTB_MASK_SIZE;
				ci.CandidateWordCount ^= CTB_MASK_DATA;
				ci.m_CandidateWordID_Counter ^= CTB_MASK_DATA;

				// CTBCandidateIndexData
				fwrite(&ci, sizeof(CTBCandidateIndexData), 1, stream);

				// decoding values
				ci.StructureSize ^= CTB_MASK_SIZE;
				ci.CandidateWordCount ^= CTB_MASK_DATA;
				ci.m_CandidateWordID_Counter ^= CTB_MASK_DATA;

				if(ci.RootLength > 0)
				{
					// encoding values, except last zero value byte.
					for(size_t i=0; i<ci.RootLength; i++)
					{
						CTBRootData[i].byte0 ^= CTB_MASK_BYTE;
						CTBRootData[i].byte1 ^= CTB_MASK_BYTE;
					}

					// CTBRootData
					fwrite(CTBRootData, sizeof(UTF32Bytes), ci.RootLength+1, stream);
				}

				if(CTBRootData != NULL)
				{
					delete [] CTBRootData;
					CTBRootData = NULL;
				}

//....................................................................................

				if(ci.CandidateWordCount > 0)
				{
					ListNode*			pSubNode = NULL;
					CCandidateWord*	pSubData = NULL;
				
					cw.StructureSize = sizeof(CTBCandidateWordData);
					while((pSubData = pData->CandidateWords.DumpData(&pSubNode)) != NULL)
					{
						cw.m_ID = pSubData->GetID();
						cw.WordLength = pSubData->Word.GetLength();

						if(cw.WordLength > 0)
						{

							CTBWordData = new UTF32Bytes[cw.WordLength + 1];
							if(CTBWordData != NULL)
							{
								if(pSubData->Word.ConvertToUTF32Bytes(CTBWordData, cw.WordLength + 1) == false)
								{
									cw.WordLength = 0;
								}
							}

						}



						 // encoding values
						cw.StructureSize ^= CTB_MASK_SIZE;
						cw.m_ID ^= CTB_MASK_DATA;

						// CTBCandidateWordData
						fwrite(&cw, sizeof(CTBCandidateWordData), 1, stream);

						 // decoding values
						cw.StructureSize ^= CTB_MASK_SIZE;
						cw.m_ID ^= CTB_MASK_DATA;

						if(cw.WordLength > 0)
						{
							// encoding values, except last zero value byte.
							for(size_t i=0; i<cw.WordLength; i++)
							{
								CTBWordData[i].byte0 ^= CTB_MASK_BYTE;
								CTBWordData[i].byte1 ^= CTB_MASK_BYTE;
							}

							// CTBWordData
							fwrite(CTBWordData, sizeof(UTF32Bytes), cw.WordLength+1, stream);
						}
						
						if(CTBWordData != NULL)
						{
							delete [] CTBWordData;
							CTBWordData = NULL;
						}

					}
				}

			}
		}

		fclose(stream);
		return true;
	}


	return false;
}


bool CCommonInputMethod::LoadCTB(const char * pFilename)
{
#ifdef OUTPUT_DEBUG_STRING
	OutputDebugString(L"CCommonInputMethod::LoadCTB()\r\n");
#endif

	FILE* stream = NULL;

	if(pFilename == NULL) return false;


	if ((stream = fopen(pFilename, "rb")) != NULL)
	{
//-------------------------------------------------------

// CTBHeader		
		CTBHeader header;
		UTF32Bytes* CTBRootsArrayData = NULL;
		CTBCandidateIndexData ci;
		UTF32Bytes*	CTBRootData = NULL;
		CTBCandidateWordData cw;
		UTF32Bytes*	CTBWordData = NULL;


		fread(&header, sizeof(CTBHeader), 1, stream);
		// decoding values
		header.StructureSize ^= CTB_MASK_SIZE;
		header.m_nRoots ^= CTB_MASK_DATA;
		header.m_CandidateIndexID_Counter ^= CTB_MASK_DATA;


		if( (header.StructureSize == sizeof(CTBHeader)) &&
			(strcmp(header.FileType, CTB_FILETYPE) == 0) &&
			(header.Version == CTB_VERSION) &&
			(header.CandidateIndexCount > 0) )
		{}
		else
		{
			fclose(stream);
			return false;
		}


// CTBRootsArrayData

		if(this->m_pRoots != NULL)
		{
			delete [] this->m_pRoots;
			this->m_pRoots = NULL;
			this->m_nRoots = 0;
		}


		this->m_nRoots = header.m_nRoots;
		if(header.m_nRoots > 0)
		{
			this->m_pRoots = new wchar_t[header.m_nRoots+1];
			CTBRootsArrayData = new UTF32Bytes[header.m_nRoots+1];

			if( (this->m_pRoots != NULL) && 
				(CTBRootsArrayData != NULL) )
			{	
				memset(this->m_pRoots, 0, sizeof(wchar_t)*(header.m_nRoots+1));
				memset(CTBRootsArrayData, 0, sizeof(UTF32Bytes)*(header.m_nRoots+1));
				
				fread(CTBRootsArrayData, sizeof(UTF32Bytes), header.m_nRoots+1, stream);

				for(size_t i=0; i<header.m_nRoots; i++)
				{
					// decoding values
					CTBRootsArrayData[i].byte0 ^= CTB_MASK_BYTE;
					CTBRootsArrayData[i].byte1 ^= CTB_MASK_BYTE;

					::UTF32BytesTowchar_t( &(this->m_pRoots[i]), &(CTBRootsArrayData[i]) );
				}
			}

			if(CTBRootsArrayData != NULL)
			{
				delete [] CTBRootsArrayData;
				CTBRootsArrayData = NULL;
			}
		}
		else
		{
			fclose(stream);
			return false;
		}

//-------------------------------------------------------

		if(header.CandidateIndexCount > 0)
		{
			this->m_Indexes.Clear();

			m_Indexes.SetCandidateIndexIDCounter(header.m_CandidateIndexID_Counter);

// CTBCandidateIndexData
			for(size_t i=0; i<header.CandidateIndexCount; i++)
			{
				fread(&ci, sizeof(CTBCandidateIndexData), 1, stream);
				
				// decoding values
				ci.StructureSize ^= CTB_MASK_SIZE;
				ci.CandidateWordCount ^= CTB_MASK_DATA;
				ci.m_CandidateWordID_Counter ^= CTB_MASK_DATA;

				
				if(ci.StructureSize != sizeof(CTBCandidateIndexData))
				{
					fclose(stream);
					return false;
				}

				// CCandidateIndex
				CCandidateIndex* pCI = NULL;
				pCI = new CCandidateIndex();
				if(pCI == NULL)
				{
					fclose(stream);
					return false;
				}

				pCI->SetID(ci.CandidateIndexID);


				if(ci.RootLength > 0)
				{
// CTBRootData
					CTBRootData = new UTF32Bytes[ci.RootLength+1];
					if(CTBRootData != NULL)
					{
						memset(CTBRootData, 0, sizeof(UTF32Bytes)*(ci.RootLength+1));
						fread(CTBRootData, sizeof(UTF32Bytes), ci.RootLength+1, stream);

						// decoding values, except last zero value byte.
						for(size_t i=0; i<ci.RootLength; i++)
						{
							CTBRootData[i].byte0 ^= CTB_MASK_BYTE;
							CTBRootData[i].byte1 ^= CTB_MASK_BYTE;
						}


						pCI->Root.SetString(CTBRootData);

						delete [] CTBRootData;
						CTBRootData = NULL;
					}
				}



				// CCandidateWordList
				pCI->CandidateWords.SetCandidateWordIDCounter(ci.m_CandidateWordID_Counter);


// CTBCandidateWordData 
				for(size_t j=0; j<ci.CandidateWordCount; j++)
				{
					fread(&cw, sizeof(CTBCandidateWordData), 1, stream);

					// decoding values
					cw.StructureSize ^= CTB_MASK_SIZE;
					cw.m_ID ^= CTB_MASK_DATA;

					if(cw.StructureSize != sizeof(CTBCandidateWordData))
					{
						fclose(stream);
						return false;
					}

					CCandidateWord* pCW = new CCandidateWord();
					if(pCW == NULL)
					{
						if(pCI != NULL)
						{
							delete pCI;
							pCI = NULL;
						}
						fclose(stream);
						return false;
					}

					pCW->SetID(cw.m_ID);
					pCW->SetPriority(0);
					pCW->PriorityDataPtr = NULL;


					if(cw.WordLength > 0)
					{
// CTBWordData
						CTBWordData = new UTF32Bytes[cw.WordLength+1];
						if(CTBWordData != NULL)
						{
							memset(CTBWordData, 0, sizeof(UTF32Bytes)*(cw.WordLength+1));
							fread(CTBWordData, sizeof(UTF32Bytes), cw.WordLength+1, stream);

							// decoding values, except last zero value byte.
							for(size_t i=0; i<cw.WordLength; i++)
							{
								CTBWordData[i].byte0 ^= CTB_MASK_BYTE;
								CTBWordData[i].byte1 ^= CTB_MASK_BYTE;
							}


							pCW->Word.SetString(CTBWordData);

							delete [] CTBWordData;
							CTBWordData = NULL;
						}
					}

					pCI->CandidateWords.Add(pCW);
					pCW = NULL;
				}

				m_Indexes.Add(pCI);
				pCI = NULL;
			}
		}

		fclose(stream);
		return true;
	}

	return false;
}
