/*
 * Copyright 2007-2008 Freescale Semiconductor, Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */
#ifndef __ASM_ARCH_MXC_MX37_PINS_H__
#define __ASM_ARCH_MXC_MX37_PINS_H__

/*!
 * @file arch-mxc/mx37_pins.h
 *
 * @brief MX37 I/O Pin List
 *
 * @ingroup GPIO_MX37
 */

#ifndef __ASSEMBLY__

/*!
 * @name IOMUX/PAD Bit field definitions
 */

/*! @{ */

/*!
 * In order to identify pins more effectively, each mux-controlled pin's
 * enumerated value is constructed in the following way:
 *
 * -------------------------------------------------------------------
 * 31-29 | 28 - 24 |  23  | 22 - 20 | 19  - 10| 9 - 0
 * -------------------------------------------------------------------
 * IO_P  |  IO_I  | RSVD_I |  GPIO_I | PAD_I  | MUX_I
 * -------------------------------------------------------------------
 *
 * Bit 0 to 9 contains MUX_I used to identify the register
 * offset (0-based. base is IOMUX_module_base) defined in the Section
 * "sw_pad_ctl & sw_mux_ctl details" of the IC Spec. The
 * similar field definitions are used for the pad control register.
 * For example, the MX37_PIN_ETM_D0 is defined in the enumeration:
 *    ( (0x28 - MUX_I_START) << MUX_I)|( (0x250 - PAD_I_START) << PAD_I)
 * It means the mux control register is at register offset 0x28. The pad control
 * register offset is: 0x250 and also occupy the least significant bits
 * within the register.
 */

/*!
 * Starting bit position within each entry of \b iomux_pins to represent the
 * MUX control register offset
 */
#define MUX_I			0
/*!
 * Starting bit position within each entry of \b iomux_pins to represent the
 * PAD control register offset
 */
#define PAD_I			10
/*!
 * Starting bit position within each entry of \b iomux_pins to represent which
 * mux mode is for GPIO (0-based)
 */
#define GPIO_I			20
/*!
 * Starting bit position which is reserved.
 */
#define RSVD_I			23

#define NON_GPIO_PORT		0x7
#define PIN_TO_MUX_MASK		((1 << (PAD_I - MUX_I)) -1)
#define PIN_TO_PAD_MASK		((1 << (GPIO_I - PAD_I)) - 1)
#define PIN_TO_ALT_GPIO_MASK	((1 << (RSVD_I - GPIO_I)) - 1)

#define NON_MUX_I		PIN_TO_MUX_MASK
#define MUX_I_START		0x0008
#define MUX_I_END		(PAD_I_START - 4)
#define PAD_I_START		0x230
#define PAD_I_END		(INPUT_CTL_START - 4)
#define INPUT_CTL_START		0x508
#define INPUT_CTL_END		0x614

#define _MXC_BUILD_PIN(gp, gi, ga, mi, pi) \
	(((gp) << MUX_IO_P) | ((gi) << MUX_IO_I) | \
	((mi - MUX_I_START) << MUX_I) | \
	((pi - PAD_I_START) << PAD_I) | \
	((ga) << GPIO_I))

#define _MXC_BUILD_GPIO_PIN(gp, gi, ga, mi, pi) \
    _MXC_BUILD_PIN(gp, gi, ga, mi, pi)

#define _MXC_BUILD_NON_GPIO_PIN(mi, pi) \
    _MXC_BUILD_PIN(NON_GPIO_PORT, 0, 0, mi, pi)

#define PIN_TO_IOMUX_MUX(pin)	((pin >> MUX_I) & PIN_TO_MUX_MASK)
#define PIN_TO_IOMUX_PAD(pin)	((pin >> PAD_I) & PIN_TO_PAD_MASK)
#define PIN_TO_ALT_GPIO(pin)	((pin >> GPIO_I) & PIN_TO_ALT_GPIO_MASK)
#define PIN_TO_IOMUX_INDEX(pin)	(PIN_TO_IOMUX_MUX(pin) >> 2)

/*! @} End IOMUX/PAD Bit field definitions */

/*!
 * This enumeration is constructed based on the Section
 * "sw_pad_ctl & sw_mux_ctl details" of the MX37 IC Spec. Each enumerated
 * value is constructed based on the rules described above.
 */
enum iomux_pins {
	MX37_PIN_KEY_ROW0 = _MXC_BUILD_NON_GPIO_PIN(0x8, 0x230),
	MX37_PIN_KEY_ROW1 = _MXC_BUILD_NON_GPIO_PIN(0xC, 0x234),
	MX37_PIN_KEY_ROW2 = _MXC_BUILD_NON_GPIO_PIN(0x10, 0x238),
	MX37_PIN_KEY_ROW3 = _MXC_BUILD_NON_GPIO_PIN(0x14, 0x23C),
	MX37_PIN_KEY_ROW4 = _MXC_BUILD_NON_GPIO_PIN(0x18, 0x240),
	MX37_PIN_KEY_ROW5 = _MXC_BUILD_NON_GPIO_PIN(0x1C, 0x244),
	MX37_PIN_KEY_ROW6 = _MXC_BUILD_NON_GPIO_PIN(0x20, 0x248),
	MX37_PIN_KEY_ROW7 = _MXC_BUILD_NON_GPIO_PIN(0x24, 0x24C),
	MX37_PIN_ETM_D0 = _MXC_BUILD_NON_GPIO_PIN(0x28, 0x250),
	MX37_PIN_ETM_D1 = _MXC_BUILD_NON_GPIO_PIN(0x2C, 0x254),
	MX37_PIN_ETM_D2 = _MXC_BUILD_NON_GPIO_PIN(0x30, 0x258),
	MX37_PIN_ETM_D3 = _MXC_BUILD_NON_GPIO_PIN(0x34, 0x25C),
	MX37_PIN_ETM_D4 = _MXC_BUILD_NON_GPIO_PIN(0x38, 0x260),
	MX37_PIN_ETM_D5 = _MXC_BUILD_NON_GPIO_PIN(0x3C, 0x264),
	MX37_PIN_ETM_D6 = _MXC_BUILD_NON_GPIO_PIN(0x40, 0x268),
	MX37_PIN_ETM_D7 = _MXC_BUILD_NON_GPIO_PIN(0x44, 0x26C),
	MX37_PIN_EIM_EB0 = _MXC_BUILD_GPIO_PIN(0, 15, 3, 0x48, 0x2A8),
	MX37_PIN_EIM_EB1 = _MXC_BUILD_GPIO_PIN(0, 14, 3, 0x4C, 0x2AC),
	MX37_PIN_EIM_OE = _MXC_BUILD_GPIO_PIN(0, 13, 3, 0x50, 0x2B0),
	MX37_PIN_EIM_CS0 = _MXC_BUILD_GPIO_PIN(1, 0, 1, 0x54, 0x2B4),
	MX37_PIN_EIM_CS1 = _MXC_BUILD_GPIO_PIN(1, 1, 1, 0x58, 0x2B8),
	MX37_PIN_EIM_ECB = _MXC_BUILD_GPIO_PIN(0, 12, 3, 0x5C, 0x2BC),
	MX37_PIN_EIM_LBA = _MXC_BUILD_GPIO_PIN(0, 11, 3, 0x60, 0x2C0),
	MX37_PIN_EIM_BCLK = _MXC_BUILD_GPIO_PIN(0, 10, 3, 0x64, 0x2C4),
	MX37_PIN_EIM_RW = _MXC_BUILD_GPIO_PIN(0, 9, 3, 0x68, 0x2C8),
	MX37_PIN_NANDF_WE_B = _MXC_BUILD_GPIO_PIN(1, 2, 4, 0x6C, 0x2CC),
	MX37_PIN_NANDF_RE_B = _MXC_BUILD_GPIO_PIN(1, 3, 4, 0x70, 0x2D0),
	MX37_PIN_NANDF_ALE = _MXC_BUILD_GPIO_PIN(1, 4, 4, 0x74, 0x2D4),
	MX37_PIN_NANDF_CLE = _MXC_BUILD_GPIO_PIN(1, 5, 4, 0x78, 0x2D8),
	MX37_PIN_NANDF_WP_B = _MXC_BUILD_GPIO_PIN(1, 6, 4, 0x7C, 0x2DC),
	MX37_PIN_NANDF_RB = _MXC_BUILD_GPIO_PIN(1, 7, 4, 0x80, 0x2E0),
	MX37_PIN_NANDF_CS0 = _MXC_BUILD_GPIO_PIN(1, 8, 4, 0x84, 0x2E4),
	MX37_PIN_NANDF_CS1 = _MXC_BUILD_GPIO_PIN(1, 9, 4, 0x88, 0x2E8),
	MX37_PIN_NANDF_CS2 = _MXC_BUILD_GPIO_PIN(1, 10, 4, 0x8C, 0x2EC),
	MX37_PIN_NANDF_CS3 = _MXC_BUILD_GPIO_PIN(1, 11, 4, 0x90, 0x2F0),
	MX37_PIN_EIM_D15 = _MXC_BUILD_NON_GPIO_PIN(0x94, 0x2F4),
	MX37_PIN_EIM_D14 = _MXC_BUILD_NON_GPIO_PIN(0x98, 0x2F8),
	MX37_PIN_EIM_D13 = _MXC_BUILD_NON_GPIO_PIN(0x9C, 0x2FC),
	MX37_PIN_EIM_D12 = _MXC_BUILD_NON_GPIO_PIN(0xA0, 0x300),
	MX37_PIN_EIM_D11 = _MXC_BUILD_NON_GPIO_PIN(0xA4, 0x304),
	MX37_PIN_EIM_D10 = _MXC_BUILD_NON_GPIO_PIN(0xA8, 0x308),
	MX37_PIN_EIM_D9 = _MXC_BUILD_NON_GPIO_PIN(0xAC, 0x30C),
	MX37_PIN_EIM_D8 = _MXC_BUILD_GPIO_PIN(0, 8, 3, 0xB0, 0x310),
	MX37_PIN_EIM_D7 = _MXC_BUILD_GPIO_PIN(2, 27, 3, 0xB4, 0x314),
	MX37_PIN_EIM_D6 = _MXC_BUILD_GPIO_PIN(2, 26, 3, 0xB8, 0x318),
	MX37_PIN_EIM_D5 = _MXC_BUILD_GPIO_PIN(2, 25, 3, 0xBC, 0x31C),
	MX37_PIN_EIM_D4 = _MXC_BUILD_GPIO_PIN(2, 24, 3, 0xC0, 0x320),
	MX37_PIN_EIM_D3 = _MXC_BUILD_GPIO_PIN(2, 23, 3, 0xC4, 0x324),
	MX37_PIN_EIM_D2 = _MXC_BUILD_GPIO_PIN(2, 22, 3, 0xC8, 0x328),
	MX37_PIN_EIM_D1 = _MXC_BUILD_GPIO_PIN(2, 21, 3, 0xCC, 0x32C),
	MX37_PIN_EIM_D0 = _MXC_BUILD_GPIO_PIN(2, 20, 3, 0xD0, 0x330),
	MX37_PIN_SD1_CMD = _MXC_BUILD_GPIO_PIN(0, 16, 3, 0xD4, 0x334),
	MX37_PIN_SD1_CLK = _MXC_BUILD_GPIO_PIN(0, 17, 3, 0xD8, 0x338),
	MX37_PIN_SD1_DATA0 = _MXC_BUILD_GPIO_PIN(0, 18, 3, 0xDC, 0x33C),
	MX37_PIN_SD1_DATA1 = _MXC_BUILD_GPIO_PIN(0, 19, 3, 0xE0, 0x340),
	MX37_PIN_SD1_DATA2 = _MXC_BUILD_GPIO_PIN(0, 20, 3, 0xE4, 0x344),
	MX37_PIN_SD1_DATA3 = _MXC_BUILD_GPIO_PIN(0, 21, 3, 0xE8, 0x348),
	MX37_PIN_SD2_CMD = _MXC_BUILD_GPIO_PIN(0, 22, 3, 0xEC, 0x34C),
	MX37_PIN_SD2_CLK = _MXC_BUILD_GPIO_PIN(0, 23, 3, 0xF0, 0x350),
	MX37_PIN_SD2_DATA0 = _MXC_BUILD_GPIO_PIN(0, 24, 3, 0xF4, 0x354),
	MX37_PIN_SD2_DATA1 = _MXC_BUILD_GPIO_PIN(0, 25, 3, 0xF8, 0x358),
	MX37_PIN_SD2_DATA2 = _MXC_BUILD_GPIO_PIN(0, 26, 3, 0xFC, 0x35C),
	MX37_PIN_SD2_DATA3 = _MXC_BUILD_GPIO_PIN(0, 27, 3, 0x100, 0x360),
	MX37_PIN_I2C1_CLK = _MXC_BUILD_GPIO_PIN(1, 12, 4, 0x104, 0x364),
	MX37_PIN_I2C1_DAT = _MXC_BUILD_GPIO_PIN(1, 13, 4, 0x108, 0x368),
	MX37_PIN_AUD3_BB_TXD = _MXC_BUILD_GPIO_PIN(1, 14, 4, 0x10C, 0x36C),
	MX37_PIN_AUD3_BB_RXD = _MXC_BUILD_GPIO_PIN(1, 15, 4, 0x110, 0x370),
	MX37_PIN_AUD3_BB_CK = _MXC_BUILD_GPIO_PIN(1, 16, 4, 0x114, 0x374),
	MX37_PIN_AUD3_BB_FS = _MXC_BUILD_GPIO_PIN(1, 17, 4, 0x118, 0x378),
	MX37_PIN_AUD5_RXFS = _MXC_BUILD_GPIO_PIN(1, 18, 4, 0x11C, 0x37C),
	MX37_PIN_AUD5_RXC = _MXC_BUILD_GPIO_PIN(1, 19, 4, 0x120, 0x380),
	MX37_PIN_AUD5_WB_TXD = _MXC_BUILD_GPIO_PIN(1, 20, 4, 0x124, 0x384),
	MX37_PIN_AUD5_WB_RXD = _MXC_BUILD_GPIO_PIN(1, 21, 4, 0x128, 0x388),
	MX37_PIN_AUD5_WB_CK = _MXC_BUILD_GPIO_PIN(1, 22, 4, 0x12C, 0x38C),
	MX37_PIN_AUD5_WB_FS = _MXC_BUILD_GPIO_PIN(1, 23, 4, 0x130, 0x390),
	MX37_PIN_CSPI1_MOSI = _MXC_BUILD_GPIO_PIN(2, 0, 4, 0x134, 0x394),
	MX37_PIN_CSPI1_MISO = _MXC_BUILD_GPIO_PIN(2, 1, 4, 0x138, 0x398),
	MX37_PIN_CSPI1_SS0 = _MXC_BUILD_GPIO_PIN(2, 2, 4, 0x13C, 0x39C),
	MX37_PIN_CSPI1_SS1 = _MXC_BUILD_GPIO_PIN(2, 3, 4, 0x140, 0x3A0),
	MX37_PIN_CSPI1_SCLK = _MXC_BUILD_GPIO_PIN(2, 4, 4, 0x144, 0x3A4),
	MX37_PIN_CSPI2_MOSI = _MXC_BUILD_GPIO_PIN(2, 5, 4, 0x148, 0x3A8),
	MX37_PIN_CSPI2_MISO = _MXC_BUILD_GPIO_PIN(2, 6, 4, 0x14C, 0x3AC),
	MX37_PIN_CSPI2_SS0 = _MXC_BUILD_GPIO_PIN(2, 7, 4, 0x150, 0x3B0),
	MX37_PIN_CSPI2_SS1 = _MXC_BUILD_GPIO_PIN(2, 8, 4, 0x154, 0x3B4),
	MX37_PIN_CSPI2_SCLK = _MXC_BUILD_GPIO_PIN(2, 9, 4, 0x158, 0x3B8),
	MX37_PIN_UART1_RXD = _MXC_BUILD_GPIO_PIN(1, 24, 4, 0x15C, 0x3BC),
	MX37_PIN_UART1_TXD = _MXC_BUILD_GPIO_PIN(1, 25, 4, 0x160, 0x3C0),
	MX37_PIN_UART1_RTS = _MXC_BUILD_GPIO_PIN(1, 26, 4, 0x164, 0x3C4),
	MX37_PIN_UART1_CTS = _MXC_BUILD_GPIO_PIN(1, 27, 4, 0x168, 0x3C8),
	MX37_PIN_UART1_DTR = _MXC_BUILD_GPIO_PIN(1, 28, 4, 0x16C, 0x3CC),
	MX37_PIN_UART1_DSR = _MXC_BUILD_GPIO_PIN(1, 29, 4, 0x170, 0x3D0),
	MX37_PIN_UART1_RI = _MXC_BUILD_GPIO_PIN(1, 30, 4, 0x174, 0x3D4),
	MX37_PIN_UART1_DCD = _MXC_BUILD_GPIO_PIN(1, 31, 4, 0x178, 0x3D8),
	MX37_PIN_OWIRE_LINE = _MXC_BUILD_GPIO_PIN(0, 31, 4, 0x17C, 0x3DC),
	MX37_PIN_JTAG_DE_B = _MXC_BUILD_NON_GPIO_PIN(0x180, 0x3E0),
	MX37_PIN_DI1_PIN11 = _MXC_BUILD_GPIO_PIN(2, 10, 4, 0x184, 0x3E4),
	MX37_PIN_DI1_PIN12 = _MXC_BUILD_GPIO_PIN(2, 11, 4, 0x188, 0x3E8),
	MX37_PIN_DI1_PIN13 = _MXC_BUILD_GPIO_PIN(2, 12, 4, 0x18C, 0x3EC),
	MX37_PIN_DI1_D0_CS = _MXC_BUILD_GPIO_PIN(2, 13, 4, 0x190, 0x3F0),
	MX37_PIN_DI1_PIN15 = _MXC_BUILD_GPIO_PIN(0, 30, 4, 0x194, 0),
	MX37_PIN_DISP1_DAT0 = _MXC_BUILD_NON_GPIO_PIN(0x198, 0x3F4),
	MX37_PIN_DISP1_DAT1 = _MXC_BUILD_NON_GPIO_PIN(0x19C, 0x3F8),
	MX37_PIN_DISP1_DAT2 = _MXC_BUILD_NON_GPIO_PIN(0x1A0, 0x3FC),
	MX37_PIN_DISP1_DAT3 = _MXC_BUILD_NON_GPIO_PIN(0x1A4, 0x400),
	MX37_PIN_DISP1_DAT4 = _MXC_BUILD_NON_GPIO_PIN(0x1A8, 0x404),
	MX37_PIN_DISP1_DAT5 = _MXC_BUILD_NON_GPIO_PIN(0x1AC, 0x408),
	MX37_PIN_DISP1_DAT6 = _MXC_BUILD_NON_GPIO_PIN(0x1B0, 0x40C),
	MX37_PIN_DISP1_DAT7 = _MXC_BUILD_NON_GPIO_PIN(0x1B4, 0x410),
	MX37_PIN_DISP1_DAT8 = _MXC_BUILD_NON_GPIO_PIN(0x1B8, 0x414),
	MX37_PIN_DISP1_DAT9 = _MXC_BUILD_NON_GPIO_PIN(0x1BC, 0x418),
	MX37_PIN_DISP1_DAT10 = _MXC_BUILD_NON_GPIO_PIN(0x1C0, 0x41C),
	MX37_PIN_DISP1_DAT11 = _MXC_BUILD_NON_GPIO_PIN(0x1C4, 0x420),
	MX37_PIN_DISP1_DAT12 = _MXC_BUILD_NON_GPIO_PIN(0x1C8, 0x424),
	MX37_PIN_DISP1_DAT13 = _MXC_BUILD_NON_GPIO_PIN(0x1CC, 0x428),
	MX37_PIN_DISP1_DAT14 = _MXC_BUILD_NON_GPIO_PIN(0x1D0, 0x42C),
	MX37_PIN_DISP1_DAT15 = _MXC_BUILD_NON_GPIO_PIN(0x1D4, 0x430),
	MX37_PIN_DISP1_DAT16 = _MXC_BUILD_GPIO_PIN(0, 28, 4, 0x1D8, 0x434),
	MX37_PIN_DISP1_DAT17 = _MXC_BUILD_GPIO_PIN(0, 29, 4, 0x1DC, 0x438),
	MX37_PIN_DISP1_DAT18 = _MXC_BUILD_GPIO_PIN(2, 14, 4, 0x1E0, 0x43C),
	MX37_PIN_DISP1_DAT19 = _MXC_BUILD_GPIO_PIN(2, 15, 4, 0x1E4, 0x440),
	MX37_PIN_DISP1_DAT20 = _MXC_BUILD_GPIO_PIN(2, 16, 4, 0x1E8, 0x444),
	MX37_PIN_DISP1_DAT21 = _MXC_BUILD_GPIO_PIN(2, 17, 4, 0x1EC, 0x448),
	MX37_PIN_DISP1_DAT22 = _MXC_BUILD_GPIO_PIN(2, 18, 4, 0x1F0, 0x44C),
	MX37_PIN_DISP1_DAT23 = _MXC_BUILD_GPIO_PIN(2, 18, 4, 0x1F4, 0x450),
	MX37_PIN_PAD_DI1_PIN3 = _MXC_BUILD_GPIO_PIN(2, 29, 4, 0x1F8, 0),
	MX37_PIN_DISP_CLK = _MXC_BUILD_GPIO_PIN(2, 30, 4, 0x1FC, 0),
	MX37_PIN_DI1_PIN2 = _MXC_BUILD_GPIO_PIN(2, 31, 4, 0x200, 0),
	MX37_PIN_BOOT_MODE1 = _MXC_BUILD_NON_GPIO_PIN(0x204, 0x454),
	MX37_PIN_BOOT_MODE0 = _MXC_BUILD_NON_GPIO_PIN(0x208, 0x458),
	MX37_PIN_WDOG_RST = _MXC_BUILD_GPIO_PIN(2, 28, 1, 0x20C, 0x464),
	MX37_PIN_GPIO1_0 = _MXC_BUILD_GPIO_PIN(0, 0, 0, 0x210, 0x468),
	MX37_PIN_GPIO1_1 = _MXC_BUILD_GPIO_PIN(0, 1, 0, 0x214, 0x46C),
	MX37_PIN_GPIO1_2 = _MXC_BUILD_GPIO_PIN(0, 2, 1, 0x218, 0x470),
	MX37_PIN_GPIO1_3 = _MXC_BUILD_GPIO_PIN(0, 3, 0, 0x21C, 0x474),
	MX37_PIN_GPIO1_4 = _MXC_BUILD_GPIO_PIN(0, 4, 0, 0x220, 0x478),
	MX37_PIN_GPIO1_5 = _MXC_BUILD_GPIO_PIN(0, 5, 0, 0x224, 0x47C),
	MX37_PIN_GPIO1_6 = _MXC_BUILD_GPIO_PIN(0, 6, 0, 0x228, 0x480),
	MX37_PIN_GPIO1_7 = _MXC_BUILD_GPIO_PIN(0, 7, 0, 0x22C, 0x484),
	MX37_PIN_GRP_H10 = _MXC_BUILD_NON_GPIO_PIN(0x230, 0x490),
	MX37_PIN_GRP_H9 = _MXC_BUILD_NON_GPIO_PIN(0x230, 0x494),
	MX37_PIN_GRP_H3 = _MXC_BUILD_NON_GPIO_PIN(0x230, 0x4D0),
	MX37_PIN_GRP_H4 = _MXC_BUILD_NON_GPIO_PIN(0x230, 0x4DC),
	MX37_PIN_GRP_H5 = _MXC_BUILD_NON_GPIO_PIN(0x230, 0x4EC),
};

#endif				/*  */
#endif				/*  */
