/*
 * Copyright (C) 2009-2010 Marco Barisione <marco@barisione.org>
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "contact.h"
#include <libringtoned/ringtoned.h>
#include <string.h>
#include "ringtone.h"


#define PCR_CUSTOM_RINGTONE_FIELD "X-RINGTONED-RINGTONE-PATH"

const gchar *
pcr_contact_get_custom_ringtone (OssoABookContact *contact)
{
    GList *values;
    const gchar *ringtone = PCR_RINGTONE_DEFAULT;

    g_return_val_if_fail (contact, PCR_RINGTONE_DEFAULT);
    g_return_val_if_fail (!osso_abook_contact_is_roster_contact (contact),
            PCR_RINGTONE_DEFAULT);

    values = osso_abook_contact_get_values (E_CONTACT (contact),
            PCR_CUSTOM_RINGTONE_FIELD);
    if (values)
    {
        if (values->next == NULL)
            ringtone = values->data;
        else
            g_warning ("Multiple values for field %s for contact %s",
                    PCR_CUSTOM_RINGTONE_FIELD,
                    osso_abook_contact_get_uid (contact));
    }

    return ringtone;
}

typedef struct
{
    OssoABookContact *master;
    OssoABookContact *rc;

    gchar *uid;

    PCRContactUpdatedCallback cb;
    gpointer user_data;
} AttachClosure;

typedef struct
{
    OssoABookContact *master;

    PCRContactUpdatedCallback cb;
    gpointer user_data;
} UpdateClosure;

static void
attach_closure_free (AttachClosure *closure)
{
    if (!closure)
        return;

    g_object_unref (closure->rc);
    g_object_unref (closure->master);
    g_free (closure->uid);
    g_free (closure);
}

static void
update_closure_free (UpdateClosure *closure)
{
    if (!closure)
        return;

    g_object_unref (closure->master);
    g_free (closure);
}

static OssoABookContact *
lookup_master_from_uid (const gchar *master_uid)
{
    OssoABookRoster *aggregator;
    GList *contacts;
    gint n_contacts;
    OssoABookContact *ret = NULL;

    aggregator = osso_abook_aggregator_get_default (NULL);
    contacts = osso_abook_aggregator_lookup
        (OSSO_ABOOK_AGGREGATOR (aggregator), master_uid);
    n_contacts = g_list_length (contacts);

    if (n_contacts == 1)
        ret = contacts->data;
    else if (n_contacts > 1)
        g_critical ("Unexpected multiple contacts for master UID %s",
                master_uid);

    g_list_free (contacts);

    return ret;
}

static void
new_master_committed_cb (EBook       *book,
                         EBookStatus  status,
                         gpointer     user_data)
{
    AttachClosure *closure = user_data;

    /* Note that the contact passed to the callback could be different
     * from the one originally passed */
    if (closure->cb)
        closure->cb (closure->master, status, closure->user_data);

    attach_closure_free (closure);
}

static void
contact_added_and_known_cb (AttachClosure *closure)
{
    /* Update the master to the one that was actually stored */
    g_object_unref (closure->master);
    closure->master = lookup_master_from_uid (closure->uid);
    g_assert (closure->master);
    g_object_ref (closure->master);

    osso_abook_contact_accept (closure->rc, closure->master, NULL);
    osso_abook_contact_async_commit (closure->master, NULL,
            new_master_committed_cb, closure);
}

static void
aggregator_contacts_added_cb (OssoABookRoster   *roster,
                              OssoABookContact **contacts,
                              AttachClosure     *closure)
{
    gint i;

    for (i = 0; contacts[i]; i++)
    {
        if (!g_strcmp0 (osso_abook_contact_get_uid (contacts[i]), closure->uid))
        {
            g_signal_handlers_disconnect_by_func (roster,
                    aggregator_contacts_added_cb, closure);
            contact_added_and_known_cb (closure);
            break;
        }
    }
}

static void
contact_added_cb (EBook       *book,
                  EBookStatus  status,
                  const gchar *uid,
                  gpointer     user_data)
{
    AttachClosure *closure = user_data;

    if (status == E_BOOK_ERROR_OK)
    {
        closure->uid = g_strdup (uid);
        if (lookup_master_from_uid (uid))
            contact_added_and_known_cb (closure);
        else
            g_signal_connect (osso_abook_aggregator_get_default (NULL),
                    "contacts-added", G_CALLBACK (aggregator_contacts_added_cb),
                    closure);
    }
    else
    {
        if (closure->cb)
            closure->cb (closure->rc, status, closure->user_data);
        attach_closure_free (closure);
    }
}

static void
existing_master_committed_cb (EBook       *book,
                              EBookStatus  status,
                              gpointer     user_data)
{
    UpdateClosure *closure = user_data;

    if (closure->cb)
        closure->cb (closure->master, status, closure->user_data);

    update_closure_free (closure);
}

typedef struct
{
    PCRContactUpdatedCallback cb;
    gpointer user_data;
    OssoABookContact *contact;
} SuccessIdleClosure;

static gboolean
success_idle_cb (gpointer user_data)
{
    SuccessIdleClosure *closure = user_data;

    closure->cb (closure->contact, E_BOOK_ERROR_OK, closure->user_data);

    g_object_unref (closure->contact);
    g_free (closure);

    return FALSE;
}

void
pcr_contact_set_custom_ringtone (OssoABookContact          *contact,
                                 const gchar               *new_ringtone,
                                 PCRContactUpdatedCallback  cb,
                                 gpointer                   user_data)
{
    const gchar *current_ringtone;
    EVCardAttribute *attr;

    g_return_if_fail (contact);
    g_return_if_fail (!osso_abook_contact_is_roster_contact (contact));

    DEBUG ("Setting ringtone for %s (%s) to %s",
            osso_abook_contact_get_display_name (contact),
            osso_abook_contact_get_uid (contact),
            new_ringtone);

    current_ringtone = pcr_contact_get_custom_ringtone (contact);
    if (pcr_ringtone_equal (current_ringtone, new_ringtone))
    {
        if (cb)
        {
            SuccessIdleClosure *success_idle_closure;

            success_idle_closure = g_new0 (SuccessIdleClosure, 1);
            success_idle_closure->cb = cb;
            success_idle_closure->user_data = user_data;
            success_idle_closure->contact = g_object_ref (contact);

            g_idle_add (success_idle_cb, success_idle_closure);
        }

        return;
    }

    /* Schedule the creation of the uncompressed ringtone */
    if (new_ringtone && new_ringtone[0])
    {
        ringtoned_decoder_schedule (new_ringtone);
        ringtoned_decoder_decode_pending ();
    }

    /* Save the contact */
    if (!osso_abook_contact_is_temporary (contact))
    {
        UpdateClosure *update_closure;

        if (current_ringtone)
            osso_abook_contact_remove_value (E_CONTACT (contact),
                    PCR_CUSTOM_RINGTONE_FIELD, current_ringtone);

        if (new_ringtone != NULL)
        {
            attr = e_vcard_attribute_new (NULL, PCR_CUSTOM_RINGTONE_FIELD);
            e_vcard_attribute_add_value (attr, new_ringtone);
            e_vcard_add_attribute (E_VCARD (contact), attr);
        }

        DEBUG ("Committing modified existing contact %s",
                osso_abook_contact_get_uid (contact));

        update_closure = g_new0 (UpdateClosure, 1);
        update_closure->master = g_object_ref (contact);
        update_closure->cb = cb;
        update_closure->user_data = user_data;

        osso_abook_contact_async_commit (contact, NULL,
                existing_master_committed_cb, update_closure);
    }
    else
    {
        GList *roster_contacts;
        OssoABookContact *rc;
        OssoABookContact *new_contact;
        const gchar *vcard_field;
        const gchar*bound_name;
        gchar *nickname;
        AttachClosure *attach_closure;

        roster_contacts = osso_abook_contact_get_roster_contacts (contact);
        if (!roster_contacts || roster_contacts->next)
        {
            g_critical ("%d roster contacts attached to a temporary master!",
                    g_list_length (roster_contacts));
            return;
        }
        rc = roster_contacts->data;
        g_list_free (roster_contacts);

        new_contact = osso_abook_contact_new ();

        vcard_field = osso_abook_contact_get_vcard_field (rc);
        bound_name = osso_abook_contact_get_bound_name (rc);

        DEBUG ("Adding %s to a new master contact", bound_name);

        attr = e_vcard_get_attribute (E_VCARD (contact), vcard_field);
        attr = e_vcard_attribute_copy (attr);
        e_vcard_add_attribute (E_VCARD (new_contact), attr);

        /* The nickname or name is needed or the UI will misbehave */
        nickname = g_strdup (e_contact_get_const (E_CONTACT (rc),
                    E_CONTACT_NICKNAME));
        if (!nickname)
        {
            /* There is no nickname in the roster contact, so we have to fake
             * it. To do so we try to be clever and strip the domain part of
             * the bound name; of course this could break if a protocol allow
             * the @ sign in names... */
            gchar *delim;
            nickname = g_strdup (bound_name);
            delim = strchr (nickname, '@');
            if (delim)
                *delim = '\0';
        }
        attr = e_vcard_attribute_new (NULL, EVC_NICKNAME);
        e_vcard_attribute_add_value (attr, nickname);
        e_vcard_add_attribute (E_VCARD (new_contact), attr);
        g_free (nickname);

        attr = e_vcard_attribute_new (NULL, PCR_CUSTOM_RINGTONE_FIELD);
        e_vcard_attribute_add_value (attr, new_ringtone);
        e_vcard_add_attribute (E_VCARD (new_contact), attr);

        attach_closure = g_new0 (AttachClosure, 1);
        attach_closure->master = new_contact; /* Leave ownership */
        attach_closure->rc = g_object_ref (rc);
        attach_closure->cb = cb;
        attach_closure->user_data = user_data;

        osso_abook_contact_async_add (new_contact, NULL, contact_added_cb,
                attach_closure);
    }
}
