/*
 * Copyright (C) 2009-2010 Marco Barisione <marco@barisione.org>
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <gst/gst.h>
#include <libosso.h>
#include <libosso-abook/osso-abook.h>
#include <libringtoned/ringtoned.h>
#include <libringtoned/profile-private.h>
#include <telepathy-glib/channel.h>
#include "dispatcher.h"


static GMainLoop *loop = NULL;
static RingtonedDispatcher *dispatcher = NULL;

static void
accounts_manager_ready_cb (OssoABookAccountManager *manager,
                           const GError            *error_in,
                           gpointer                 user_data)
{
    if (error_in)
    {
        g_printerr ("Error while waiting for the accounts manager to be ready: "
                "%s\n", error_in->message);
        g_main_loop_quit (loop);
        return;
    }

    DEBUG ("Accounts manager ready");

    dispatcher = ringtoned_dispatcher_new ();
}

static void
aggregator_ready_cb (OssoABookWaitable *aggregator,
                     const GError      *error_in,
                     gpointer           user_data)
{
    if (error_in)
    {
        g_printerr ("Error while waiting for the aggregator to be ready: "
                "%s\n", error_in->message);
        g_main_loop_quit (loop);
        return;
    }

    DEBUG ("Aggregator ready");

    osso_abook_account_manager_call_when_ready (NULL,
        accounts_manager_ready_cb, user_data, NULL);
}

static gboolean
quit_idle_cb (gpointer user_data)
{
    DEBUG ("SIGUSR1 received, quitting");
    g_main_loop_quit (loop);
    return FALSE;
}

static void
sigusr1_handler (int signum)
{
    g_idle_add (quit_idle_cb, NULL);
}

int
main (int    argc,
      char **argv)
{
    osso_context_t *osso_cxt;
    OssoABookRoster *aggregator;
    GError *error = NULL;

    osso_cxt = osso_initialize ("ringtoned", "0.1", FALSE, NULL);
    osso_abook_init (&argc, &argv, osso_cxt);
    gst_init(&argc, &argv);

    DEBUG ("Starting");

    aggregator = osso_abook_aggregator_get_default (&error);
    if (!aggregator)
    {
        g_printerr ("Error getting the aggregator: %s\n",
            error ? error->message : "unknown error");
        g_clear_error (&error);
        return 1;
    }

    loop = g_main_loop_new (NULL, FALSE);

    osso_abook_waitable_call_when_ready (
            OSSO_ABOOK_WAITABLE (aggregator),
            aggregator_ready_cb, NULL, NULL);

    ringtoned_profile_init_daemon ();

    signal (SIGUSR1, sigusr1_handler);

    g_main_loop_run (loop);

    if (dispatcher)
        g_object_unref (dispatcher);

    g_main_loop_unref (loop);

    return 0;
}
