/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include "call.h"
#include "call-private.h"
#include <libringtoned/ringtoned.h>
#include <libringtoned/utils.h>
#include <telepathy-glib/util.h>
#include "ringtoned-enums.h"


#define RINGTONED_CALL_GET_PRIVATE(obj) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RINGTONED_TYPE_CALL, \
                                  RingtonedCallPrivate))

#define ID_RINGTONE 101

enum
{
    PROP_0,
    PROP_CHANNEL,
    PROP_STATUS,
    PROP_MUTE,
    PROP_IDENTIFIER,
    PROP_VCARD_FIELD,
    PROP_ACCOUNT,
    PROP_CONTACTS,
    PROP_RINGTONE_PATH,
    PROP_CA_CONTEXT,
};

struct _RingtonedCallPrivate
{
    TpChannel *channel;
    RingtonedCallStatus status;
    gboolean mute;

    gchar *identifier;
    gchar *vcard_field;
    McAccount *account;
    GList *contacts;

    gchar *ringtone_path;
    gboolean playing;
    ca_context *ca_ctx;
};

G_DEFINE_TYPE (RingtonedCall, ringtoned_call, G_TYPE_OBJECT)

RingtonedCall *
ringtoned_call_new (TpChannel   *channel,
                    const gchar *vcard_field,
                    const gchar *identifier,
                    McAccount   *account,
                    ca_context  *ca_ctx)
{
    return g_object_new (RINGTONED_TYPE_CALL,
            "channel", channel,
            "vcard-field", vcard_field,
            "identifier", identifier,
            "account", account,
            "ca-context", ca_ctx,
            NULL);
}

static void play_real (RingtonedCall *self);

static gboolean
play_real_idle_cb (gpointer user_data)
{
    RingtonedCall *self = user_data;

    play_real (self);

    return FALSE;
}

static void
playback_finished_cb (ca_context *c,
                      guint32     id,
                      gint        error_code,
                      gpointer    user_data)
{
    RingtonedCall *self = user_data;

    if (error_code == CA_SUCCESS)
    {
        DEBUG ("Repeating %s", self->priv->ringtone_path);
        g_idle_add (play_real_idle_cb, self);
    }
}

static void
play_real (RingtonedCall *self)
{
    ca_proplist *props;
    const gchar *playable;
    gint r;

    g_return_if_fail (self->priv->ca_ctx);

    ca_proplist_create (&props);
    /* FIXME: Avoid recalculating this when repeating */
    playable = ringtoned_decoder_get_playable_path (self->priv->ringtone_path);
    ca_proplist_sets (props, CA_PROP_MEDIA_FILENAME, playable);
    ca_proplist_sets (props, "ringtoned.role", "call");

    DEBUG ("Actually playing %s", playable);

    self->priv->playing = TRUE;

    r = ca_context_play_full (self->priv->ca_ctx, ID_RINGTONE, props,
            playback_finished_cb, self);
    if (r != CA_SUCCESS)
        DEBUG ("Playing error %d: %s", r, ca_strerror (r));

    ca_proplist_destroy (props);
}

void
ringtoned_call_play_ringtone (RingtonedCall *self)
{
    if (self->priv->status != RINGTONED_CALL_STATUS_RINGING)
    {
        DEBUG ("ringtoned_call_play_ringtone() called, but not in "
                "ringing status");
        return;
    }

    if (self->priv->playing)
    {
        DEBUG ("Already playing");
        return;
    }

    if (IS_EMPTY (self->priv->ringtone_path))
    {
        DEBUG ("No ringtone set, not playing");
        return;
    }

    if (!self->priv->ca_ctx)
    {
        DEBUG ("No canberra context, not playing");
        return;
    }

    if (ringtoned_profile_get_current_volume () == 0)
    {
        DEBUG ("Volume muted, not playing anything");
        return;
    }

    DEBUG ("Playing %s", self->priv->ringtone_path);

    play_real (self);
}

static void
stop_ringtone (RingtonedCall *self)
{
    if (self->priv->playing)
    {
        DEBUG ("Stop playing %s", self->priv->ringtone_path);

        ca_context_cancel (self->priv->ca_ctx, ID_RINGTONE);
        self->priv->playing = FALSE;
    }
}

static gboolean
unref_self_cb (gpointer user_data)
{
    RingtonedCall *self = user_data;

    g_object_unref (self);

    return FALSE;
}

static void
set_status (RingtonedCall       *self,
            RingtonedCallStatus  new_status)
{
    /* We cannot go back to ringing */
    g_return_if_fail (new_status != RINGTONED_CALL_STATUS_RINGING);

    if (self->priv->status == new_status)
        return;

    DEBUG ("Changing status from %s to %s (call: %p)",
            ringtoned_call_status_get_nick (self->priv->status),
            ringtoned_call_status_get_nick (new_status),
            self);

    if (self->priv->status == RINGTONED_CALL_STATUS_RINGING)
    {
        stop_ringtone (self);

        /* We increased the ref count when we started ringing and now
         * we drop it. This way if no plugin is using this object it will
         * be finalized and removed from the list of calls in the dispatcher.
         * We do the unref in the idle so the caller of this function doesn't
         * suddenly find that the object it was using was destroyed. */
        g_idle_add (unref_self_cb, self);
    }

    self->priv->status = new_status;
    g_object_notify (G_OBJECT (self), "status");
}

static gint
find_handle (GArray   *handles,
             TpHandle  handle)
{
    gint i;

    for (i = 0; i < (gint) handles->len; i++)
    {
        if (g_array_index (handles, TpHandle, i) == handle)
            return i;
    }

    return -1;
}

static void
group_members_changed_cb (TpChannel     *channel,
                          gchar         *message,
                          GArray        *added,
                          GArray        *removed,
                          GArray        *local_pending,
                          GArray        *remote_pending,
                          guint          actor,
                          guint          reason,
                          RingtonedCall *self)
{
    TpHandle self_handle;

    g_return_if_fail (self->priv->channel == channel);

    self_handle = tp_channel_group_get_self_handle (self->priv->channel);

    if (find_handle (added, self_handle) >= 0)
    {
        DEBUG ("Call accepted (call: %p, channel: %p)",
                self, self->priv->channel);
        set_status (self, RINGTONED_CALL_STATUS_ACTIVE);
    }
    else if (find_handle (removed, self_handle) >= 0)
    {
        DEBUG ("Call rejected (call: %p, channel: %p)",
                self, self->priv->channel);
        set_status (self, RINGTONED_CALL_STATUS_CLOSED);
    }
}

static void
channel_invalidated_cb (TpProxy       *proxy,
                        guint          domain,
                        gint           code,
                        gchar         *message,
                        RingtonedCall *self)
{
    DEBUG ("Channel invalidated (call: %p, channel: %p)",
            self, self->priv->channel);

    /* Usually the self handle is removed from the channel and then the
     * channel is invalidated, but we also have to handle the connection
     * manager misbehaving or crashing */
    set_status (self, RINGTONED_CALL_STATUS_CLOSED);
}

static void
set_mute (RingtonedCall *self,
          gboolean       new_mute)
{
    if (self->priv->mute == new_mute)
        return;

    if (!new_mute)
    {
        g_warning ("You cannot unmute an already muted call");
        return;
    }

    if (self->priv->status != RINGTONED_CALL_STATUS_RINGING)
    {
        g_warning ("Cannot mute a non-ringing call");
        return;
    }

    DEBUG ("Muting call %p", self);

    stop_ringtone (self);

    self->priv->mute = new_mute;
    g_object_notify (G_OBJECT (self), "mute");
}

static void
ringtoned_call_get_property (GObject    *object,
                             guint       param_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
    RingtonedCall *self = RINGTONED_CALL (object);

    switch (param_id)
    {
        case PROP_CHANNEL:
            g_value_set_object (value, self->priv->channel);
            break;
        case PROP_STATUS:
            g_value_set_enum (value, self->priv->status);
            break;
        case PROP_MUTE:
            g_value_set_boolean (value, self->priv->mute);
            break;
        case PROP_IDENTIFIER:
            g_value_set_string (value, self->priv->identifier);
            break;
        case PROP_VCARD_FIELD:
            g_value_set_string (value, self->priv->vcard_field);
            break;
        case PROP_ACCOUNT:
            g_value_set_object (value, self->priv->account);
            break;
        case PROP_CONTACTS:
            g_value_set_boxed (value, self->priv->contacts);
            break;
        case PROP_RINGTONE_PATH:
            g_value_set_string (value, self->priv->ringtone_path);
            break;
        case PROP_CA_CONTEXT:
            g_value_set_pointer (value, self->priv->ca_ctx);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
set_channel (RingtonedCall *self,
             TpChannel     *channel)
{
    g_return_if_fail (self->priv->channel == NULL);

    self->priv->channel = g_object_ref (channel);

    g_signal_connect (channel, "group-members-changed",
            G_CALLBACK (group_members_changed_cb), self);

    g_signal_connect (channel, "invalidated",
            G_CALLBACK (channel_invalidated_cb), self);
    if (tp_proxy_get_invalidated (channel))
        /* This should not happen, but... */
        channel_invalidated_cb (TP_PROXY (channel), 0, 0, NULL, self);
}

static void
ringtoned_call_set_property (GObject      *object,
                             guint         param_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
    RingtonedCall *self = RINGTONED_CALL (object);

    switch (param_id)
    {
        case PROP_CHANNEL:
            set_channel (self, g_value_get_object (value));
            break;
        case PROP_IDENTIFIER:
            g_return_if_fail (self->priv->identifier == NULL);
            self->priv->identifier = g_value_dup_string (value);
            break;
        case PROP_MUTE:
            set_mute (self, g_value_get_boolean (value));
            break;
        case PROP_VCARD_FIELD:
            g_return_if_fail (self->priv->vcard_field == NULL);
            self->priv->vcard_field = g_value_dup_string (value);
            break;
        case PROP_ACCOUNT:
            g_return_if_fail (self->priv->account == NULL);
            self->priv->account = g_value_dup_object (value);
            break;
        case PROP_RINGTONE_PATH:
            g_free (self->priv->ringtone_path);
            self->priv->ringtone_path = g_value_dup_string (value);
            break;
        case PROP_CA_CONTEXT:
            g_return_if_fail (self->priv->ca_ctx == NULL);
            self->priv->ca_ctx = g_value_get_pointer (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
ringtoned_call_init (RingtonedCall *self)
{
    self->priv = RINGTONED_CALL_GET_PRIVATE (self);

    self->priv->status = RINGTONED_CALL_STATUS_RINGING;
    /* Keep a ref as long as we are ringing */
    g_object_ref (self);
}

static void
ringtoned_call_constructed (GObject *object)
{
    RingtonedCall *self = RINGTONED_CALL (object);
    OssoABookAggregator *aggregator;
    GList *contacts;

    g_assert (self->priv->channel);

    aggregator = OSSO_ABOOK_AGGREGATOR (osso_abook_aggregator_get_default (NULL));

    /* We could use a generic function for searching, but I *think* that
     * this is the only way to get the fastest possible matching */
    if (!tp_strdiff (self->priv->vcard_field,  EVC_TEL))
        contacts = osso_abook_aggregator_find_contacts_for_phone_number (
                aggregator, self->priv->identifier, TRUE);
    else if (!tp_strdiff (self->priv->vcard_field, EVC_X_SIP))
        contacts = osso_abook_aggregator_find_contacts_for_sip_address (
                aggregator, self->priv->identifier);
    else
        contacts = osso_abook_aggregator_find_contacts_for_im_contact (
                aggregator, self->priv->identifier, self->priv->account);

    g_list_foreach (contacts, (GFunc) g_object_ref, NULL);
    self->priv->contacts = contacts;
}

static void
ringtoned_call_finalize (GObject *object)
{
    RingtonedCall *self = RINGTONED_CALL (object);

    stop_ringtone (self);

    if (self->priv->channel) {
        g_signal_handlers_disconnect_by_func (self->priv->channel,
                G_CALLBACK (group_members_changed_cb), self);
        g_signal_handlers_disconnect_by_func (self->priv->channel,
                G_CALLBACK (channel_invalidated_cb), self);
        g_object_unref (self->priv->channel);
    }

    if (self->priv->account)
        g_object_unref (self->priv->account);

    g_list_foreach (self->priv->contacts, (GFunc) g_object_unref, NULL);
    g_list_free (self->priv->contacts);

    g_free (self->priv->identifier);
    g_free (self->priv->ringtone_path);

    G_OBJECT_CLASS (ringtoned_call_parent_class)->finalize (object);
}

static void
ringtoned_call_class_init (RingtonedCallClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    object_class->constructed = ringtoned_call_constructed;
    object_class->finalize = ringtoned_call_finalize;
    object_class->get_property = ringtoned_call_get_property;
    object_class->set_property = ringtoned_call_set_property;

    g_object_class_install_property (
            object_class,
            PROP_CHANNEL,
            g_param_spec_object (
                "channel",
                "Channel",
                "The TpChannel for this call",
                TP_TYPE_CHANNEL,
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_STATUS,
            g_param_spec_enum (
                "status",
                "Status",
                "The status of the call",
                RINGTONED_TYPE_CALL_STATUS,
                RINGTONED_CALL_STATUS_RINGING,
                G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_MUTE,
            g_param_spec_boolean (
                "mute",
                "Mute",
                "Whether the call was muted",
                FALSE,
                G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_IDENTIFIER,
            g_param_spec_string (
                "identifier",
                "Identifier",
                "The identifier of the caller",
                "",
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_VCARD_FIELD,
            g_param_spec_string (
                "vcard-field",
                "VCard field",
                "The VCard field for the identifier",
                "",
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_ACCOUNT,
            g_param_spec_object (
                "account",
                "Account",
                "The McAccount for the call",
                MC_TYPE_ACCOUNT,
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_CONTACTS,
            g_param_spec_boxed (
                "contacts",
                "Contacts",
                "A GList of OssoABookContacts matching the caller identifier",
                RINGTONED_TYPE_CONTACTS_LIST,
                G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_RINGTONE_PATH,
            g_param_spec_string (
                "ringtone-path",
                "Ringtone path",
                "The file path of the ringtone",
                "",
                G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

    g_object_class_install_property (
            object_class,
            PROP_CA_CONTEXT,
            g_param_spec_pointer (
                "ca-context",
                "canberra context",
                "The libcanberra context to use for the ringtone",
                G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
                G_PARAM_STATIC_STRINGS));

    g_type_class_add_private (object_class, sizeof (RingtonedCallPrivate));
}

GType
ringtoned_contacts_list_get_type (void)
{
    static GType our_type = 0;

    if (G_UNLIKELY (our_type == 0))
        our_type = g_boxed_type_register_static ("RingtonedContactsList",
                (GBoxedCopyFunc) ringtoned_contacts_list_copy,
                (GBoxedFreeFunc) ringtoned_contacts_list_free);

    return our_type;
}

void
ringtoned_contacts_list_free (GList *contacts)
{
    g_list_foreach (contacts, (GFunc) g_object_unref, NULL);
    g_list_free (contacts);
}

GList *
ringtoned_contacts_list_copy (GList *contacts)
{
    GList *copy;

    copy = g_list_copy (contacts);
    g_list_foreach (copy, (GFunc) g_object_ref, NULL);

    return copy;
}
