/*
 * Copyright (C) 2010 Nokia Corporation
 *   @author Mathias Hasselmann <mathias.hasselmann@maemo.org>
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <libosso-abook/osso-abook.h>
#include <libringtoned/ringtoned.h>
#include "contact.h"
#include "ringtone-chooser.h"

typedef struct
{
    OssoABookMenuExtension parent_instance;
} PerUserRingtoneExtension;

typedef struct
{
    OssoABookMenuExtensionClass parent_class;
} PerUserRingtoneExtensionClass;

G_GNUC_INTERNAL GType
per_contact_ringtone_extension_get_type (void) G_GNUC_CONST;

OSSO_ABOOK_DEFINE_MENU_PLUGIN (PerUserRingtoneExtension,
        per_contact_ringtone_extension, OSSO_ABOOK_TYPE_MENU_EXTENSION,,);

static void
per_contact_ringtone_extension_init (PerUserRingtoneExtension *self)
{
}

typedef struct
{
    OssoABookMenuExtension *ext;
    GtkWidget *button;
} SetRingtoneClosure;

static void
set_ringtone_closure_free (SetRingtoneClosure *closure)
{
    if (!closure)
        return;

    if (closure->button)
        g_object_remove_weak_pointer (G_OBJECT (closure->button),
                (gpointer *) &closure->button);

    g_free (closure);
}

static void
ringtoned_saved_cb (OssoABookContact *contact,
                    EBookStatus       status,
                    gpointer          user_data)
{
    SetRingtoneClosure *closure = user_data;
    OssoABookContact *old_contact;

    if (status == E_BOOK_ERROR_OK)
        DEBUG ("%s (%s) with ringtone %s saved",
                osso_abook_contact_get_display_name (contact),
                osso_abook_contact_get_uid (contact),
                pcr_contact_get_custom_ringtone (contact));
    else
    {
        g_critical ("Failed to save the custom ringtone for %s (%s), code: %d",
                contact ? osso_abook_contact_get_display_name (contact)
                        : "<unset>",
                contact ? osso_abook_contact_get_uid (contact) : "<unset>",
                status);
        hildon_banner_show_information (NULL, NULL,
                "Failed to save the custom ringtone");
    }

    old_contact = osso_abook_menu_extension_get_contact (closure->ext);
    if (contact != old_contact)
    {
        /* The contact changed, i.e. the previous temporary master contact
         * was saved and became a non-temporary one.
         * The is no simple way to just update the starter, so we just close
         * it. This difference of behaviour could be confusing for the user,
         * but setting a ringtone for a roster-only contact should be rare
         * enough */
        GtkWindow *parent = osso_abook_menu_extension_get_parent (closure->ext);
        gtk_widget_destroy (GTK_WIDGET (parent));
    }
    else if (closure->button)
    {
        const OssoABookMenuEntry *entries;

        /* Update the button text, it's "set custom ringtone" if there is
         * no ringtone currently set, but "change custom rigtone" if a
         * ringtone is already set */
        entries = osso_abook_menu_extension_get_menu_entries (closure->ext);
        g_assert (entries);
        g_assert (entries[0].label);
        gtk_button_set_label (GTK_BUTTON (closure->button), entries[0].label);
    }

    set_ringtone_closure_free (closure);
}

static void
dialog_response_cb (GtkDialog          *dialog,
                    gint                response_id,
                    SetRingtoneClosure *closure)
{
    if (response_id == GTK_RESPONSE_OK)
    {
        OssoABookContact *contact;
        gchar *ringtone;

        g_object_get (dialog, "ringtone", &ringtone, NULL);
        contact = osso_abook_menu_extension_get_contact (closure->ext);

        pcr_contact_set_custom_ringtone (contact, ringtone,
                ringtoned_saved_cb, closure);

        g_free (ringtone);
    }
    else
        set_ringtone_closure_free (closure);

    gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
set_ringtone_cb (GtkWidget              *button,
                 OssoABookMenuExtension *ext)
{
    OssoABookContact *contact;
    SetRingtoneClosure *closure;
    const gchar *current_ringtone;
    GtkWindow *parent;
    GtkWidget *dialog;

    contact = osso_abook_menu_extension_get_contact (ext);
    if (!contact)
    {
        g_critical ("Trying to set a ringtone for a null contact");
        return;
    }

    closure = g_new0 (SetRingtoneClosure, 1);
    closure->ext = ext;
    closure->button = button;
    g_object_add_weak_pointer (G_OBJECT (closure->button),
            (gpointer *) &closure->button);

    current_ringtone = pcr_contact_get_custom_ringtone (contact);
    parent = osso_abook_menu_extension_get_parent (ext);
    dialog = pcr_ringtone_chooser_new (parent, current_ringtone);
    g_signal_connect (dialog, "response", G_CALLBACK (dialog_response_cb),
            closure);
    gtk_widget_show (dialog);
}

static OssoABookMenuEntry set_ringtone_entries[] = {
    { "Set custom ringtone", 0, 0,
      G_CALLBACK (set_ringtone_cb), NULL },
};

static OssoABookMenuEntry change_ringtone_entries[] = {
    { "Change custom ringtone", 0, 0,
      G_CALLBACK (set_ringtone_cb), NULL },
};

static const OssoABookMenuEntry *
get_menu_entries (OssoABookMenuExtension *ext)
{
    OssoABookContact *contact;
    const gchar *ringtone = PCR_RINGTONE_DEFAULT;

    contact = osso_abook_menu_extension_get_contact (ext);
    if (contact)
        ringtone = pcr_contact_get_custom_ringtone (contact);

    if (g_strcmp0 (ringtone, PCR_RINGTONE_DEFAULT) == 0)
        return set_ringtone_entries;
    else
        return change_ringtone_entries;
}

static int
get_n_menu_entries (OssoABookMenuExtension *ext)
{
    g_assert (G_N_ELEMENTS (set_ringtone_entries) ==
              G_N_ELEMENTS (change_ringtone_entries));

    return G_N_ELEMENTS (set_ringtone_entries);
}

static void
per_contact_ringtone_extension_class_init (PerUserRingtoneExtensionClass *cls)
{
    OssoABookMenuExtensionClass *ext_class;

    ext_class = OSSO_ABOOK_MENU_EXTENSION_CLASS (cls);
    ext_class->get_n_menu_entries = get_n_menu_entries;
    ext_class->get_menu_entries = get_menu_entries;
    ext_class->name =  OSSO_ABOOK_MENU_NAME_CONTACT_VIEW;
}

static void
per_contact_ringtone_extension_class_finalize (PerUserRingtoneExtensionClass *cls)
{
}

/* Prototype needed to make gcc happy as this is not a static function */
gchar *g_module_check_init (GModule *module);

G_MODULE_EXPORT gchar *
g_module_check_init (GModule *module)
{
    gint fake_argc = 1;
    gchar *fake_argv[] = {"osso-addressbook"};

    /* The address book unloads this every time the contact starter is
     * closed and reloads it when a new contact is selected. This breaks
     * our GTypes stuff, so we just make the module resident */
    g_module_make_resident (module);

    if (!gst_init_check (&fake_argc, &fake_argv, NULL))
        g_critical ("Couldn't initialize GStreamer, the ringtone preview "
                "won't be available");

    return NULL;
}
