/*************************************************************************}
{ fsman.cpp - filesystem functions using shell                            }
{                                                                         }
{ (c) Alexey Parfenov, 2011                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This library is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This library is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 23 Oct 2012                                              }
{*************************************************************************/

#include "fsman.h"

FSMan::FSMan(QObject *parent) :
    QObject(parent)
{
    doFetchLists = true;
    doUseCache = true;
    doUseNativeMethods = false;
    doUpOnFail = true;
    doUseCallbackFilters = true;
    curDir = "";
    searching = false;
    searchRx.setPatternSyntax(QRegExp::Wildcard);
    searchRx.setCaseSensitivity(Qt::CaseInsensitive);
}

bool caseInsensitiveLessThan(const QString &s1, const QString &s2)
{
    return s1.toLower() < s2.toLower();
}

QString FSMan::which(const QString& program)
{
#ifdef Q_OS_WIN
    return FSMan::getShellCommandOutput("for %i in ("+program+".exe) do @echo.%~$PATH:i").trimmed();
#else
    QStringList args;
    args.append(program);
    return FSMan::getProcessOutput("which", args).trimmed();
#endif
}

void FSMan::_fetchForce(const QString &_dir, QStringList &_dirs, QStringList &_files)
{
    ls(_dir, _dirs, _files);

    QRegExp rx;
    rx.setPatternSyntax(QRegExp::WildcardUnix);
    rx.setCaseSensitivity(Qt::CaseInsensitive);
    QStringList filesFiltered;
    foreach(QString name, _files)
    {
        foreach(QString filter, nameFilters)
        {
            rx.setPattern(filter);
            if(rx.exactMatch(name))
            {
                filesFiltered.append(name);
                break;
            }
        }
    }
    _files = filesFiltered;

    if(doUseCallbackFilters)
    {
        QStringList dirsFiltered;
        filesFiltered.clear();
        bool passed;
        foreach(QString dirName, _dirs)
        {
            passed = true;
            emit onDirFilter(&passed, dirName, _dir);
            if(passed)
                dirsFiltered.append(dirName);
        }
        foreach(QString fileName, _files)
        {
            passed = true;
            emit onFileFilter(&passed, fileName, _dir);
            if(passed)
                filesFiltered.append(fileName);
        }
        _files = filesFiltered;
        _dirs = dirsFiltered;
    }
}

void FSMan::nextSearchStep()
{
    QTimer::singleShot(0, this, SLOT(onSearchStep()));
}

void FSMan::_fetch(const QString& _dir, QStringList& _dirs, QStringList& _files)
{
    if(doFetchLists)
    {
        if(doUseCache)
        {
            if(cache.contains(_dir))
            {
                FSManCacheRecord crec = cache.value(_dir);
                _files = crec.files;
                _dirs = crec.dirs;
            }
            else
            {
                _fetchForce(_dir, _dirs, _files);
                FSManCacheRecord rec;
                rec.files = _files;
                rec.dirs = _dirs;
                cache.insert(_dir, rec);
            }
        }
        else
        {
            _fetchForce(_dir, _dirs, _files);
        }
    }
}

bool FSMan::_cd()
{
    QString lastDir = curDir;
    bool result = true;
    while(!FSMan::isDirAccessible(curDir, doUseNativeMethods))
    {
        result = false;
        curDir = getParentDir(curDir);
        if(lastDir == curDir)
            break;
    }
    _fetch(curDir, dirs, files);
    return result;
}

void FSMan::getCurrentFilesFull(QStringList& list)
{
    list.clear();
    if(curDir.endsWith("/"))
        foreach(QString item, files)
            list.append(curDir+item);
    else
        foreach(QString item, files)
            list.append(curDir+"/"+item);
}

void FSMan::getCurrentDirsFull(QStringList& list)
{
    list.clear();
    if(curDir.endsWith("/"))
        foreach(QString item, dirs)
            list.append(curDir+item);
    else
        foreach(QString item, dirs)
            list.append(curDir+"/"+item);
}

bool FSMan::cd(const QString& dirPath)
{
    curDir = dirPath;
    bool result = true;

    if(doUseCache)
    {
        if(cache.contains(curDir))
        {
            _fetch(curDir, dirs, files);
        }
        else
        {
            result = _cd();
        }
    }
    else
    {
        result = _cd();
    }

    return result;
}

QString FSMan::getParentDir(const QString& dirPath)
{
    bool charMet = false;
    bool isSep;
    QChar c;
    int n = dirPath.size()-1;
    QString result;
    int a;
    for(a=n; a>=0; a--)
    {
        c = dirPath.at(a);
        isSep = (c == '/') || (c == '\\');
        if(isSep)
        {
            if(charMet)
                break;
        }
        else
        {
            charMet = true;
        }
    }
    result = dirPath.left(a);
    if(result.isEmpty())
    {
#ifdef Q_OS_WIN
        return dirPath.at(0)+':';
#else
        return "/";
#endif
    }
    else
    {
        return result;
    }
}

void FSMan::setUsingNativeMethods(bool doUse)
{
#ifndef Q_OS_WIN
    doUseNativeMethods = doUse;
#else
    Q_UNUSED(doUse);
#endif
}

bool FSMan::isDirAccessible(const QString& dirPath, bool useNativeMethod)
{
#ifndef Q_OS_WIN
    if(useNativeMethod)
    {
        QString command = "\
ESCAPEDPATH=`echo -e \""+shellEscape(dirPath)+"\"`/\n\
if [ -d \"${ESCAPEDPATH}\" ]; then\n\
    echo 1\n\
fi\n";
        QString result = FSMan::getShellCommandOutput(command);
        return (result.trimmed() == "1");
    }
    else
    {
#endif
        QDir dir;
        bool result = dir.cd(dirPath);
        return result;
#ifndef Q_OS_WIN
    }
#endif
}

QString FSMan::getShellCommandOutput(const QString& cmd)
{
    QStringList args;
    QString shell;
#ifdef Q_OS_WIN
    shell = "cmd";
    args.append("/c");
#else
    shell = "sh";
    args.append("-c");
#endif
    args.append(cmd);
    return FSMan::getProcessOutput(shell, args);
}

QString FSMan::shellEscape(const QString& line)
{
#ifdef Q_OS_WIN
    return line;
#endif
    QByteArray bytes = line.toUtf8();
    QString result;
    int n = bytes.size();
    int realCode;
    for(int a=0; a<n; a++)
    {
        realCode = bytes.at(a);
        if(realCode<0)
            realCode = realCode + 256;
        result += "\\x"+QString::number(realCode, 16);
    }
    return result;
}

bool FSMan::_removeFile(const QString &filename, bool useNativeMethod, FSManCache* _cache)
{
    bool result;
#ifndef Q_OS_WIN
    if(useNativeMethod)
    {
        QString command = "rm -f `echo -e \""+shellEscape(filename)+"\"`";
        QString output = getShellCommandOutput(command).trimmed();
        result = output.isEmpty();
    }
    else
#endif
    {
        result = QFile::remove(filename);
    }

    if(result && _cache)
        _cache->remove(getParentDir(filename));

    return result;
}

bool FSMan::_removeDir(const QString &dirname, bool useNativeMethod, FSManCache* _cache)
{
    QStringList dirsList;
    QStringList filesList;
    ls(dirname, dirsList, filesList, useNativeMethod);
    QString rootDir = dirname;
    if(rootDir.endsWith("/") && (rootDir.size() > 1))
        rootDir.chop(1);
    QString prependDir = rootDir+"/";

    if(_cache)
        _cache->remove(rootDir);

    foreach(QString dirsListItem, dirsList)
    {
        dirsListItem.prepend(prependDir);
        if(!_removeDir(dirsListItem, useNativeMethod, _cache))
            return false;
    }

    foreach(QString filesListItem, filesList)
    {
        filesListItem.prepend(prependDir);
        if(!_removeFile(filesListItem, useNativeMethod))
            return false;
    }

    bool result;
#ifndef Q_OS_WIN
    if(useNativeMethod)
    {
        QString command = "rm -rf `echo -e \""+shellEscape(dirname)+"\"`";
        QString output = getShellCommandOutput(command).trimmed();
        result = output.isEmpty();
    }
    else
#endif
    {
        QDir _dir;
        result = _dir.rmdir(dirname);
    }

    if(result && _cache)
        _cache->remove(getParentDir(rootDir));

    return result;
}

void FSMan::startSearch(const QString &expression, const QString startDir)
{
    stopSearch();
    if(expression.isEmpty())
    {
        emit onSearchFinished();
        return;
    }

    QString _dir = startDir;

    if(_dir.isEmpty())
        _dir = curDir;
    if(_dir.isEmpty())
    {
#ifdef Q_OS_WIN
        _dir = "C:/";
#else
        _dir = "/";
#endif
    }
    else
    {
        _dir.replace("\\", "/");
        if(_dir.endsWith("/") && (_dir.size() > 1))
            _dir.chop(1);
    }

    searchDirs.clear();
    searchDirs.append(_dir);
    searching = true;
    searchRx.setPattern(expression);

    nextSearchStep();
}

void FSMan::ls(const QString& path, QStringList& dirsList, QStringList& filesList, bool useNativeMethod)
{
    dirsList.clear();
    filesList.clear();
#ifndef Q_OS_WIN
    if(useNativeMethod)
    {
        QString command = "\
ESCAPEDPATH=`echo -e \""+shellEscape(path)+"\"`/\n\
ls -a1 \"${ESCAPEDPATH}\" 2>/dev/null | while read line\n\
do\n\
if [ -d \"${ESCAPEDPATH}$line\" ]; then\n\
    echo d $line\n\
else\n\
    echo f $line\n\
fi\n\
done\n";
        QString output = FSMan::getShellCommandOutput(command);
        QStringList lines = output.split('\n', QString::SkipEmptyParts);
        QString trLine;
        foreach(QString line, lines)
        {
            trLine = line.trimmed();
            if(trLine.startsWith("d "))
            {
                trLine = trLine.right(trLine.size()-2);
                if((trLine != ".") && (trLine != ".."))
                    dirsList.append(trLine);
            }
            else
            {
                if(trLine.startsWith("f "))
                    filesList.append(trLine.right(trLine.size()-2));
            }
        }
    }
    else
    {
#endif
        QDir dir;
        dir.cd(path);
        QDir::Filters filters = QDir::AllDirs | QDir::Files | QDir::NoDotAndDotDot | QDir::Hidden | QDir::System;
        QFileInfoList list = dir.entryInfoList(filters);
        foreach(QFileInfo entry, list)
        {
            if(entry.isDir())
                dirsList.append(entry.fileName());
            else
                filesList.append(entry.fileName());
        }
#ifndef Q_OS_WIN
    }
#endif
    qSort(dirsList.begin(), dirsList.end(), caseInsensitiveLessThan);
    qSort(filesList.begin(), filesList.end(), caseInsensitiveLessThan);
}

QString FSMan::getHomeDir()
{
    QString homeDir;
#ifdef QT_GUI_LIB
    homeDir = QDesktopServices::storageLocation(QDesktopServices::HomeLocation);
#else
    #ifdef Q_OS_WIN
        homeDir = getShellCommandOutput("echo %HOMEDRIVE%%HOMEPATH%").trimmed();
        if(homeDir.isEmpty())
            homeDir = "C:/";
    #else
        homeDir = getShellCommandOutput("echo ~").trimmed();
        if(homeDir.isEmpty())
            homeDir = "/";
    #endif
#endif
   return homeDir;
}

bool FSMan::getRawProcessOutput(QByteArray& dest, const QString& cmd, const QStringList& args)
{
    QProcess proc;
    proc.start(cmd, args, QIODevice::ReadOnly);
    if(!proc.waitForFinished())
        return false;
    dest.append(proc.readAllStandardOutput());
    proc.close();
    return true;
}

QString FSMan::getProcessOutput(const QString& cmd, const QStringList& args)
{
    QByteArray resultRaw;
    if(getRawProcessOutput(resultRaw, cmd, args))
        return QString::fromUtf8(resultRaw.constData());
    else
        return "";
}

QString FSMan::getProcessOutput(const QString& cmd)
{
    return getProcessOutput(cmd, QStringList());
}

void FSMan::onSearchStep()
{
    if(!searching)
        return;

    QString _dir = searchDirs.takeLast();

    QStringList _dirs;
    QStringList _files;

    _fetch(_dir, _dirs, _files);

    QString _appendDir = _dir;
    if(_appendDir.endsWith("/"))
        _appendDir.chop(1);

    foreach(QString item, _dirs)
    {
        if(searchRx.indexIn(item) >= 0)
            emit onSearchResult(true, item, _dir);
        searchDirs.append(_appendDir + "/" + item);
    }

    foreach(QString item, _files)
    {
        if(searchRx.indexIn(item) >= 0)
            emit onSearchResult(false, item, _dir);
    }

    if(searchDirs.isEmpty())
    {
        stopSearch();
        emit onSearchFinished();
    }
    else
    {
        nextSearchStep();
    }
}
