/*************************************************************************}
{ equalizerwindow.cpp - equalizer settings and presets window             }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 06 Jun 2012                                              }
{*************************************************************************/

#include "equalizerwindow.h"
#include "mainwindow.h"

EqualizerWindow::EqualizerWindow(QWidget *parent) :
    CommonDialog(parent)
{
    presetOff = QString("<")+tr("off", "presetLabel")+QString(">");
    isChanged = false;
    initGui();
}

EqualizerWindow::~EqualizerWindow()
{
    deinitGui();
}

QString EqualizerWindow::showEqualizerWindow(const QString &presetName)
{
    EqualizerWindow* win = new EqualizerWindow(MAINWINDOW);
    win->selectPreset(presetName);
    win->showDialog();
    win->checkForSave();
    QString result = win->selectedPreset;
    delete win;
    return result;
}

void EqualizerWindow::initGui()
{
    CommonDialog::initGui();
    setWindowTitle(tr("Equalizer"));

    btnNew = new Button(this);
    btnRename = new Button(this);
    btnSave = new Button(this);
    btnDelete = new Button(this);

    btnNew->setIcon(SKIN.icoItemNew);
    btnRename->setIcon(SKIN.icoItemRename);
    btnSave->setIcon(SKIN.icoItemSave);
    btnDelete->setIcon(SKIN.icoItemDelete);

    connect(btnNew, SIGNAL(clicked()), SLOT(onNewClick()));
    connect(btnRename, SIGNAL(clicked()), SLOT(onRenameClick()));
    connect(btnSave, SIGNAL(clicked()), SLOT(onSaveClick()));
    connect(btnDelete, SIGNAL(clicked()), SLOT(onDeleteClick()));

    bars = (QWidget**)(malloc(sizeof(QWidget*)*eq()->getBandsCount()));
    QWidget* bar;
    Slider* slider;
    Label* labelFreq;
    Label* labelLevel;
    QString s;
    int f;

    for(int a=0; a<eq()->getBandsCount(); a++)
    {
        bar = new QWidget(this);

        slider = new Slider(bar);
        slider->setMinimum(eq()->getBandLowerBound(a));
        slider->setMaximum(eq()->getBandUpperBound(a));

        f = eq()->getBandFrequency(a);
        if(f<1000)
        {
            s.setNum(f);
            s.append(tr("Hz"));
        }
        else
        {
            f = qRound(f*2/1000)*5;
            s.setNum(f);
            if(s.endsWith('0'))
                s.chop(1);
            else
                s.insert(s.size()-1, QLocale::system().decimalPoint());
            s.append(tr("kHz"));
        }
        labelFreq = new Label(bar);
        labelFreq->setText(s);
        labelFreq->setAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
        labelFreq->setStyleSheet("font-size:15px;");

        labelLevel = new Label(bar);
        labelLevel->setAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
        labelLevel->setStyleSheet("font-size:15px;");

        bars[a] = bar;
    }

    selector = new QPushButton(this);
    connect(selector, SIGNAL(clicked()), SLOT(onSelectorClick()));

    updateLevels();
    updateSelectorLabel();
}

void EqualizerWindow::deinitGui()
{
    for(int a=0; a<eq()->getBandsCount(); a++)
    {
        foreach(QObject* child, bars[a]->children())
            delete child;
        delete bars[a];
    }
    delete bars;

    delete btnNew;
    delete btnRename;
    delete btnSave;
    delete btnDelete;

    delete selector;

    CommonDialog::deinitGui();
}

void EqualizerWindow::reposition(const QSize &sz)
{
    CommonDialog::reposition(sz);

    QWidget* bar;
    Slider* slider;
    Label* labelFreq = NULL;
    Label* labelLevel;

    int w = sz.width();
    int barWidth = w / eq()->getBandsCount();
    int h = sz.height();
    int sp = CONTROLBUTTON_SIZE + CONTROLBUTTON_PADDING;

    bool showFreq = true;

    int labelFreqHeight;
    if(showFreq)
        labelFreqHeight = ((Label*)(bars[0]->children().at(1)))->fontMetrics().height();
    else
        labelFreqHeight = 0;

    for(int a=0; a<eq()->getBandsCount(); a++)
    {
        bar = bars[a];
        slider = (Slider*)(bar->children().at(0));
        labelFreq = (Label*)(bar->children().at(1));
        labelLevel = (Label*)(bar->children().at(2));

        bar->move(barWidth*a, sp);
        bar->resize(barWidth, h - sp);

        labelFreq->move(0, 0);
        labelFreq->resize(barWidth, labelFreqHeight);

        labelLevel->resize(barWidth, labelLevel->fontMetrics().height());
        labelLevel->move(0, bar->height()-labelFreqHeight);

        slider->move(0, labelFreqHeight);
        slider->resize(barWidth-1, bar->height()-labelFreqHeight-labelLevel->height());
        connect(slider, SIGNAL(valueChanged(int)), SLOT(onLevelChange(int)));
    }

    int x = w;
    x = x - sp; btnDelete->move(x, -CONTROLBUTTON_PADDING);
    x = x - sp; btnSave->move(x, -CONTROLBUTTON_PADDING);
    x = x - sp; btnRename->move(x, -CONTROLBUTTON_PADDING);
    x = x - sp; btnNew->move(x, -CONTROLBUTTON_PADDING);

    selector->move(0, 0);
    selector->resize(x, sp);

    updateSelectorLabel();
}

void EqualizerWindow::updateLevels()
{
    for(int a=0; a<eq()->getBandsCount(); a++)
        updateLevel(a);
}

void EqualizerWindow::updateLevel(int bandIndex)
{
    Label* label = (Label*)(bars[bandIndex]->children().at(2));
    Slider* slider = (Slider*)(bars[bandIndex]->children().at(0));
    int level = qRound(eq()->getBandLevel(bandIndex));
    slider->setValue(level);
    updateLevelLabel(label, level);
}

void EqualizerWindow::updateLevelLabel(Label *label, int level)
{
    QString s;
    s.setNum(level);
    if(level >= 0)
        s.prepend("+");
    s.append(tr("dB"));
    label->setText(s);
}

int EqualizerWindow::getBarIndex(QWidget *bar)
{
    for(int a=0; a<eq()->getBandsCount(); a++)
        if(bars[a] == bar)
            return a;
    return -1;
}

void EqualizerWindow::updateSelectorLabel()
{
    QString presetLabel;
    if(selectedPreset.isEmpty())
        presetLabel = presetOff;
    else
        presetLabel = selectedPreset;

    QString txt(tr("Preset")+": "+presetLabel);
    if(selector->fontMetrics().width(txt) > selector->width())
        txt = presetLabel;

    selector->setText(txt);
}

void EqualizerWindow::selectPreset(const QString &presetName)
{
    if(presetName.isEmpty())
    {
        eq()->setDefaultLevels();
        PLAYER->setEqualizerEnabled(false);
        setControlsEnabled(false);
    }
    else
    {
        eq()->loadFromPreset(presetName);
        PLAYER->setEqualizerEnabled(true);
        setControlsEnabled(true);
    }

    selectedPreset = presetName;
    updateLevels();
    updateSelectorLabel();
    isChanged = false;
}

void EqualizerWindow::setControlsEnabled(bool isEnabled)
{
    for(int a=0; a<eq()->getBandsCount(); a++)
        ((Slider*)bars[a]->children()[0])->setEnabled(isEnabled);
}

void EqualizerWindow::checkForSave()
{
    if(!isChanged)
        return;
    if(selectedPreset.isEmpty())
        return;
    QMessageBox::StandardButton result = QMessageBox::question(
                this,
                tr("Save changes"),
                tr("The current preset was modified. Do you want to apply changes?"),
                QMessageBox::Yes | QMessageBox::No,
                QMessageBox::Yes);
    if(result == QMessageBox::Yes)
        onSaveClick();
}

Equalizer *EqualizerWindow::eq() const
{
    return PLAYER->getEqualizer();
}

void EqualizerWindow::onLevelChange(int level)
{
    Slider* slider = (Slider*)(sender());
    QWidget* bar = (QWidget*)(slider->parent());
    Label* label = (Label*)(bar->children().at(2));
    eq()->setBandLevel(getBarIndex(bar), level);
    updateLevelLabel(label, level);
    isChanged = true;
}

void EqualizerWindow::onNewClick()
{
    checkForSave();
    bool ok;
    QString presetName;
    forever
    {
        presetName = QInputDialog::getText(
                                        this,
                                        tr("New preset"),
                                        tr("Enter a name for a new preset"),
                                        QLineEdit::Normal,
                                        "",
                                        &ok).trimmed();
        if(!ok)
            return;
        if(presetName.isEmpty())
        {
            QMessageBox::critical(
                        this,
                        tr("New preset"),
                        tr("Preset name cannot be empty!")
                        );
            continue;
        }
        if(eq()->hasValidPreset(presetName))
        {
            QMessageBox::critical(
                        this,
                        tr("New preset"),
                        tr("Preset with such name already exists!")
                        );
            continue;
        }

        eq()->setDefaultLevels();
        eq()->saveToPreset(presetName);
        selectedPreset = presetName;
        updateLevels();
        updateSelectorLabel();
        PLAYER->setEqualizerEnabled(true);
        setControlsEnabled(true);
        MAINWINDOW->showNotification(tr("New preset has been created and selected"));
        isChanged = false;
        break;
    }
}

void EqualizerWindow::onRenameClick()
{
    if(selectedPreset.isEmpty())
    {
        QMessageBox::critical(
                    this,
                    tr("Rename the preset"),
                    tr("This is a system preset, therefore it cannot be renamed.")
                    );
        return;
    }

    bool ok;
    QString presetName;
    forever
    {
        presetName = QInputDialog::getText(
                                        this,
                                        tr("Rename the preset"),
                                        tr("Enter a new name for the current preset"),
                                        QLineEdit::Normal,
                                        selectedPreset,
                                        &ok).trimmed();
        if(!ok)
            return;
        if(presetName == selectedPreset)
            return;
        if(presetName.isEmpty())
        {
            QMessageBox::critical(
                        this,
                        tr("Rename the preset"),
                        tr("Preset name cannot be empty!")
                        );
            continue;
        }
        if(eq()->hasValidPreset(presetName))
        {
            QMessageBox::critical(
                        this,
                        tr("Rename the preset"),
                        tr("There is another preset with such name!")
                        );
            continue;
        }

        eq()->renamePreset(selectedPreset, presetName);
        selectedPreset = presetName;
        updateSelectorLabel();
        MAINWINDOW->showNotification(tr("The current preset has been renamed"));
        break;
    }
}

void EqualizerWindow::onSaveClick()
{
    if(selectedPreset.isEmpty())
    {
        QMessageBox::critical(
                    this,
                    tr("Save the preset"),
                    tr("This is a system preset, therefore it cannot be saved.")
                    );
        return;
    }
    eq()->saveToPreset(selectedPreset);
    MAINWINDOW->showNotification(tr("The current preset has been updated"));
    isChanged = false;
}

void EqualizerWindow::onDeleteClick()
{
    if(selectedPreset.isEmpty())
    {
        QMessageBox::critical(
                    this,
                    tr("Delete the preset"),
                    tr("This is a system preset, therefore it cannot be deleted.")
                    );
        return;
    }
    if(QMessageBox::Yes == QMessageBox::question(
                this,
                tr("Delete the preset"),
                tr("Are you sure you want to delete the current preset?")+"\n\""+selectedPreset+"\"",
                QMessageBox::Yes | QMessageBox::No, QMessageBox::No))
    {
        eq()->deletePreset(selectedPreset);
        eq()->setDefaultLevels();
        selectedPreset = "";
        updateLevels();
        updateSelectorLabel();
        PLAYER->setEqualizerEnabled(false);
        setControlsEnabled(false);
        MAINWINDOW->showNotification(tr("The current preset has been deleted"));
        isChanged = false;
    }
}

void EqualizerWindow::onSelectorClick()
{
    checkForSave();
    QStringList presets = eq()->getValidPresetsList();
    int index = presets.indexOf(selectedPreset);
    index++;
    presets.prepend(presetOff);
    index = SelectWindow::showSelectDialog(this, presets, index, tr("Select a preset"));
    if(index > 0)
        selectPreset(presets.at(index));
    else
        selectPreset("");
}
