/*************************************************************************}
{ phononplayer.cpp - Phonon-based audio player                            }
{                                                                         }
{ This file is a part of the project                                      }
{   Rhapsodie - Music player for N900                                     }
{                                                                         }
{ (c) Alexey Parfenov, 2012                                               }
{                                                                         }
{ e-mail: zxed@alkatrazstudio.net                                         }
{                                                                         }
{ This program is free software; you can redistribute it and/or           }
{ modify it under the terms of the GNU General Public License             }
{ as published by the Free Software Foundation; either version 3 of       }
{ the License, or (at your option) any later version.                     }
{                                                                         }
{ This program is distributed in the hope that it will be useful,         }
{ but WITHOUT ANY WARRANTY; without even the implied warranty of          }
{ MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        }
{ General Public License for more details.                                }
{                                                                         }
{ You may read GNU General Public License at:                             }
{   http://www.gnu.org/copyleft/gpl.html                                  }
{                                                                         }
{ last modified: 29 May 2012                                              }
{*************************************************************************/

#include "phononplayer.h"
#include "mainwindow.h"

PhononPlayer::PhononPlayer(QObject *parent) :
    QObject(parent)
{
    updateKnownMime();
    updateSupportedMime();
    findUnzip();

    equalizer = new Equalizer(this);
    equalizer->setPresetsIni(MAINWINDOW->getIniSettings());
    equalizerEnabled = false;

    lastPos = -1;

    sound = new PhononSound(this); // just to prevent errors when getting properties
    cacheSound = NULL;

    tmpFilenameBase = QDesktopServices::storageLocation(QDesktopServices::TempLocation)+"/___"+APP_NAME+"_temp.";

#ifdef MAEMO5
    QDBusConnection::systemBus().connect(
                    "org.freedesktop.Hal",
                    "/org/freedesktop/Hal/devices/platform_headphone",
                    "org.freedesktop.Hal.Device",
                    "Condition",
                    this,
                    SLOT(onHeadsetCondition())
                    );

    QDBusConnection::systemBus().connect(
                    MCE_SERVICE,
                    MCE_SIGNAL_PATH,
                    MCE_SIGNAL_IF,
                    MCE_CALL_STATE_SIG,
                    this,
                    SLOT(onCallStateChange(QString))
                    );

    QDBusMessage message = QDBusMessage::createMethodCall(
                MCE_SERVICE,
                MCE_REQUEST_PATH,
                MCE_REQUEST_IF,
                MCE_CALL_STATE_GET);
    message = QDBusConnection::systemBus().call(message);
    QString callState = message.arguments().at(0).toString();
    calling = (callState != MCE_CALL_STATE_NONE);
#else
    calling = false;
#endif
}

PhononPlayer::~PhononPlayer()
{
    stop();
    delete equalizer;
    delete sound;
    delete cacheSound;
}

bool PhononPlayer::play(const QString &filename, bool doCloseCurrent)
{
    killLater(sound, doCloseCurrent);
    if(!loadFromCache(filename))
    {
        sound = new PhononSound(this, equalizerEnabled);
        sound->open(filename);
    }
    else
    {
        if(sound->getState() == pp_state_Error)
        {
            QTimer::singleShot(0, this, SIGNAL(onError()));
            return true;
        }
    }

    connect(sound, SIGNAL(onPlayEnd()), this, SIGNAL(onPlayEnd()));
    connect(sound, SIGNAL(onError()), this, SIGNAL(onError()));
    connect(sound, SIGNAL(onStateChanged()), this, SIGNAL(onStateChanged()));
    connect(sound, SIGNAL(onTitleChange()), this, SIGNAL(onTitleChange()));
    sound->play();
    if(!sound->getTitle().isEmpty())
        emit onTitleChange();

    return true;
}

void PhononPlayer::cache(const QString &filename)
{
    // there's no way to make a decent pre-caching with equalizer enabled
    // due to glitches in playback when two equalizers are active at one time
    if(equalizerEnabled)
        return;
    if(cacheExists(filename))
        return;
    killLater(cacheSound);
    cacheSound = new PhononSound(this);
    cacheSound->open(filename);
}

void PhononPlayer::clearCache(const QString &filename)
{
    if(cacheExists(filename))
        clearCache();
}

void PhononPlayer::clearCache()
{
    killLater(cacheSound);
    cacheSound = NULL;
}

void PhononPlayer::pause()
{
    sound->pause();
}

void PhononPlayer::unpause()
{
    sound->play();
}

void PhononPlayer::stop()
{
    sound->close();
}

bool PhononPlayer::isHeadsetConnected()
{
#ifdef MAEMO5
    QFile statusFile("/sys/devices/platform/gpio-switch/headphone/state");
    statusFile.open(QIODevice::ReadOnly);
    QString statusText = QString::fromAscii(statusFile.readAll().constData()).trimmed().toLower();
    statusFile.close();
    return statusText == "connected";
#else
    return false; // stub
#endif
}

bool PhononPlayer::setEqualizerEnabled(bool doEnable)
{
    if(!getEqualizer()->getEffect())
        return false;
    sound->setEqualizerEnabled(doEnable);
    killLater(cacheSound, true);
    cacheSound = NULL;
    equalizerEnabled = doEnable;
    return true;
}

void PhononPlayer::updateSupportedMime()
{
    supportedMime = Phonon::BackendCapabilities::availableMimeTypes();
    supportedExtensions.clear();

    int n = supportedMime.size();
    for(int a=0; a<n ;a++)
        supportedMime[a] = supportedMime[a].trimmed().toLower();

    QHashIterator<QString, QStringList> i(mimeExtensions);
    while(i.hasNext())
    {
         i.next();
         const QString& value = i.key();
         foreach(QString mimeType, supportedMime)
         {
             if(mimeType == value)
             {
                 supportedExtensions.append(i.value());
                 break;
             }
         }
    }

    supportedExtensions.removeDuplicates();
    supportedExtensions.sort(); // alphabetically
}

void PhononPlayer::updateKnownMime()
{
    QFileEx mimes;
    QString filename = APP_DIR + "data/mimes";

    mimes.setFileName(filename);
    if(!mimes.open(QFile::ReadOnly))
        return;
    QStringList lines = mimes.readLinesUTF8();
    mimes.close();

    mimeExtensions.clear();
    knownExtensions.clear();
    QString mimeType;
    QString ext;
    QStringList list;
    int a, n;
    foreach(QString line, lines)
    {
        line = line.trimmed();
        if(line.isEmpty()) // empty line or a line with whitespace only
            continue;
        if(line.startsWith("#")) // comment line
            continue;
        line = line.toLower();
        list = line.split(",", QString::SkipEmptyParts);
        if(list.size() < 2) // no extensions specified or a string with commas only
            continue;
        mimeType = list.takeFirst().trimmed();
        mimeType.replace("/x-", "/"); // convert eXperimental MIMEs to normal ones
        n = list.size();
        for(a=0; a<n; a++)
        {
            ext = list[a].trimmed();
            knownExtensions.append(ext);
            list[a] = ext;
        }
        mimeExtensions[mimeType] = list;
        mimeType.replace("/", "/x-"); // add eXperimental version of a current MIME
        mimeExtensions[mimeType] = list;
    }

    knownExtensions.removeDuplicates();
    knownExtensions.sort(); // alphabetically
}

void PhononPlayer::findUnzip()
{
    unzipPath = FSMan::which("unzip");
}

void PhononPlayer::onHeadsetCondition()
{
    if(!isHeadsetConnected())
        emit onHeadsetStatusChange(false);
    else
        QTimer::singleShot(3000, this, SLOT(onCheckHeadset())); // safe interval
}

void PhononPlayer::onCheckHeadset()
{
    if(isHeadsetConnected())
        emit onHeadsetStatusChange(true);
}

void PhononPlayer::onCallStateChange(QString callState)
{
#ifdef MAEMO5
    bool newCalling = (callState != MCE_CALL_STATE_NONE);
    if(newCalling == calling)
        return;
    calling = newCalling;
    if(calling)
        emit onCallStart();
    else
        emit onCallEnd();
#else
    Q_UNUSED(callState);
#endif
}

void PhononPlayer::onKillObsoleteObject()
{
    if(!obsoleteObjects.isEmpty())
        obsoleteObjects.takeFirst()->deleteLater();
}

bool PhononPlayer::unzip(const QString &filename, QString& tmpFilename)
{
    if(unzipPath.isEmpty())
        return false;

    QFile f(filename);
    if(!f.open(QIODevice::ReadOnly))
        return false;

    if((f.size() > 1024*1024*10) || (f.size() < 2))
    {
        f.close();
        return false;
    }

    char magic[2];
    magic[0] = magic[1] = 0;
    f.seek(0);
    f.peek(&magic[0], 2);
    f.close();
    if((magic[0]!='P')||(magic[1]!='K')) // zip file header
        return false;

    QByteArray mem;
    FSMan::getRawProcessOutput(mem, unzipPath, (QStringList() << "-p" << filename));

    int n = mem.size();
    if(!n)
        return false; // no output was returned

    // we must create an actual file to circumvent most
    // phonon problems regarding "magick" file identification
    QFileInfo info(filename);
    tmpFilename = tmpFilenameBase + randomHash() + "." + info.suffix();
    QFile tmp(tmpFilename);
    if(!tmp.open(QIODevice::WriteOnly))
        return false;
    int w = tmp.write(mem);
    if(w != n)
    {
        tmp.close();
        return false;
    }
    tmp.close();

    return true;
}

void PhononPlayer::killLater(PhononSound *soundObject, bool doClose)
{
    if(!soundObject)
        return;
    soundObject->prepareToDie(doClose);
    obsoleteObjects.append(soundObject);
    QTimer::singleShot(15000, this, SLOT(onKillObsoleteObject())); // 15 seconds more to live
}

bool PhononPlayer::loadFromCache(const QString &filename)
{
    // there's no way to make a decent pre-caching with equalizer enabled
    // due to glitches in playback when two equalizers are active at one time
    if(equalizerEnabled)
        return false;
    if(!cacheExists(filename))
        return false;
    sound = cacheSound;
    cacheSound = NULL;
    return true;
}

bool PhononPlayer::cacheExists(const QString &filename)
{
    if(!cacheSound)
        return false;
    if(cacheSound->getCurrentFilename() != filename)
        return false;
    return true;
}
