/* This file is part of the KDE libraries
   Copyright (C) 2000-2005 David Faure <faure@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "filterbase.h"
#include <config.h>

#include <QtCore/QDebug>
#include <QtCore/QIODevice>
#include "gzipfilter.h"
#ifdef HAVE_BZIP2_SUPPORT
#include "bzip2filter.h"
#endif
#ifdef HAVE_XZ_SUPPORT
#include "xzfilter.h"
#endif

using namespace Quasar;

class FilterBase::Private
{
public:
    Private()
        : m_flags(WithHeaders) {}
    FilterFlags m_flags;
};

FilterBase::FilterBase()
    : m_dev( 0L ), m_bAutoDel( false ), d(new Private)
{
}

FilterBase::~FilterBase()
{
    if ( m_bAutoDel )
        delete m_dev;
    delete d;
}

void FilterBase::setDevice( QIODevice * dev, bool autodelete )
{
    m_dev = dev;
    m_bAutoDel = autodelete;
}

QIODevice * FilterBase::device()
{
    return m_dev;
}

bool FilterBase::inBufferEmpty() const
{
    return inBufferAvailable() == 0;
}

bool FilterBase::outBufferFull() const
{
    return outBufferAvailable() == 0;
}

FilterBase * FilterBase::findFilterByFileName( const QString & fileName )
{
    if ( fileName.endsWith( QLatin1String(".gz"), Qt::CaseInsensitive ) )
    {
        return new GzipFilter;
    }
#ifdef HAVE_BZIP2_SUPPORT
    if ( fileName.endsWith( QLatin1String(".bz2"), Qt::CaseInsensitive ) )
    {
        return new Bzip2Filter;
    }
#endif
#ifdef HAVE_XZ_SUPPORT
    if ( fileName.endsWith( QLatin1String(".lzma"), Qt::CaseInsensitive ) || fileName.endsWith( QLatin1String(".xz"), Qt::CaseInsensitive ) )
    {
        return new XzFilter;
    }
#endif
    else
    {
        // not a warning, since this is called often with other mimetypes (see #88574)...
        // maybe we can avoid that though?
        qDebug() << "FilterBase::findFilterByFileName : no filter found for " << fileName;
    }

    return 0;
}

FilterBase * FilterBase::findFilterByMimeType( const QString & mimeType )
{
    if (mimeType == QLatin1String("application/x-gzip")) {
        return new GzipFilter;
    }
#ifdef HAVE_BZIP2_SUPPORT
    if (mimeType == QLatin1String("application/x-bzip")
        || mimeType == QLatin1String("application/x-bzip2") // old name, kept for compatibility
       ) {
        return new Bzip2Filter;
    }
#endif
#ifdef HAVE_XZ_SUPPORT
    if ( mimeType == QLatin1String( "application/x-lzma" ) // legacy name, still used
        || mimeType == QLatin1String( "application/x-xz" ) // current naming
       ) {
        return new XzFilter;
    }
#endif

    // not a warning, since this is called often with other mimetypes (see #88574)...
    // maybe we can avoid that though?
    qDebug() << "no filter found for" << mimeType;
    return 0;
}

void FilterBase::terminate()
{
}

void FilterBase::reset()
{
}

void FilterBase::setFilterFlags(FilterFlags flags)
{
    d->m_flags = flags;
}

FilterBase::FilterFlags FilterBase::filterFlags() const
{
    return d->m_flags;
}

void FilterBase::virtual_hook( int, void* )
{ /*BASE::virtual_hook( id, data );*/ }
