#include <QtDeclarative>
#include <QtMaemo5>

#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>

#include "quickwidget.hpp"

#define SETTING_QUICK_FILE "QuickWidgetFile"
#define SETTING_QUICK_SIZE "QuickWidgetSize"
#define SETTING_QUICK_ROTATE "QuickWidgetRotate"
#define SETTING_SAVE_WAITINTERVAL 3000


QuickWidget *QuickWidget::createAndShowNew(const QString& fileName, 
					   const QSize& size,
					   bool rotate)
{
    QuickWidget *widget = new QuickWidget;
    widget->initView(fileName, size, rotate);
    widget->show();
    QeMaemo5DynamicWidgetHelper::globalInstance()->registerWidget(widget);
    return widget;
}

void QuickWidget::initView(const QString& fileName, const QSize& size, bool rotate)
{
    size_ = size;
    saveSetting(SETTING_QUICK_FILE, fileName);
    saveSetting(SETTING_QUICK_SIZE, size_);
    saveSetting(SETTING_QUICK_ROTATE, rotate);
    view_->setSource(QUrl::fromLocalFile(fileName));
    if ( view_->status() == QDeclarativeView::Error) {
        errorWidget();
    }
    else if (size_.isValid())
    {
        // use given fixed size and scale to it
        qDebug() << "scaling";
        view_->setResizeMode(QDeclarativeView::SizeRootObjectToView);
    }
    else
    {
      size_ = view_->sceneRect().size().toSize();
    }    
    connect(view_, SIGNAL(sceneResized(QSize)), this, SLOT(resizer(QSize)));
    qDebug() << size_;
    view_->resize(size_);
    if (rotate) {
      QDBusConnection::systemBus()			\
	.call(QDBusMessage::createMethodCall(MCE_SERVICE, 
					     MCE_REQUEST_PATH, 
					     MCE_REQUEST_IF, 
					     MCE_ACCELEROMETER_ENABLE_REQ));
      QDBusConnection::systemBus().connect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
					   MCE_DEVICE_ORIENTATION_SIG, 
					   this, SLOT(rotateScreen(QString)));
    }
}

void QuickWidget::rotateScreen(const QString& orientation) {

  qreal angle = 90;
  if (orientation == QLatin1String(MCE_ORIENTATION_PORTRAIT) ||
      orientation == QLatin1String(MCE_ORIENTATION_PORTRAIT_INVERTED)) {
    if (!landscape_) {
      return;
    }
    angle = -90;
    landscape_ = false;
  } else {
    if (landscape_) {
      return;
    }
    landscape_ = true;
  }
  size_.transpose();
  view_->resize(size_);
  view_->rotate(angle);
}

QuickWidget::QuickWidget(QWidget *parent)
    : QeMaemo5DynamicHomescreenWidget(true, parent),
      size_(QSize()),
      view_(0),
      wrapper_(0)
{
    //layout()->setSizeConstraint(QLayout::SetNoConstraint);
    view_ = new QDeclarativeView(this);
    view_->engine()->addImportPath(QString("/opt/qtm12/imports"));
    view_->setStyleSheet("background:transparent");
    view_->setAttribute(Qt::WA_TranslucentBackground);
    wrapper_ = new QuickWidgetWrapper(this);
    connect(this, SIGNAL(isVisibleOnCurrentHomescreenChanged(bool)), 
	    wrapper_, SLOT(setOnHomeScreen(bool)));
    view_->rootContext()->setContextProperty("runtime", wrapper_);
    landscape_ = true;
}

QuickWidget::~QuickWidget()
{

  QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, 
								   MCE_REQUEST_PATH, 
								   MCE_REQUEST_IF, 
								   MCE_ACCELEROMETER_DISABLE_REQ));
      delete wrapper_;
}



bool QuickWidget::restoreWidgetState()
{
    QString fileName = loadSetting(SETTING_QUICK_FILE).toString();
    QSize size = loadSetting(SETTING_QUICK_SIZE, QSize()).value<QSize>();
    bool rotate = loadSetting(SETTING_QUICK_ROTATE).toBool();
    initView(fileName, size, rotate);
    return true;
}

void QuickWidget::errorWidget()
{
    const QList<QDeclarativeError>& errs = view_->errors();
    qDebug() << errs;
    QByteArray errmsg;
    QTextStream qts(&errmsg);
    qts << "import Qt 4.7" << endl
        << "Rectangle { width: 200; height: txt.height; color: \"#ff0000\"; radius: 8" << endl
        << "Text { id: txt; anchors.centerIn: parent; text: \"";
    for (int i = 0; i < errs.size(); ++i) {
        qts << errs.at(i).toString().replace(QString("\""), QString("\\\"")) <<
endl;
    }
    qts << "\"; font.pixelSize: 15; wrapMode: Text.Wrap; width: parent.width-20}" << endl
        << "}" << endl;
    qDebug() <<  errmsg;
    QDeclarativeComponent component(view_->engine());
    component.setData(errmsg, QUrl());
    qDebug() << component.errorString();
    QDeclarativeItem *errwdgt =
      qobject_cast<QDeclarativeItem*>(component.create(view_->rootContext()));
    view_->scene()->addItem(errwdgt);
    size_ = QSize(errwdgt->width(), errwdgt->height());
}

void QuickWidget::resizer(const QSize& size)
{
  if (size_.isValid())
  {
      resize(size);
  }
}

