/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*- */
/* dbus-print-message.h  Utility function to print out a message
 *
 * Copyright (C) 2003 Philip Blundell <philb@gnu.org>
 * Copyright (C) 2003 Red Hat, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
#include "dbus-print-message.h"

static const char*
type_to_name (int message_type)
{
	//g_debug ("%s", G_STRFUNC);
  switch (message_type)
    {
    case DBUS_MESSAGE_TYPE_SIGNAL:
      return "signal";
    case DBUS_MESSAGE_TYPE_METHOD_CALL:
      return "method call";
    case DBUS_MESSAGE_TYPE_METHOD_RETURN:
      return "method return";
    case DBUS_MESSAGE_TYPE_ERROR:
      return "error";
    default:
      return "(unknown message type)";
    }
}


static void
indent (int depth, GString *dbusVerboseMsg)
{
	//g_debug ("%s", G_STRFUNC);
  while (depth-- > 0) {
	  //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s   ", x); g_free(x);}
	  g_string_append_printf (dbusVerboseMsg, "   ");
  }
}

static void
print_iter (DBusMessageIter *iter, dbus_bool_t literal, int depth, GString *dbusVerboseMsg)
{
	//g_debug ("%s", G_STRFUNC);
  do
    {
      int type = dbus_message_iter_get_arg_type (iter);

      if (type == DBUS_TYPE_INVALID)
      break;

      indent(depth, dbusVerboseMsg);

      switch (type)
      {
      case DBUS_TYPE_STRING:
        {
          char *val;
          dbus_message_iter_get_basic (iter, &val);
          if (!literal) {
        	//{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sstring \"", x); g_free(x);}
            g_string_append_printf (dbusVerboseMsg, "string \"");
          }
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s%s", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "%s", val);
          if (!literal) {
            //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s\"\n", x); g_free(x);}
            g_string_append_printf (dbusVerboseMsg, "\"\n");
          }
          break;
        }

      case DBUS_TYPE_SIGNATURE:
        {
          char *val;
          dbus_message_iter_get_basic (iter, &val);
          if (!literal) {
          	//{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%ssignature \"", x); g_free(x);}
            g_string_append_printf (dbusVerboseMsg, "signature \"");
          }
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s%s", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "%s", val);
          if (!literal) {
            //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s\"\n", x); g_free(x);}
            g_string_append_printf (dbusVerboseMsg, "\"\n");
          }
          break;
        }

      case DBUS_TYPE_OBJECT_PATH:
        {
          char *val;
          dbus_message_iter_get_basic (iter, &val);
          if (!literal) {
           	//{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sobject path \"", x); g_free(x);}
            g_string_append_printf (dbusVerboseMsg, "object path \"");
          }
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s%s", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "%s", val);
          if (!literal) {
            //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s\"\n", x); g_free(x);}
            g_string_append_printf (dbusVerboseMsg, "\"\n");
          }
        }

      case DBUS_TYPE_INT16:
        {
          dbus_int16_t val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sint16 %d\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "int16 %d\n", val);
          break;
        }

      case DBUS_TYPE_UINT16:
        {
          dbus_uint16_t val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%suint16 %u\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "uint16 %u\n", val);
          break;
        }

      case DBUS_TYPE_INT32:
        {
          dbus_int32_t val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sint32 %d\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "int32 %d\n", val);
          break;
        }

      case DBUS_TYPE_UINT32:
        {
          dbus_uint32_t val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%suint32 %u\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "uint32 %u\n", val);
          break;
        }

      case DBUS_TYPE_INT64:
        {
          dbus_int64_t val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sint64 %lld\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "int64 %lld\n", val);
          break;
        }

      case DBUS_TYPE_UINT64:
        {
          dbus_uint64_t val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%suint64 %llu\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "uint64 %llu\n", val);
          break;
        }

      case DBUS_TYPE_DOUBLE:
        {
          double val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sdouble %g\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "double %g\n", val);
          break;
        }

      case DBUS_TYPE_BYTE:
        {
          unsigned char val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sbyte %d\n", x, val); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "byte %d\n", val);
          break;
        }

      case DBUS_TYPE_BOOLEAN:
        {
          dbus_bool_t val;
          dbus_message_iter_get_basic (iter, &val);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sboolean %s\n", x, val ? "true" : "false"); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "boolean %s\n", val ? "true" : "false");
          break;
        }

      case DBUS_TYPE_VARIANT:
        {
          DBusMessageIter subiter;

          dbus_message_iter_recurse (iter, &subiter);

          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%svariant", x); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "variant ");
          print_iter (&subiter, literal, depth+1, dbusVerboseMsg);
          break;
        }
      case DBUS_TYPE_ARRAY:
        {
          int current_type;
          DBusMessageIter subiter;

          dbus_message_iter_recurse (iter, &subiter);

          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sarray [\n", x); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "array [\n");
          while ((current_type = dbus_message_iter_get_arg_type (&subiter)) != DBUS_TYPE_INVALID) {
            print_iter (&subiter, literal, depth+1, dbusVerboseMsg);
            dbus_message_iter_next (&subiter);
            if (dbus_message_iter_get_arg_type (&subiter) != DBUS_TYPE_INVALID) {
                //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s,", x); g_free(x);}
            	g_string_append_printf (dbusVerboseMsg, ",");
            }
          }
          indent(depth, dbusVerboseMsg);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s]\n", x); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "]\n");
          break;
        }
      case DBUS_TYPE_DICT_ENTRY:
        {
          DBusMessageIter subiter;

          dbus_message_iter_recurse (iter, &subiter);

          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sdict entry(\n", x); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "dict entry(\n");
          print_iter (&subiter, literal, depth+1, dbusVerboseMsg);
          dbus_message_iter_next (&subiter);
          print_iter (&subiter, literal, depth+1, dbusVerboseMsg);
          indent(depth, dbusVerboseMsg);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s)\n", x); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, ")\n");
          break;
        }

      case DBUS_TYPE_STRUCT:
        {
          int current_type;
          DBusMessageIter subiter;

          dbus_message_iter_recurse (iter, &subiter);

          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%sstruct {\n", x); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "struct {\n");
          while ((current_type = dbus_message_iter_get_arg_type (&subiter)) != DBUS_TYPE_INVALID) {
            print_iter (&subiter, literal, depth+1, dbusVerboseMsg);
            dbus_message_iter_next (&subiter);
            if (dbus_message_iter_get_arg_type (&subiter) != DBUS_TYPE_INVALID) {
                //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s,", x); g_free(x);}
                g_string_append_printf (dbusVerboseMsg, ",");
            }
          }
          indent(depth, dbusVerboseMsg);
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s}\n", x); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, "}\n");
          break;
        }

      default:
          //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s (dbus-monitor too dumb to decipher arg type '%c')\n", x, type); g_free(x);}
          g_string_append_printf (dbusVerboseMsg, " (dbus-monitor too dumb to decipher arg type '%c')\n", type);
          break;
      }
    } while (dbus_message_iter_next (iter));
}

void
dbus_print_message (DBusMessage *message, dbus_bool_t literal, GString *dbusVerboseMsg)
{
	//g_debug ("%s", G_STRFUNC);
  DBusMessageIter iter;
  const char *sender;
  const char *destination;
  int message_type;

  message_type = dbus_message_get_type (message);
  sender = dbus_message_get_sender (message);
  destination = dbus_message_get_destination (message);
  //g_debug ("(%p) type[%d] sender[%s] destination[%s]",dbusVerboseMsg, message_type, sender, destination);

  if (!literal)
    {
      //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s%s sender=%s -> dest=%s", x,
      g_string_append_printf (dbusVerboseMsg, "%s sender=%s -> dest=%s",
            type_to_name (message_type),
            sender ? sender : "(null sender)",
            destination ? destination : "(null destination)"); //g_free(x);}

      switch (message_type)
      {
      case DBUS_MESSAGE_TYPE_METHOD_CALL:
      case DBUS_MESSAGE_TYPE_SIGNAL:
		  //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s path=%s; interface=%s; member=%s\n", x,
    	  g_string_append_printf (dbusVerboseMsg, " path=%s; interface=%s; member=%s\n",
              dbus_message_get_path (message),
              dbus_message_get_interface (message),
              dbus_message_get_member (message)); //g_free(x);}
        //g_debug ("(%p) path[%s] interface[%s] member[%s]", dbusVerboseMsg->str, dbus_message_get_path (message), dbus_message_get_interface (message), dbus_message_get_member (message));
        break;

      case DBUS_MESSAGE_TYPE_METHOD_RETURN:
		  //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s reply_serial=%u\n", x,
    	  g_string_append_printf (dbusVerboseMsg, " reply_serial=%u\n",
          dbus_message_get_reply_serial (message)); //g_free(x);}
        //g_debug ("(%p) reply_serial[%u]", dbusVerboseMsg->str, dbus_message_get_reply_serial (message));
        break;

      case DBUS_MESSAGE_TYPE_ERROR:
		  //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s error_name=%s reply_serial=%u\n", x,
    	  g_string_append_printf (dbusVerboseMsg, " error_name=%s reply_serial=%u\n",
              dbus_message_get_error_name (message),
          dbus_message_get_reply_serial (message)); //g_free(x);}
        //g_debug ("(%p) reply_serial[%u] error_name[%s]", dbusVerboseMsg->str, dbus_message_get_reply_serial (message), dbus_message_get_error_name (message));
        break;

      default:
		  //{gchar *x = *dbusVerboseMsg; *dbusVerboseMsg = g_strdup_printf ("%s\n", x); g_free(x);}
    	  g_string_append_printf (dbusVerboseMsg, "\n");
    	  break;
      }
    }

  dbus_message_iter_init (message, &iter);
  print_iter (&iter, literal, 1, dbusVerboseMsg);
  //fflush (stdout);

}

