/*
 * This file is a part of Queen Beecon Widget
 * queen-beecon-settings-manager: Utility for QBW Settings Management
 *
 * http://talk.maemo.org/showthread.php?t=45388
 *
 * Copyright (c) 2010 No!No!No!Yes! (Alessandro Peralma)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <hildon/hildon.h>
#include <string.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include "queen-beecon-logger.h"
#include "queen-beecon.h"
#include "queen-beecon-progress-animation.h"
#include "queen-beecon-settings-manager.h"

extern struct _colorsRGB p_colors[];
extern gchar * p_fonts[];
extern gchar * p_fonts_types[];

//Settings file functions
void queen_beecon_read_settings ( QueenBeecon *self, gboolean needInstanceSettings, gboolean isImportFromADB, gchar *importData)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V1, "(%p) %s", self, G_STRFUNC);
	gchar *filename=NULL, *sourceGroup=NULL;
	gboolean fileExists;
	GKeyFile *keyFile=NULL;
	GError *error=NULL;

	if(self->priv->widgetID == NULL) {
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Widget instance not initialized, not reading settings..", self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Widget instance not initialized, not reading settings..", self);
		return;
	}

	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile %p", self, keyFile);
	if (!isImportFromADB) {
		sourceGroup=self->priv->widgetID;
	} else {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Import from ADB", self);
		sourceGroup="queen-beecon-exported-instance";
		fileExists = g_key_file_load_from_data (keyFile, importData, strlen(importData), G_KEY_FILE_KEEP_COMMENTS, NULL);
		if (!fileExists) {
			hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Import from ADB buffer error", self);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Import from ADB buffer error", self);
			return;
		} else goto needInstanceSettings;
	}
	filename = g_strconcat (HOME_DIR, QUEEN_BEECON_SETTINGS_FILE, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) filename %s", self, filename);
	fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileExists %d", self, fileExists);

	gdouble settingsVer = queen_beecon_g_key_file_get_double (keyFile, "config", "version", QUEEN_BEECON_SETTINGS_VERSION);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) settingsVer %f", self, settingsVer);

	guint cmdListInConsistent = 0;
	if (fileExists) {
		//gets "global" data/commands list
		//BEGIN get list
		g_strfreev(self->priv->c_commands);
		self->priv->c_commands = (gchar **) g_key_file_get_string_list (keyFile, "config", "c_commands", &(self->priv->c_size) ,&error);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) c_size=%d", self, self->priv->c_size);
		gint i;	for (i=0;i<self->priv->c_size;i++) qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) c_commands[%d]=%s", self, i, self->priv->c_commands[i]);
		if (error) {
			hildon_banner_show_informationf (NULL, NULL, "(%p) QBW ERROR g_key_file_get_string_list (keyFile, \"config\", \"c_commands\", ...)", self);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) ERROR g_key_file_get_string_list (keyFile, \"config\", \"c_commands\", ...)", self);
			cmdListInConsistent++;
			g_clear_error (&error);
			error = NULL;
		}

		guint consistencyCheck = -1;

		g_strfreev(self->priv->c_titles);
		self->priv->c_titles = g_key_file_get_string_list (keyFile, "config", "c_titles", &consistencyCheck ,&error);
		gint j;	for (j=0;j<consistencyCheck;j++) {
			;
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) c_titles[%d]=%s", self, j, self->priv->c_titles[j]);
		}
		if (error) {
			hildon_banner_show_informationf (NULL, NULL, "(%p) QBW ERROR g_key_file_get_string_list (keyFile, \"config\", \"c_titles\", ...)", self);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) ERROR g_key_file_get_string_list (keyFile, \"config\", \"c_titles\", ...)", self);
			cmdListInConsistent++;
			g_clear_error (&error);
			error = NULL;
		}

		if(consistencyCheck != self->priv->c_size) {
			hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Settings file corrupted!\nRebuilding new one!", self);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) consistencyCheck FAILED=%d", self, consistencyCheck);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Settings file corrupted!", self);
			cmdListInConsistent++;
		}
		//END
	} else
		cmdListInConsistent++;
	if (cmdListInConsistent) {
		#define QUEEN_BEECON_TOTAL_FIXED_COMMANDS 13
		gchar * p_titles[] = { "appOK", "appWRN", "appERR", "sysERR", "Uptime:", "Battery(%):", "Battery(mAh):", "Boot Reason:", "Boot Count:", "External IP:", "Internal IP:", "Rootfs(%):", "Free Rootfs:", NULL};
		gchar * p_commands[] = {
		"echo test && exit 0",
		"echo test && exit 1",
		"echo test && exit 2",
		"all_work_and_no_play_makes_jack_a_dull_boy",
		"uptime|cut -d\" \" -f4-|sed 's/\\, *load.*//'",
		"hal-device bme | awk -F\"[. ]\" '$5 == \"is_charging\" {chrg = $7}; $5 == \"percentage\" {perc = $7} END if (chrg == \"false\") {print perc \"%\"} else {print \"Chrg\"}'",
		"hal-device bme | grep battery.reporting | awk -F. '{print $3}' | sort | awk '$1 == \"current\" { current = $3}; $1== \"design\" {print current \"/\" $3}'",
		"cat /proc/bootreason",
		"cat /var/lib/dsme/boot_count",
		"wget -q -O - api.myiptest.com | awk -F \"\\\"\" '{print $4}'",
		"/sbin/ifconfig | grep \"inet addr\" | awk -F: '{print $2}' | awk '{print $1}'",
		"df | awk '$1 == \"rootfs\" {print $5}'",
		"df -h | awk ' $1 == \"rootfs\" {print $4\"B\"}'",
		NULL};
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Records empty, filling with pre-loaded vals", self);
		self->priv->c_size = QUEEN_BEECON_TOTAL_FIXED_COMMANDS;
		//clean possible oldies
		g_strfreev(self->priv->c_titles);
		self->priv->c_titles = NULL;
		g_strfreev(self->priv->c_commands);
		self->priv->c_commands = NULL;
		//assign new from stack values
		self->priv->c_titles = g_strdupv(p_titles);
		self->priv->c_commands = g_strdupv(p_commands);
	}

needInstanceSettings:
	if (needInstanceSettings) {
		/*getting instance specific data*/
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Getting instance specific data: %s", self, sourceGroup);
		//BEGIN
		self->priv->widgetType = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "widgetType", 0);//0=Beecon
		self->priv->widgetVisible = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "widgetVisible", 1);//0=Hidden; 1=Visible
		self->priv->operationalStatus = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "operationalStatus", 0);//0=Maximized/Enabled
		self->priv->hideCanvas = queen_beecon_g_key_file_get_boolean (keyFile, sourceGroup, "hideCanvas", FALSE);//FALSE=Show Canvas by default
		self->priv->snippetBgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "snippetBgRGB", 0);//0=Black

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->snippetExtBgRGB = %p", self, self->priv->snippetExtBgRGB);
		g_free(self->priv->snippetExtBgRGB);self->priv->snippetExtBgRGB=NULL;
		self->priv->snippetExtBgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "snippetExtBgRGB", p_colors[self->priv->resFgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->snippetExtBgRGB = %p", self, self->priv->snippetExtBgRGB);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_lt0_ImgFilename = %p", self, self->priv->beecon_lt0_ImgFilename);
		g_free(self->priv->beecon_lt0_ImgFilename);self->priv->beecon_lt0_ImgFilename=NULL;
		self->priv->beecon_lt0_ImgFilename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_lt0_ImgFilename", "queen-beecon-syserr.png"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_lt0_ImgFilename = %p", self, self->priv->beecon_lt0_ImgFilename);

		self->priv->beecon_lt0_ImgZoom = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_lt0_ImgZoom", 0);//0=100% 1=Fit to Widget
		self->priv->beecon_lt0_BgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_lt0_BgRGB", 2);//2=Gray

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_lt0_ExtBgRGB = %p", self, self->priv->beecon_lt0_ExtBgRGB);
		g_free(self->priv->beecon_lt0_ExtBgRGB);self->priv->beecon_lt0_ExtBgRGB=NULL;
		self->priv->beecon_lt0_ExtBgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_lt0_ExtBgRGB", p_colors[self->priv->beecon_lt0_BgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_lt0_ExtBgRGB = %p", self, self->priv->beecon_lt0_ExtBgRGB);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq0_ImgFilename = %p", self, self->priv->beecon_eq0_ImgFilename);
		g_free(self->priv->beecon_eq0_ImgFilename);self->priv->beecon_eq0_ImgFilename=NULL;
		self->priv->beecon_eq0_ImgFilename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_eq0_ImgFilename", "queen-beecon-appok.png"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq0_ImgFilename = %p", self, self->priv->beecon_eq0_ImgFilename);

		self->priv->beecon_eq0_ImgZoom = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_eq0_ImgZoom", 0);//0=100% 1=Fit to Widget
		self->priv->beecon_eq0_BgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_eq0_BgRGB", 3);//3=Green

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq0_ExtBgRGB = %p", self, self->priv->beecon_eq0_ExtBgRGB);
		g_free(self->priv->beecon_eq0_ExtBgRGB);self->priv->beecon_eq0_ExtBgRGB=NULL;
		self->priv->beecon_eq0_ExtBgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_eq0_ExtBgRGB", p_colors[self->priv->beecon_eq0_BgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq0_ExtBgRGB = %p", self, self->priv->beecon_eq0_ExtBgRGB);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq1_ImgFilename = %p", self, self->priv->beecon_eq1_ImgFilename);
		g_free(self->priv->beecon_eq1_ImgFilename);self->priv->beecon_eq1_ImgFilename=NULL;
		self->priv->beecon_eq1_ImgFilename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_eq1_ImgFilename", "queen-beecon-appwrn.png"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq1_ImgFilename = %p", self, self->priv->beecon_eq1_ImgFilename);

		self->priv->beecon_eq1_ImgZoom = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_eq1_ImgZoom", 0);//0=100% 1=Fit to Widget
		self->priv->beecon_eq1_BgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_eq1_BgRGB", 4);//4=yellow

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq1_ExtBgRGB = %p", self, self->priv->beecon_eq1_ExtBgRGB);
		g_free(self->priv->beecon_eq1_ExtBgRGB);self->priv->beecon_eq1_ExtBgRGB=NULL;
		self->priv->beecon_eq1_ExtBgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_eq1_ExtBgRGB", p_colors[self->priv->beecon_eq1_BgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_eq1_ExtBgRGB = %p", self, self->priv->beecon_eq1_ExtBgRGB);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_ge2_ImgFilename = %p", self, self->priv->beecon_ge2_ImgFilename);
		g_free(self->priv->beecon_ge2_ImgFilename);self->priv->beecon_ge2_ImgFilename=NULL;
		self->priv->beecon_ge2_ImgFilename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_ge2_ImgFilename", "queen-beecon-apperr.png"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_ge2_ImgFilename = %p", self, self->priv->beecon_ge2_ImgFilename);

		self->priv->beecon_ge2_ImgZoom = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_ge2_ImgZoom", 0);//0=100% 1=Fit to Widget
		self->priv->beecon_ge2_BgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_ge2_BgRGB", 5);//5=red

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_ge2_ExtBgRGB = %p", self, self->priv->beecon_ge2_ExtBgRGB);
		g_free(self->priv->beecon_ge2_ExtBgRGB);self->priv->beecon_ge2_ExtBgRGB=NULL;
		self->priv->beecon_ge2_ExtBgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_ge2_ExtBgRGB", p_colors[self->priv->beecon_ge2_BgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_ge2_ExtBgRGB = %p", self, self->priv->beecon_ge2_ExtBgRGB);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_idxge2_ImgFilename = %p", self, self->priv->beecon_idxge2_ImgFilename);
		g_free(self->priv->beecon_idxge2_ImgFilename);self->priv->beecon_idxge2_ImgFilename=NULL;
		self->priv->beecon_idxge2_ImgFilename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_idxge2_ImgFilename", "queen-beecon-index.png"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_idxge2_ImgFilename = %p", self, self->priv->beecon_idxge2_ImgFilename);

		self->priv->beecon_idxge2_ImgZoom = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_idxge2_ImgZoom", 0);//0=100% 1=Fit to Widget
		self->priv->beecon_idxge2_BgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "beecon_idxge2_BgRGB", 11);//5=aqua

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_idxge2_ExtBgRGB = %p", self, self->priv->beecon_idxge2_ExtBgRGB);
		g_free(self->priv->beecon_idxge2_ExtBgRGB);self->priv->beecon_idxge2_ExtBgRGB=NULL;
		self->priv->beecon_idxge2_ExtBgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "beecon_idxge2_ExtBgRGB", p_colors[self->priv->beecon_idxge2_BgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->beecon_idxge2_ExtBgRGB = %p", self, self->priv->beecon_idxge2_ExtBgRGB);

		self->priv->widWidth = queen_beecon_g_key_file_get_double (keyFile, sourceGroup, "widWidth", 130.0);//130px
		self->priv->widHeight = queen_beecon_g_key_file_get_double (keyFile, sourceGroup, "widHeight", 115.0);//115px

//----------
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->cmdImgFilename = %p", self, self->priv->cmdImgFilename);
		g_free(self->priv->cmdImgFilename);self->priv->cmdImgFilename=NULL;
		self->priv->cmdImgFilename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "cmdImgFilename", "queen-beecon.png"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->cmdImgFilename = %p", self, self->priv->cmdImgFilename);

		self->priv->cmdImgAngle = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdImgAngle", 0); //Horizontal, Left to Right
		self->priv->cmdImgZoom = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdImgZoom", 0);//0=100% 1=Fit to Widget
		self->priv->cmdFgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdFgRGB", 1);//1-White

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->cmdExtFgRGB = %p", self, self->priv->cmdExtFgRGB);
		g_free(self->priv->cmdExtFgRGB);self->priv->cmdExtFgRGB=NULL;
		self->priv->cmdExtFgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "cmdExtFgRGB", p_colors[self->priv->cmdFgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->cmdExtFgRGB = %p", self, self->priv->cmdExtFgRGB);

		self->priv->cmdTextAngle = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdTextAngle", 0); //Horizontal, Left to Right
		self->priv->cmdVisibilityPosition = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdVisibilityPosition", 2);//2=top-center
		self->priv->cmdImgVisibilityPosition = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdImgVisibilityPosition", 5);//1=center
		self->priv->cmdJustify = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdJustify", 0);//0=left
		self->priv->cmdFontName = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "cmdFontName", 3);//
		self->priv->cmdFontSize = queen_beecon_g_key_file_get_double (keyFile, sourceGroup, "cmdFontSize", 12);//12

		gchar *cmdFontTtl = g_strdup_printf("%s %s %ld", p_fonts[self->priv->cmdFontName/4], self->priv->cmdFontName%4?p_fonts_types[self->priv->cmdFontName%4]:"", (long) self->priv->cmdFontSize);//AP1
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Command Font: %s",self, cmdFontTtl);
		g_free(self->priv->cmdExtFont);self->priv->cmdExtFont=NULL;
		self->priv->cmdExtFont = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "cmdExtFont", cmdFontTtl));//No Image filename
		g_free(cmdFontTtl);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->cmdExtFont = [%p] %s", self, self->priv->cmdExtFont, self->priv->cmdExtFont);

//----------
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->resImgFilename = %p", self, self->priv->resImgFilename);
		g_free(self->priv->resImgFilename);self->priv->resImgFilename=NULL;
		self->priv->resImgFilename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "resImgFilename", "queen-beecon-resimg.png"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->resImgFilename = %p", self, self->priv->resImgFilename);

		self->priv->resImgAngle = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resImgAngle", 0); //Horizontal, Left to Right
		self->priv->resImgZoom = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resImgZoom", 0);//0=100% 1=Fit to Widget
		self->priv->resFgRGB = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resFgRGB", 1);//1=White

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->resExtFgRGB = %p", self, self->priv->resExtFgRGB);
		g_free(self->priv->resExtFgRGB);self->priv->resExtFgRGB=NULL;
		self->priv->resExtFgRGB = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "resExtFgRGB", p_colors[self->priv->resFgRGB].colorRGB));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->resExtFgRGB = %p", self, self->priv->resExtFgRGB);

		self->priv->resTextAngle = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resTextAngle", 0); //Horizontal, Left to Right
		self->priv->resVisibilityPosition = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resVisibilityPosition", 8);//center-center
		self->priv->resImgVisibilityPosition = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resImgVisibilityPosition", 9);//bottom-center
		self->priv->resJustify = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resJustify", 0);//0=left

		self->priv->resFontName = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "resFontName", 2);//
		self->priv->resFontSize = queen_beecon_g_key_file_get_double (keyFile, sourceGroup, "resFontSize", 12);//12

		gchar *resFontTtl = g_strdup_printf("%s %s %ld", p_fonts[self->priv->resFontName/4], self->priv->resFontName%4?p_fonts_types[self->priv->resFontName%4]:"", (long) self->priv->resFontSize);//AP1
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Result Font: %s",self, resFontTtl);
		g_free(self->priv->resExtFont);self->priv->resExtFont=NULL;
		self->priv->resExtFont = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "resExtFont", resFontTtl));//No Image filename
		g_free(resFontTtl);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->resExtFont = [%p] %s", self, self->priv->resExtFont, self->priv->resExtFont);
//----------
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceTitle = %p [%s]", self, self->priv->instanceTitle, self->priv->instanceTitle);
		if (self->priv->instanceTitle!=NULL) g_free(self->priv->instanceTitle); self->priv->instanceTitle = NULL;
		self->priv->instanceTitle = queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "instanceTitle", NULL); //If value not found assume first command in fixed list
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceTitle = %p [%s]", self, self->priv->instanceTitle, self->priv->instanceTitle);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceCmd = %p [%s]", self, self->priv->instanceCmd, self->priv->instanceCmd);
		if (self->priv->instanceCmd!=NULL) g_free(self->priv->instanceCmd); self->priv->instanceCmd = NULL;
		self->priv->instanceCmd = queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "instanceCmd", NULL); //If value not found assume first command in fixed list
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceCmd = %p [%s]", self, self->priv->instanceCmd, self->priv->instanceCmd);

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->rememberMe = %p [%s]", self, self->priv->rememberMe, self->priv->rememberMe);
		if (self->priv->rememberMe!=NULL) g_free(self->priv->rememberMe); self->priv->rememberMe = NULL;
		self->priv->rememberMe = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "rememberMe", "")); //Remember me string to use for any kind of status persistency storage across scripts/command executions
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->rememberMe = %p [%s]", self, self->priv->rememberMe, self->priv->rememberMe);

		//BEGIN progress animation
		self->priv->progressAnimationFrames = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "progressAnimationFrames", QBW_PROGRESS_ANIMATION_FRAMES_DEFAULT);//QBW_PROGRESS_ANIMATION_FRAMES_DEFAULT=Animation Frames is default (0=Hides animation)
		self->priv->progressAnimationTimer = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "progressAnimationTimer", QBW_PROGRESS_ANIMATION_FRAMES_TIMER_MSECS/QBW_PROGRESS_ANIMATION_FRAMES_TIMER_STEPS_MSECS);//QBW_PROGRESS_ANIMATION_FRAMES_TIMER_MSECS=Animation Frames timer default
		self->priv->progressAnimationPos = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "progressAnimationPos", 4);//Animation Position Centered by default

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->progressAnimationBasename = %p", self, self->priv->progressAnimationBasename);
		g_free(self->priv->progressAnimationBasename);self->priv->progressAnimationBasename=NULL;
		self->priv->progressAnimationBasename = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "progressAnimationBasename", "queen-beecon-progress"));//No Image filename
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->progressAnimationBasename = %p", self, self->priv->progressAnimationBasename);

		//----------
		//BEGIN update policy
		self->priv->updOnStartup = queen_beecon_g_key_file_get_boolean (keyFile, sourceGroup, "updOnStartup", FALSE);//FALSE= Do NOT Update when widget launched
		self->priv->updOnClick = queen_beecon_g_key_file_get_boolean (keyFile, sourceGroup, "updOnClick", TRUE);//TRUE= Update when clicked
		self->priv->updOnDesktop = queen_beecon_g_key_file_get_boolean (keyFile, sourceGroup, "updOnDesktop", FALSE);//TRUE= Do NOT Update when widget's desktop switched
		self->priv->updOnSight = queen_beecon_g_key_file_get_boolean (keyFile, sourceGroup, "updOnSight", FALSE);//FALSE= Update even if out of sight
		self->priv->delayIndex = (guint) queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "delayIndex", 0);//0=Assume no timer event
		if (self->priv->delayIndex) self->priv->delaySeconds = GetSeconds(self->priv->delayIndex);
		self->priv->customIntervalSecs = (guint) queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "customIntervalSecs", 0);//0=Assume no timer event
		if (self->priv->delayIndex==9) self->priv->delaySeconds = self->priv->customIntervalSecs;
		self->priv->updNeworkPolicy = (guint) queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "updNeworkPolicy", NETWORK__UNRELATED);
//---------- DBUS parameters
		self->priv->updOnDBUS = queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "updOnDBUS", 0);//FALSE= Do NOT Update on DBUS events
		self->priv->updOnDBUSBus = (guint) queen_beecon_g_key_file_get_integer (keyFile, sourceGroup, "updOnDBUSBus", 0); //Defaults to System DBUS

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->updOnDBUSMatchRule = %p [%s]", self, self->priv->updOnDBUSMatchRule, self->priv->updOnDBUSMatchRule);
		if (self->priv->updOnDBUSMatchRule!=NULL) g_free(self->priv->updOnDBUSMatchRule); self->priv->updOnDBUSMatchRule = NULL;
		self->priv->updOnDBUSMatchRule = g_strdup(queen_beecon_g_key_file_get_string (keyFile, sourceGroup, "updOnDBUSMatchRule", ""));
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->updOnDBUSMatchRule = %p [%s]", self, self->priv->updOnDBUSMatchRule, self->priv->updOnDBUSMatchRule);

		//END
	}
	g_key_file_free (keyFile);
	if (!isImportFromADB) g_free (filename);

	queen_beecon_write_settings (self, TRUE);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Settings file loaded successfully", self);
}

void queen_beecon_write_settings (QueenBeecon *self, gboolean needInstanceSettings)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V1, "(%p) %s", self, G_STRFUNC);
	GKeyFile *keyFile=NULL;
	gboolean fileExists;

	gchar *fileData=NULL;
	FILE *iniFile=NULL;
	gsize size;
	gchar *filename=NULL;

	if(self->priv->widgetID == NULL) {
		hildon_banner_show_informationf (NULL, NULL, "(%p) QBW Widget instance not initialized, not writing settings..", self);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Widget instance not initialized, not writing settings..", self);
		return;
	}

	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile %p", self, keyFile);

	filename = g_strconcat (HOME_DIR, QUEEN_BEECON_SETTINGS_FILE, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) filename %s", self, filename);
	fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileExists %d", self, fileExists);

	g_key_file_set_double (keyFile, "config", "version", QUEEN_BEECON_SETTINGS_VERSION);

	if(self->priv->c_titles != NULL && self->priv->c_commands != NULL)
	{
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->c_titles=%p self->priv->c_commands=%p", self, self->priv->c_titles, self->priv->c_commands);
		g_key_file_set_string_list(keyFile, "config", "c_titles", (const gchar **)(self->priv->c_titles),self->priv->c_size);
		g_key_file_set_string_list(keyFile, "config", "c_commands", (const gchar **)(self->priv->c_commands),self->priv->c_size);
	}

	if (needInstanceSettings) {
		g_key_file_set_integer (keyFile, self->priv->widgetID, "widgetType", self->priv->widgetType);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "widgetVisible", self->priv->widgetVisible);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "operationalStatus", self->priv->operationalStatus);//AP3
		g_key_file_set_boolean (keyFile, self->priv->widgetID, "hideCanvas", self->priv->hideCanvas);//FALSE=Show Canvas by default
		g_key_file_set_integer (keyFile, self->priv->widgetID, "snippetBgRGB", self->priv->snippetBgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "snippetExtBgRGB", self->priv->snippetExtBgRGB);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_lt0_ImgZoom", self->priv->beecon_lt0_ImgZoom);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_lt0_BgRGB", self->priv->beecon_lt0_BgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_lt0_ExtBgRGB", self->priv->beecon_lt0_ExtBgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_lt0_ImgFilename", self->priv->beecon_lt0_ImgFilename);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_eq0_ImgZoom", self->priv->beecon_eq0_ImgZoom);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_eq0_BgRGB", self->priv->beecon_eq0_BgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_eq0_ExtBgRGB", self->priv->beecon_eq0_ExtBgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_eq0_ImgFilename", self->priv->beecon_eq0_ImgFilename);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_eq1_ImgZoom", self->priv->beecon_eq1_ImgZoom);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_eq1_BgRGB", self->priv->beecon_eq1_BgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_eq1_ExtBgRGB", self->priv->beecon_eq1_ExtBgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_eq1_ImgFilename", self->priv->beecon_eq1_ImgFilename);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_ge2_ImgZoom", self->priv->beecon_ge2_ImgZoom);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_ge2_BgRGB", self->priv->beecon_ge2_BgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_ge2_ExtBgRGB", self->priv->beecon_ge2_ExtBgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_ge2_ImgFilename", self->priv->beecon_ge2_ImgFilename);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_idxge2_ImgZoom", self->priv->beecon_idxge2_ImgZoom);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "beecon_idxge2_BgRGB", self->priv->beecon_idxge2_BgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_idxge2_ExtBgRGB", self->priv->beecon_idxge2_ExtBgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "beecon_idxge2_ImgFilename", self->priv->beecon_idxge2_ImgFilename);//AP3
		g_key_file_set_double (keyFile, self->priv->widgetID, "widWidth", self->priv->widWidth);
		g_key_file_set_double (keyFile, self->priv->widgetID, "widHeight", self->priv->widHeight);

		if(self->priv->instanceTitle == NULL || self->priv->instanceCmd == NULL)
		{
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceTitle = %p [%s]", self, self->priv->instanceTitle, self->priv->instanceTitle);
			if (self->priv->instanceTitle != NULL) {g_free(self->priv->instanceTitle);self->priv->instanceTitle = NULL;}
			self->priv->instanceTitle = g_strdup(self->priv->c_titles[0]);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceTitle = %p [%s]", self, self->priv->instanceTitle, self->priv->instanceTitle);

			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceCmd = %p [%s]", self, self->priv->instanceCmd, self->priv->instanceCmd);
			if (self->priv->instanceCmd != NULL) {g_free(self->priv->instanceCmd);self->priv->instanceCmd = NULL;}
			self->priv->instanceCmd = g_strdup(self->priv->c_commands[0]);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) self->priv->instanceCmd = %p [%s]", self, self->priv->instanceCmd, self->priv->instanceCmd);
		}
		g_key_file_set_string (keyFile, self->priv->widgetID, "instanceTitle", self->priv->instanceTitle);
		g_key_file_set_string (keyFile, self->priv->widgetID, "instanceCmd", self->priv->instanceCmd);
		g_key_file_set_string (keyFile, self->priv->widgetID, "rememberMe", self->priv->rememberMe);
		g_key_file_set_string (keyFile, self->priv->widgetID, "cmdImgFilename", self->priv->cmdImgFilename);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdImgAngle", self->priv->cmdImgAngle);//AP1
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdImgZoom", self->priv->cmdImgZoom);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdFgRGB", self->priv->cmdFgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "cmdExtFgRGB", self->priv->cmdExtFgRGB);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdTextAngle", self->priv->cmdTextAngle);//AP1
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdVisibilityPosition", self->priv->cmdVisibilityPosition);//AP2
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdImgVisibilityPosition", self->priv->cmdImgVisibilityPosition);//AP2
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdJustify", self->priv->cmdJustify);//AP2
		g_key_file_set_string (keyFile, self->priv->widgetID, "cmdExtFont", self->priv->cmdExtFont);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "cmdFontName", self->priv->cmdFontName);//AP1
		g_key_file_set_double (keyFile, self->priv->widgetID, "cmdFontSize", self->priv->cmdFontSize);//AP1
		g_key_file_set_string (keyFile, self->priv->widgetID, "resImgFilename", self->priv->resImgFilename);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resImgAngle", self->priv->resImgAngle);//AP1
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resImgZoom", self->priv->resImgZoom);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resFgRGB", self->priv->resFgRGB);//AP3
		g_key_file_set_string (keyFile, self->priv->widgetID, "resExtFgRGB", self->priv->resExtFgRGB);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resTextAngle", self->priv->resTextAngle);//AP1
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resVisibilityPosition", self->priv->resVisibilityPosition);//AP2
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resImgVisibilityPosition", self->priv->resImgVisibilityPosition);//AP2
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resJustify", self->priv->resJustify);//AP2
		g_key_file_set_string (keyFile, self->priv->widgetID, "resExtFont", self->priv->resExtFont);//AP3
		g_key_file_set_integer (keyFile, self->priv->widgetID, "resFontName", self->priv->resFontName);//AP1
		g_key_file_set_double (keyFile, self->priv->widgetID, "resFontSize", self->priv->resFontSize);//AP1
		g_key_file_set_integer (keyFile, self->priv->widgetID, "progressAnimationFrames", self->priv->progressAnimationFrames);
		g_key_file_set_integer (keyFile, self->priv->widgetID, "progressAnimationTimer", self->priv->progressAnimationTimer);
		g_key_file_set_integer (keyFile, self->priv->widgetID, "progressAnimationPos", self->priv->progressAnimationPos);
		g_key_file_set_string (keyFile, self->priv->widgetID, "progressAnimationBasename", self->priv->progressAnimationBasename);
		g_key_file_set_boolean (keyFile, self->priv->widgetID, "updOnStartup", self->priv->updOnStartup);
		g_key_file_set_boolean (keyFile, self->priv->widgetID, "updOnClick", self->priv->updOnClick);
		g_key_file_set_boolean (keyFile, self->priv->widgetID, "updOnDesktop", self->priv->updOnDesktop);
		g_key_file_set_boolean (keyFile, self->priv->widgetID, "updOnSight", self->priv->updOnSight);
		g_key_file_set_integer (keyFile, self->priv->widgetID, "delayIndex", self->priv->delayIndex);
		g_key_file_set_integer (keyFile, self->priv->widgetID, "customIntervalSecs", self->priv->customIntervalSecs);
		g_key_file_set_integer (keyFile, self->priv->widgetID, "updNeworkPolicy", self->priv->updNeworkPolicy);
		g_key_file_set_integer (keyFile, self->priv->widgetID, "updOnDBUS", self->priv->updOnDBUS);
		g_key_file_set_integer (keyFile, self->priv->widgetID, "updOnDBUSBus", self->priv->updOnDBUSBus);//AP3
		g_key_file_set_string  (keyFile, self->priv->widgetID, "updOnDBUSMatchRule", self->priv->updOnDBUSMatchRule);//AP3
	}
	filename = g_strconcat (HOME_DIR, QUEEN_BEECON_SETTINGS_FILE, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) filename %s", self, filename);

	fileData = g_key_file_to_data (keyFile, &size, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileData %p", self, fileData);

	iniFile = fopen (filename, "w");
	fputs (fileData, iniFile);
	fclose (iniFile);
	g_key_file_free (keyFile);
	g_free (fileData);
	g_free (filename);
}

gchar *queen_beecon_g_key_file_get_string(GKeyFile *key_file, const gchar *group_name, const gchar *key, gchar *defaultValue)
{
	GError *error=NULL;
	gchar *setting = g_key_file_get_string(key_file, group_name, key,&error);
	if (error) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) No setting [%s] ->[%s]", NULL, key, defaultValue);
		g_clear_error (&error);//AP3
		return(defaultValue);
	} else
		return(setting);
}

gboolean queen_beecon_g_key_file_get_boolean(GKeyFile *key_file, const gchar *group_name, const gchar *key, gboolean defaultValue)
{
	GError *error=NULL;
	gboolean setting = g_key_file_get_boolean(key_file, group_name, key,&error);
	if (error) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) No setting [%s] ->[%d]", NULL, key, defaultValue);
		g_clear_error (&error);//AP3
		return(defaultValue);
	} else
		return(setting);
}

gint queen_beecon_g_key_file_get_integer(GKeyFile *key_file, const gchar *group_name, const gchar *key, gint defaultValue)
{
	GError *error=NULL;
	gint setting = g_key_file_get_integer(key_file, group_name, key,&error);
	if (error) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) No setting [%s] ->[%d]", NULL, key, defaultValue);
		g_clear_error (&error);//AP3
		return(defaultValue);
	}else
		return(setting);
}

gdouble queen_beecon_g_key_file_get_double(GKeyFile *key_file, const gchar *group_name, const gchar *key, gdouble defaultValue)
{
	GError *error=NULL;
	gdouble setting = g_key_file_get_double(key_file, group_name, key, &error);
	if (error) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) No setting [%s] ->[%f]", NULL, key, defaultValue);
		g_clear_error (&error);//AP3
		return(defaultValue);
	} else
		return(setting);
}

gboolean write_settings_have_key(const gchar *group_name, const gchar *key, void *value, char keyType)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) %s %s %s %p %c", NULL, G_STRFUNC, group_name, key, value, keyType);
	GKeyFile *keyFile = NULL;
	gchar *filename = NULL;
	gchar *fileData = NULL;
	FILE *iniFile = NULL;
	gsize size;
	gboolean fileExists;
	GError *error = NULL;
	gboolean RC = TRUE;

	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile %p", NULL, keyFile);
	filename = g_strconcat (HOME_DIR, QUEEN_BEECON_SETTINGS_FILE, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) filename %s", NULL, filename);
	fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileExists %d", NULL, fileExists);

	if (fileExists) {
		if (!g_key_file_has_key(keyFile, group_name, key, &error)) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) g_key_file_has_key failed: %s\n", NULL, error->message);
			g_clear_error (&error);
			RC = FALSE; // If key is not present in settings file then returns FALSE
		} else {
			switch (keyType) {
			case 'b':
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) bool %s %s %d", NULL, group_name, key, *(gboolean *)value);
				g_key_file_set_boolean (keyFile, group_name, key, *(gboolean *)value);
				break;
			case 'i':
			case 'u':
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) int uint %s %s %d", NULL, group_name, key, *(gint *)value);
				g_key_file_set_integer (keyFile, group_name, key, *(gint *)value);
				break;
			case 'd':
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) double %s %s %f", NULL, group_name, key, *(gdouble *)value);
				g_key_file_set_double (keyFile, group_name, key, *(gdouble *)value);
				break;
			case 's':
			default:
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) string %s %s %s", NULL, group_name, key, (gchar *)value);
				g_key_file_set_string (keyFile, group_name, key, (gchar *)value);
			}
			fileData = g_key_file_to_data (keyFile, &size, NULL);
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileData %p", NULL, fileData);
			iniFile = fopen (filename, "w");
			fputs (fileData, iniFile);
			fclose (iniFile);
			g_free (fileData);
		}
	} else RC = FALSE;
	g_free (filename);
	g_key_file_free (keyFile);
	return(RC);
}

gchar *read_settings_have_key (const gchar *group_name, const gchar *key)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) %s %s %s", NULL, G_STRFUNC, group_name, key);
	gchar *filename = NULL;
	gboolean fileExists;
	GKeyFile *keyFile = NULL;
	GError *error=NULL;
	gchar *prm = NULL;

	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile %p", NULL, keyFile);
	filename = g_strconcat (HOME_DIR, QUEEN_BEECON_SETTINGS_FILE, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) filename %s", NULL, filename);
	fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileExists %d", NULL, fileExists);

	prm = g_strdup(g_key_file_get_string(keyFile, group_name, key, &error));
	if (error) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) No setting [%s] on key file", NULL, key);
		g_clear_error (&error);//AP3
		if (prm) g_free (prm);
	}
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) gn=%s k=%s vp=%p v=%s", NULL, group_name, key, prm, prm);
	g_key_file_free (keyFile);
	g_free (filename);
	return prm;
}

void queen_beecon_config_file_housekeeping(gpointer data)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V1, "(%p) %s", NULL, G_STRFUNC);

	GKeyFile *keyFile = NULL;
	gchar *filename = NULL;
	gchar *fileData = NULL;
	FILE *iniFile = NULL;
	gsize size;
	gboolean fileExists;
	GError *error = NULL;
    gchar **groups = NULL;
	gint i;
	gsize groupLength;

	// :) Do not mess up with the housekeeper!!!
	if (QUEEN_BEECON_GET_CLASS(data)->housekeeping) return;

	QUEEN_BEECON_GET_CLASS(data)->housekeeping=TRUE;
	keyFile = g_key_file_new();
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) keyfile%p", data, keyFile);
	filename = g_strconcat (HOME_DIR, QUEEN_BEECON_SETTINGS_FILE, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) filename %s",data, filename);
	fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileExists %d", data, fileExists);

	if (fileExists) {
	    groups = g_key_file_get_groups (keyFile, &groupLength);
    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Found Groups=%d", data, groupLength);
	    for(i=0; i<groupLength; i++) {
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Group=%s", data, groups[i]);
	    	if ( !strncmp(QUEEN_BEECON_INSTANCE_BASENAME, groups[i], strlen(QUEEN_BEECON_INSTANCE_BASENAME)) ) {
		    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Instance groups here = [%s] = [%s]", data, QUEEN_BEECON_INSTANCE_BASENAME, groups[i]);
		    	gchar *instanceDir = g_strdup_printf("%s/%s", SYSTEM_INSTANCE_BASE_DIR, groups[i]);
		    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Instance dir = [%s]", data, instanceDir);
	    		GDir *dir = g_dir_open (instanceDir, 0, NULL);
	    		g_free(instanceDir);
	    		if (!dir) {
	    			gchar *confirmMsg = g_strdup_printf("Warning!!!\nQBW Instance [%s]\nis not currently active in the system!\n\nPurge configuration file from obsolete settings?", groups[i]);
	    			GtkWidget *confirmDlg = hildon_note_new_confirmation (GTK_WINDOW(NULL), confirmMsg);
	    			gint retcode = gtk_dialog_run (GTK_DIALOG (confirmDlg));
	    			gtk_widget_destroy (confirmDlg);
	    			g_free(confirmMsg);
					if (retcode == GTK_RESPONSE_OK) {
						qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) No Instance Directory... Time to delete group from Config File [%s]", data, groups[i]);
						if (!g_key_file_remove_group (keyFile, groups[i], &error)) {
							qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Delete group error [%s]", data, error->message);
							hildon_banner_show_informationf (NULL, NULL, "Obsolete QBW instance [%s] Purge ERROR!!!", groups[i]);
						} else {
							hildon_banner_show_informationf (NULL, NULL, "Obsolete QBW instance [%s] Purge OK!!!", groups[i]);
						}
					}
				} else {
					qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Instance Directory successfully open ... Instance is alive and kicking [%s]", data, groups[i]);
					g_dir_close (dir);
       		    }
	    	}
	    }
		g_strfreev(groups);

		fileData = g_key_file_to_data (keyFile, &size, NULL);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) fileData %p", data, fileData);
		iniFile = fopen (filename, "w");
		fputs (fileData, iniFile);
		fclose (iniFile);
		g_free (fileData);
	}
	g_free (filename);
	g_key_file_free (keyFile);
	QUEEN_BEECON_GET_CLASS(data)->housekeeping=FALSE; // Everything is clean and easy!!!
}
