/*
 * This file is a part of Queen Beecon Widget
 * queen-beecon-exec-async: Utility for QBW Asynchronous Command Execution Management
 *
 * http://talk.maemo.org/showthread.php?t=45388
 *
 * Copyright (c) 2010 No!No!No!Yes! (Alessandro Peralma)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <string.h>
#include "queen-beecon-logger.h"
#include "queen-beecon.h"
#include "queen-beecon-progress-animation.h"
#include "queen-beecon-dbus-monitor.h"
#include "queen-beecon-dbus-service.h"
#include "queen-beecon-settings-manager.h"

extern gchar *qbwExecReason[];

// Asynchronous command execution management
void queen_beecon_async_sync_exec_out(QueenBeecon *self, const gchar *whoareyou)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	if (++self->priv->syncOutExec == 2) { // Have process and output synchronized?!?!?!
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) SYNCHED!!!",self);
        qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) execAuxOutput 1 %p (%s)", self, self->priv->execAuxOutput, self->priv->execAuxOutput);
    	guint lenOut = strlen(self->priv->execAuxOutput);
        qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) execAuxOutput 2 %p (%s)", self, self->priv->execAuxOutput, self->priv->execAuxOutput);
    	if(self->priv->execAuxOutput[lenOut-1]=='\n') self->priv->execAuxOutput[lenOut-1]='\0';
        qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) execAuxOutput 3 %p (%s)", self, self->priv->execAuxOutput, self->priv->execAuxOutput);
    	if (self->priv->cmdExitStatus==-1) {
    		gchar *chkOut = g_strdup(self->priv->execAuxOutput);
    		g_free(self->priv->execAuxOutput);
    		self->priv->execAuxOutput = g_strdup_printf("[!?%04x]\n%s", self->priv->cmdStatus, chkOut);//If command has output we issue a newline after Exit Code
    		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) execAuxOutput 5 %p (%s)", self, self->priv->execAuxOutput, self->priv->execAuxOutput);
    		g_free(chkOut);
    	}
    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) EXEC OUTPUT [%s]", self, self->priv->execAuxOutput);
    	gtk_label_set_markup (GTK_LABEL (self->priv->cmdResult_lb), self->priv->execAuxOutput);//AP
    	g_free(self->priv->execAuxOutput);self->priv->execAuxOutput = NULL;

    	queen_beecon_update_content_layout (self);
    	queen_beecon_exec_animation_progress(self, QBW_STOP_PROGRESS_ANIMATION);

    	/* Remove timeout callback */
    	if (self->priv->timeout_id) {g_source_remove( self->priv->timeout_id );self->priv->timeout_id = 0;}

#ifndef EXEC_VIA_MUTEX
    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) RESET execInProgress...",self);
        self->priv->execInProgress = FALSE; // Reset execInProgress flag ready for next execution
        self->priv->hotSpotIndex = 0; //Reset Hot Spot Index to other update policy (only clicks on widget change this index)
#else
        //g_mutex_unlock(self->priv->execInProgressMutex);
#endif
	}
}

void queen_beecon_async_execute_cb_child_watch(GPid pid, gint status, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s => status [0x%04X]", self, G_STRFUNC, status);

	if ((status & 0xff)>0 || ((status >> 8 & 0xff) > 0x70)) {
		self->priv->cmdExitStatus=-1;
		self->priv->cmdStatus=status&0xffff;
	} else {
		self->priv->cmdExitStatus=status >> 8 & 0xff;
	}

    /* Close pid */
    g_spawn_close_pid( pid );

    queen_beecon_async_sync_exec_out(self, "EXEC");

    self->priv->child_watch_id = 0;
}

gboolean queen_beecon_async_execute_cb_out_watch(GIOChannel *channel, GIOCondition  cond, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) %s", self, G_STRFUNC);
	GIOStatus  ret;
    gchar *string = NULL;
    gchar *out;
    gsize  size;

    if( cond == G_IO_HUP ) {
        queen_beecon_async_sync_exec_out(self, "OUTPUT");
        self->priv->io_watch_id = 0;
        return( FALSE );
    }
	ret = g_io_channel_read_line(channel, &string, &size, NULL, NULL);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) ret=%d size=%d string=%p(%s)", self, ret, size, string, string);
	out = g_strconcat (self->priv->execAuxOutput, string, NULL);
	g_free(self->priv->execAuxOutput);
	self->priv->execAuxOutput = out;
	g_free(string);
	return( TRUE );
}

gboolean queen_beecon_async_execute_cb_timeout(QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) %s", self, G_STRFUNC);
    queen_beecon_exec_animation_progress(self, QBW_FRAME_PROGRESS_ANIMATION);
    return( TRUE );
}

GIOChannel *queen_beecon_async_execute_set_up_io_channel (gint fd, GIOCondition cond, GIOFunc func, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) %s", self, G_STRFUNC);
	GIOChannel *ioc;

	/* set up handler for data */
	ioc = g_io_channel_unix_new (fd);
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) ioc=%p",self, ioc);

	/* Set IOChannel encoding to none to make
	 *  it fit for binary data */
	//g_io_channel_set_encoding (ioc, NULL, NULL);
	//g_io_channel_set_buffered (ioc, FALSE);

	/* Tell the io channel to close the file descriptor
	 *  when the io channel gets destroyed */
	g_io_channel_set_close_on_unref (ioc, TRUE);

	/* g_io_add_watch() adds its own reference,
	 *  which will be dropped when the watch source
	 *  is removed from the main loop (which happens
	 *  when we return FALSE from the callback) */
	self->priv->io_watch_id = g_io_add_watch (ioc, cond, func, self);
	g_io_channel_unref (ioc);

	return ioc;
}

void queen_beecon_async_execute(gchar *cmd, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V1, "(%p) %s", self, G_STRFUNC);
    GPid        pid;
    gchar      *argv[] = { "/bin/sh", "-c", cmd, NULL };
    gint        out;
    gboolean    ret;
	GError 		*error=NULL;

	if (self->priv->execAuxOutput) g_free(self->priv->execAuxOutput);
	self->priv->execAuxOutput = g_strdup("");
	self->priv->syncOutExec=0;

	ret = g_spawn_async_with_pipes( NULL, argv, NULL, G_SPAWN_DO_NOT_REAP_CHILD, NULL, NULL, &pid, NULL, &out, NULL, &error );
    if( ! ret ) {
    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) SPAWN ERROR [%s]",self, error->message );
    	self->priv->cmdStatus = error->code;
    	self->priv->cmdExitStatus=-1;
    	g_clear_error (&error);
    	self->priv->syncOutExec++; // Process never exec-ed
        queen_beecon_async_sync_exec_out(self, "SPAWN FAILED");
        return;
    }

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) SPAWN SUCCEEDED [PID=%d]",self, pid);

    /* Now use GIOChannels to monitor stdout */
    self->priv->out_ch = queen_beecon_async_execute_set_up_io_channel(out, G_IO_IN | G_IO_HUP, (GIOFunc)queen_beecon_async_execute_cb_out_watch, self);

    /* Add watch function to catch termination of the process. This function will clean any remnants of process. */
    self->priv->child_watch_id = g_child_watch_add(pid, (GChildWatchFunc)queen_beecon_async_execute_cb_child_watch, self );

	queen_beecon_exec_animation_progress(self, QBW_CUSTOM_PROGRESS_ANIMATION);
    queen_beecon_exec_animation_progress(self, QBW_START_PROGRESS_ANIMATION);

    ///* Install timeout fnction that will move the progress bar */
    if (self->priv->qbwOnSight && self->priv->progressAnimationFrames) self->priv->timeout_id = g_timeout_add( self->priv->progressAnimationTimer*QBW_PROGRESS_ANIMATION_FRAMES_TIMER_STEPS_MSECS, (GSourceFunc)queen_beecon_async_execute_cb_timeout, self);
}

/**
 * g_strreplace
 * @string: a string
 * @search: search string
 * @replacement: replacement string
 *
 * Replaces all occurences of @search in a new copy of @string
 *   by @replacement.
 *
 * Return value: A newly allocated string, where all @search
 *   strings are replaced by the @replacement string.
 *
 * Since: 2.6
 **/
gchar *g_strreplace (const gchar *string, const gchar *search, const gchar *replacement)
{
  gchar *str, **arr;

  g_return_val_if_fail (string != NULL, NULL);
  g_return_val_if_fail (search != NULL, NULL);

  if (replacement == NULL)
    replacement = "";

  arr = g_strsplit (string, search, -1);
  if (arr != NULL && arr[0] != NULL)
    str = g_strjoinv (replacement, arr);
  else
    str = g_strdup (string);

  g_strfreev (arr);

  return str;
}

/**
 * QBW Internal parameter substitution in run scripts
 **/
gchar *queen_beecon_g_command_param_subst(QueenBeecon *self, const gchar *cmd, const gchar *action)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s", self, G_STRFUNC);
	gchar *r = NULL;
	gchar *x = g_strdup(cmd);
	gchar *b = NULL;
	r = g_strreplace(x,"$QBW_ID",                    self->priv->qbwDbusId); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_ON_SIGHT",              self->priv->qbwOnSight?"true":"false"); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_IS_CONNECTED",          self->priv->isConnected?"true":"false"); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_EXEC_REASON",           action); g_free(x); x = r;
	b = g_strdup_printf("%d", self->priv->hotSpotIndex);
	r = g_strreplace(x,"$QBW_HOTSPOT_PRESS",         b); g_free(b); b = NULL; g_free(x); x = r;
	r = g_strreplace(x,"$QBW_CURRENT_RESULTS_TEXT",  gtk_label_get_label(GTK_LABEL (self->priv->cmdResult_lb))); g_free(x); x = r;
    r = g_strreplace(x,"$QBW_HTTP_PROXY",            self->priv->qbwHttpProxy       ? self->priv->qbwHttpProxy       :""); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_HTTPS_PROXY",           self->priv->qbwHttpsProxy      ? self->priv->qbwHttpsProxy      :""); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_IGNORE_HOSTS_PROXY",    self->priv->qbwIgnoreHosts     ? self->priv->qbwIgnoreHosts     :""); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_FTP_PROXY",             self->priv->qbwFtpProxy        ? self->priv->qbwFtpProxy        :""); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_SOCKS_PROXY",           self->priv->qbwSocksProxy      ? self->priv->qbwSocksProxy      :""); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_RTSP_PROXY",            self->priv->qbwRTSPProxy       ? self->priv->qbwRTSPProxy       :""); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_AUTO_CONFIG_PROXY_URL", self->priv->qbwAutoConfigProxy ? self->priv->qbwAutoConfigProxy :""); g_free(x); x = r;
    r = g_strreplace(x,"$QBW_REMEMBER_ME",           self->priv->rememberMe         ? self->priv->rememberMe         :""); g_free(x); x = r;
	r = g_strreplace(x,"$QBW_DBUS_VERBOSE_OUTPUT",   self->priv->dbusVerboseMsg     ? self->priv->dbusVerboseMsg->str:""); g_free(x); x = r;
	return (r);
}
