/*
 * This file is a part of Queen Beecon Widget
 * queen-beecon-dbus-monitor: Utility for QBW DBUS Monitor Management and Operations
 *
 * http://talk.maemo.org/showthread.php?t=45388
 *
 * Copyright (c) 2010 No!No!No!Yes! (Alessandro Peralma)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version. or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <hildon/hildon.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include "queen-beecon-logger.h"
#include "queen-beecon.h"
#include "dbus-print-message.h"
#include "queen-beecon-dbus-monitor.h"

extern gchar *qbwExecReason[];

gchar *qbwDBUSMonitorAction[] = {
	"QBW_INIT_DBUS_MONITOR",
	"QBW_DEINIT_DBUS_MONITOR"
};

gboolean queen_beecon_dbus_monitor_parse_match(DBusMessage *message, gchar *rule)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s MATCH_RULE[%s]", NULL, G_STRFUNC, rule);
	Tokenz tokens[17];
	gboolean inquote=FALSE, inkey=FALSE, invalue=FALSE;
	guint t = 0; // token index

	if (rule==NULL) return FALSE;

	gchar *s = g_strdup(rule);
	gchar *p = s;
	gchar *k = s;
	gchar *v = s;

	if (*p=='\0') {g_free(s);return FALSE;}
	while (*p!=0 && t<16) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) t=%d inquote=%d inkey=%d invalue=%d *p=%c p=%p(%s) k=%p(%s) v=%p(%s)", NULL, t, inquote, inkey, invalue, *p, p, p, k, k, v, v);
		switch (*p) {
		case '\\':
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) t=%d inquote=%d inkey=%d invalue=%d *p=%c", NULL, t, inquote, inkey, invalue, *p);
			if (invalue) {inquote=TRUE;p++;break;}
			p++;
			break;
		case '\'':
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V4, "(%p) t=%d inquote=%d inkey=%d invalue=%d *p=%c", NULL, t, inquote, inkey, invalue, *p);
			if (inquote) {inquote=FALSE;p++;}
			if (inkey) {inkey=FALSE;invalue=TRUE;p++;v=p;break;} // found ' value leader
			if (invalue&&!inquote) {*p='\0';tokens[t].val=g_strdup(v);p++;invalue=FALSE;t++;break;} // found ' value trailer
			p++;
			break;
		case ' ':
		case '\t':
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) t=%d inquote=%d inkey=%d invalue=%d *p=%c", NULL, t, inquote, inkey, invalue, *p);
			inquote = FALSE;
			if (!invalue) *p='\0'; // leading or trailing blanks or tabs surrounding key or value
			p++;
			break;
		case '=':
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) t=%d inquote=%d inkey=%d invalue=%d *p=%c", NULL, t, inquote, inkey, invalue, *p);
			inquote = FALSE;
			if (inkey) {*p='\0';tokens[t].key=g_strdup(k);p++;break;}
			p++;
			break;
		case ',':
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) t=%d inquote=%d inkey=%d invalue=%d *p=%c", NULL, t, inquote, inkey, invalue, *p);
			inquote = FALSE;
			if (!invalue) {p++;k=p;inkey=FALSE;break;}
			p++;
			break;
		default:
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) t=%d inquote=%d inkey=%d invalue=%d *p=%c", NULL, t, inquote, inkey, invalue, *p);
			if (!inkey&&!invalue){k=p;inkey=TRUE;}
			p++;
			break;
		}
	}
	g_free(s);

	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) TOTAL TOKENS = %d", NULL, t);
	gboolean ismatch = TRUE;
	int i;
	for (i=0;i<t;i++) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) TOKENS[%d]=[%s,%s]", NULL, i, tokens[i].key, tokens[i].val);
		if (!g_strcmp0(tokens[i].key,"type")) {
			int message_type = dbus_message_get_type (message);
			if (!g_strcmp0(tokens[i].val, "signal")        && message_type != DBUS_MESSAGE_TYPE_SIGNAL       ) {
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) sender mismatch [%s]", NULL, tokens[i].val);
				ismatch=FALSE;break;
			}
			if (!g_strcmp0(tokens[i].val, "method call")   && message_type != DBUS_MESSAGE_TYPE_METHOD_CALL  ) {
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) method call mismatch [%s]", NULL, tokens[i].val);
				ismatch=FALSE;break;
			}
			if (!g_strcmp0(tokens[i].val, "method return") && message_type != DBUS_MESSAGE_TYPE_METHOD_RETURN) {
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) method return mismatch [%s]", NULL, tokens[i].val);
				ismatch=FALSE;break;
			}
			if (!g_strcmp0(tokens[i].val, "error")         && message_type != DBUS_MESSAGE_TYPE_ERROR        ) {
				qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) error mismatch [%s]", NULL, tokens[i].val);
				ismatch=FALSE;break;
			}
		}
		if (!g_strcmp0(tokens[i].key,"sender"))      if (g_strcmp0(tokens[i].val, dbus_message_get_sender      (message))) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) sender mismatch [%s<=>%s]", NULL, tokens[i].val, dbus_message_get_sender      (message));
			ismatch=FALSE;
			break;
		}
		if (!g_strcmp0(tokens[i].key,"destination")) if (g_strcmp0(tokens[i].val, dbus_message_get_destination (message))) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) destination mismatch [%s<=>%s]", NULL, tokens[i].val, dbus_message_get_destination (message));
			ismatch=FALSE;
			break;
		}
		if (!g_strcmp0(tokens[i].key,"path"))        if (g_strcmp0(tokens[i].val, dbus_message_get_path        (message))) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) path mismatch [%s<=>%s]", NULL, tokens[i].val, dbus_message_get_path        (message));
			ismatch=FALSE;
			break;
		}
		if (!g_strcmp0(tokens[i].key,"interface"))   if (g_strcmp0(tokens[i].val, dbus_message_get_interface   (message))) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) interface mismatch [%s<=>%s]", NULL, tokens[i].val, dbus_message_get_interface   (message));
			ismatch=FALSE;
			break;}
		if (!g_strcmp0(tokens[i].key,"member"))      if (g_strcmp0(tokens[i].val, dbus_message_get_member      (message))) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) member mismatch [%s<=>%s]", NULL, tokens[i].val, dbus_message_get_member      (message));
			ismatch=FALSE;
			break;
		}
	}
	for (i=0;i<t;i++) {g_free(tokens[i].key); g_free(tokens[i].val);}

	return ismatch;
}

DBusHandlerResult queen_beecon_dbus_monitor_filter_function(DBusConnection *connection, DBusMessage *message, QueenBeecon *self)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s dc=%p msg=%p mr=%s Dbus already in progress? %s",self, G_STRFUNC, connection, message, self->priv->updOnDBUSMatchRule, self->priv->execDBUSMatchInProgress?"TRUE":"FALSE");
	if (self->priv->execDBUSMatchInProgress) return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	self->priv->execDBUSMatchInProgress = TRUE;

	self->priv->dbusVerboseMsg = g_string_new ("");
	dbus_print_message (message, FALSE, self->priv->dbusVerboseMsg);
	if (queen_beecon_dbus_monitor_parse_match(message, self->priv->updOnDBUSMatchRule) == TRUE) {
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) MATCH!!!",self);
		queen_beecon_update_content (self, qbwExecReason[QBW_DBUS_MONITOR]);
	}
	g_string_free(self->priv->dbusVerboseMsg,TRUE);self->priv->dbusVerboseMsg = NULL;

	self->priv->execDBUSMatchInProgress = FALSE;
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

gboolean queen_beecon_dbus_monitor_manager(QueenBeecon *self, QbwDBUSMonitorAction action, guint DBUSBus, const gchar *DBUSMatchRule)
{
	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V2, "(%p) %s action=%s",self, G_STRFUNC, qbwDBUSMonitorAction[action]);

	dbus_error_init (&self->priv->dbus_mon_error);

	switch (action) {
	case QBW_INIT_DBUS_MONITOR:
		self->priv->dbus_mon_connection = dbus_bus_get (!DBUSBus?DBUS_BUS_SYSTEM:DBUS_BUS_SESSION, &self->priv->dbus_mon_error);
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) dbus_bus_get Bus:[%d] (0=System 1=Session) dbus=%p",self, DBUSBus, self->priv->dbus_mon_connection);

	    if (dbus_error_is_set (&self->priv->dbus_mon_error)){
			gchar *msg=g_strdup_printf("QBW(%s): DBUS Monitor Error connecting to the daemon bus [%s]", self->priv->qbwDbusId, self->priv->dbus_mon_error.message); GtkWidget *info = hildon_note_new_information (NULL, msg);gtk_dialog_run (GTK_DIALOG (info));gtk_object_destroy (GTK_OBJECT (info));g_free(msg);
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) DBUS Monitor Error connecting to the daemon bus [%s]",self, self->priv->dbus_mon_error.message);
	        dbus_error_free (&self->priv->dbus_mon_error);
	        return FALSE; //FALSE on error
	    }

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Adding Match Rule [%s]",self, DBUSMatchRule);
	    dbus_bus_add_match (self->priv->dbus_mon_connection, DBUSMatchRule, &self->priv->dbus_mon_error);
	    if (dbus_error_is_set (&self->priv->dbus_mon_error)){
			gchar *msg=g_strdup_printf("QBW(%s): DBUS Monitor Error [%s] adding Match Rule [%s]\nMonitor Disabled!", self->priv->qbwDbusId, self->priv->dbus_mon_error.message, DBUSMatchRule); GtkWidget *info = hildon_note_new_information (NULL, msg);gtk_dialog_run (GTK_DIALOG (info));gtk_object_destroy (GTK_OBJECT (info));g_free(msg);
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) DBUS Monitor Error [%s] adding Match Rule [%s]", self, self->priv->dbus_mon_error.message, DBUSMatchRule);
	        dbus_error_free (&self->priv->dbus_mon_error);

	        qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) dbus_connection_unref:[%d] (0=System 1=Session)",self, DBUSBus);
		    //dbus_connection_close (self->priv->dbus_mon_connection);
		    dbus_connection_unref (self->priv->dbus_mon_connection);
			self->priv->dbus_mon_connection = NULL;

		    return FALSE; //FALSE on error
	    }

		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Adding Filter [%s]",self, DBUSMatchRule);
	    if (!dbus_connection_add_filter (self->priv->dbus_mon_connection, (DBusHandleMessageFunction)queen_beecon_dbus_monitor_filter_function, self, NULL)) {
			gchar *msg=g_strdup_printf("QBW(%s): DBUS Monitor Undefined Error adding Filter [%s]\nMonitor Disabled!", self->priv->qbwDbusId, DBUSMatchRule); GtkWidget *info = hildon_note_new_information (NULL, msg);gtk_dialog_run (GTK_DIALOG (info));gtk_object_destroy (GTK_OBJECT (info));g_free(msg);
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Error dbus_connection_add_filter failed", self);

			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Removing Match Rule [%s]",self, DBUSMatchRule);
		    dbus_bus_remove_match (self->priv->dbus_mon_connection, DBUSMatchRule, &self->priv->dbus_mon_error);
		    if (dbus_error_is_set (&self->priv->dbus_mon_error)){
				gchar *msg=g_strdup_printf("QBW(%s): DBUS Monitor Error [%s] removing Match Rule [%s]\nMonitor Disabled!", self->priv->qbwDbusId, self->priv->dbus_mon_error.message, DBUSMatchRule); GtkWidget *info = hildon_note_new_information (NULL, msg);gtk_dialog_run (GTK_DIALOG (info));gtk_object_destroy (GTK_OBJECT (info));g_free(msg);
		    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) DBUS Monitor Error [%s] removing Match Rule [%s]", self, self->priv->dbus_mon_error.message, DBUSMatchRule);
		        dbus_error_free (&self->priv->dbus_mon_error);
		    }

	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) dbus_connection_unref:[%d] (0=System 1=Session)",self, DBUSBus);
		    //dbus_connection_close (self->priv->dbus_mon_connection);
		    dbus_connection_unref (self->priv->dbus_mon_connection);
			self->priv->dbus_mon_connection = NULL;

		    return FALSE; //FALSE on error
	    } else {
	    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Filter Added", self);
	    }
	    dbus_connection_flush(self->priv->dbus_mon_connection);
		break;
	case QBW_DEINIT_DBUS_MONITOR:
		/* unreference dbus connection if present */
		qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) dbus_mon_connection:[%p])",self, self->priv->dbus_mon_connection);
		if (self->priv->dbus_mon_connection) {
			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Removing Filter [%s]",self, DBUSMatchRule);
		    dbus_connection_remove_filter (self->priv->dbus_mon_connection, (DBusHandleMessageFunction)queen_beecon_dbus_monitor_filter_function, self);

			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) Removing Match Rule [%s]",self, DBUSMatchRule);
		    dbus_bus_remove_match (self->priv->dbus_mon_connection, DBUSMatchRule, &self->priv->dbus_mon_error);
		    if (dbus_error_is_set (&self->priv->dbus_mon_error)){
				gchar *msg=g_strdup_printf("QBW(%s): DBUS Monitor Error [%s] removing Match Rule [%s]\nMonitor Disabled!", self->priv->qbwDbusId, self->priv->dbus_mon_error.message, DBUSMatchRule); GtkWidget *info = hildon_note_new_information (NULL, msg);gtk_dialog_run (GTK_DIALOG (info));gtk_object_destroy (GTK_OBJECT (info));g_free(msg);
		    	qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) DBUS Monitor Error [%s] removing Match Rule [%s]", self, self->priv->dbus_mon_error.message, DBUSMatchRule);
		        dbus_error_free (&self->priv->dbus_mon_error);
		    }

		    dbus_connection_flush(self->priv->dbus_mon_connection);

			qbw_logger(QBW_LOGGER_LOG, QBW_LOGGER_V3, "(%p) dbus_connection_unref:[%d] (0=System 1=Session)",self, DBUSBus);
		    dbus_connection_unref (self->priv->dbus_mon_connection);
			self->priv->dbus_mon_connection = NULL;
		}
		break;
	default:
		return FALSE; // Unhandled command
	}
	return TRUE;
}
