/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**
**
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/
#include <qmaemo5kineticscroller.h>
#include <private/qmaemo5kineticscroller_p.h>

#include <QGraphicsView>
#include <QScrollBar>
#include <QAbstractItemView>
#include <QCoreApplication>

#include <QtDebug>

QT_BEGIN_NAMESPACE

// #define KINETIC_SCROLLER_DEBUG

#ifdef KINETIC_SCROLLER_DEBUG
#  define qKSDebug  qDebug
#else
#  define qKSDebug  while (false) qDebug
#endif

bool qt_sendSpontaneousEvent(QObject *receiver, QEvent *event)
{
    return QCoreApplication::sendSpontaneousEvent(receiver, event);
}

/*! \class QMaemo5KineticScroller
    \brief The QMaemo5KineticScroller class allows kinetic scrolling for the given widget.

    The kinetic scroller installs an event filter on the given widget
    (a QAbstractScrollArea in most cases) to handle mouse presses and moves on the widget.
    They will be interpreted as scroll actions in most cases.

    Even though this kinetic scroller has a huge number of settings that you can play with,
    we recommend to leave them all at the default states.
    In case you really want to change them you can try out the kineticscroller example
    in the maemo5 examples directory.

    \sa QWidget
*/

/*!
    Constructs a new kinetic scroller with the given widget \a w as parent and widget to
    scroll.
*/
QMaemo5KineticScroller::QMaemo5KineticScroller(QWidget *w)
    : QObject(w), d_ptr(new QMaemo5KineticScrollerPrivate())
{
    Q_D(QMaemo5KineticScroller);
    d->q_ptr = this;
    d->init(w);
}

/*!
    Constructs a new kinetic scroller with the given widget \a w as parent and widget to
    scroll.
*/
QMaemo5KineticScroller::QMaemo5KineticScroller(QAbstractScrollArea *w)
    : QObject(w), d_ptr(new QMaemo5KineticScrollerPrivate())
{
    Q_D(QMaemo5KineticScroller);
    d->q_ptr = this;
    d->init(w);
}

/*! \internal
*/
QMaemo5KineticScroller::QMaemo5KineticScroller(QMaemo5KineticScrollerPrivate &dd, QWidget *w)
    : QObject(w), d_ptr(&dd)
{
    Q_D(QMaemo5KineticScroller);
    d->q_ptr = this;
    d->init(w);
}

/*!
    Destroys the scroller.
*/
QMaemo5KineticScroller::~QMaemo5KineticScroller()
{
}

/*!
    This filter function changes, blocks or creates mouse events \a e for QObject \a o
    depending on the detected kinetic scrolling state.
    It is mostly an internal implementation.

*/
bool QMaemo5KineticScroller::eventFilter(QObject *o, QEvent *e)
{
    Q_D(QMaemo5KineticScroller);

    bool res = false;

    if (!d->ignoreEvents) {
        if (o == d->filterWidget) {
            switch (e->type()) {
            case QEvent::MouseButtonPress:
                res = d->handleMousePress(static_cast<QMouseEvent *>(e));
                break;
            case QEvent::MouseMove:
                res = d->handleMouseMove(static_cast<QMouseEvent *>(e));
                break;
            case QEvent::MouseButtonRelease:
                res = d->handleMouseRelease(static_cast<QMouseEvent *>(e));
                break;
            case QEvent::ChildAdded:
            case QEvent::ChildRemoved:
                d->handleChildEvent(static_cast<QChildEvent *>(e));
                break;
            default:
                break;
            }
        }
    }
    return res ? true : QObject::eventFilter(o, e);
}

/*! \internal
    The timer event will be triggered by the scroll and idle timer.
*/
void QMaemo5KineticScroller::timerEvent(QTimerEvent *te)
{
    Q_D(QMaemo5KineticScroller);

    if (te->timerId() == d->idleTimerId)
        d->handleIdleTimer();
    else if (te->timerId() == d->scrollTimerId)
        d->handleScrollTimer();
}

QMaemo5KineticScrollerPrivate::QMaemo5KineticScrollerPrivate()
    : mode(QMaemo5KineticScroller::AutoMode),
    lastType(QEvent::User), buttonPressed(false), moved(false), lastIn(true),
    firstDrag(true), centerOnChildFocusPending(false), lowFrictionMode(false),
    scrollTo(-1, -1), bounceSteps(3), maxOverShoot(150, 150), vmaxOverShoot(130),
    overShootDist(0,0), overShooting(0),
    minVelocity(10), maxVelocity(3500), fastVelocityFactor(0.01), deceleration(0.85),
    scrollsPerSecond(20), panningThreshold(25), directionErrorMargin(10), force(50),
    dragInertia(0.85), scrollTime(1000),
    childWidget(0),
    idleTimerId(0), scrollTimerId(0), widget(0), filterWidget(0), ignoreEvents(false)
{ }

QMaemo5KineticScrollerPrivate::~QMaemo5KineticScrollerPrivate()
{ }

void QMaemo5KineticScrollerPrivate::init(QWidget *w)
{
    Q_Q(QMaemo5KineticScroller);

    widget = w;

    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(w)) {
        if (QAbstractItemView *itemview = qobject_cast<QAbstractItemView *>(area)) {
            itemview->setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);
            itemview->setVerticalScrollMode(QAbstractItemView::ScrollPerPixel);
        }
        filterWidget = area->viewport();
    } else {
        filterWidget = w;
    }
    if (filterWidget)
        filterWidget->installEventFilter(q);
}

void QMaemo5KineticScrollerPrivate::sendEvent(QWidget *w, QEvent *e)
{
    ignoreEvents = true;
    //QCoreApplication::sendEvent(w, e);
    qt_sendSpontaneousEvent(w, e);
    ignoreEvents = false;
}

/*!
    Returns the rect with the lowest and highest valid position values.

    \sa scrollTo()
*/
QRect QMaemo5KineticScroller::positionRange() const
{
    Q_D(const QMaemo5KineticScroller);

    QRect r;
    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(d->widget)) {
        if (QAbstractSlider *s = area->horizontalScrollBar()) {
            r.setLeft(s->minimum());
            r.setRight(s->maximum());
        }
        if (QAbstractSlider *s = area->verticalScrollBar()) {
            r.setTop(s->minimum());
            r.setBottom(s->maximum());
        }
    }
    return r;
}

/*!
    \brief Returns the current position of the scrolling.
    Note that overshooting is not considered a real scrolling so the position might be (0,0)
    even if the user is currently dragging the widget over the "normal" positionRange.

    \sa positionRange()
*/
QPoint QMaemo5KineticScroller::position() const
{
    Q_D(const QMaemo5KineticScroller);

    QPoint p;
    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(d->widget)) {
        if (QAbstractSlider *s = area->horizontalScrollBar())
            p.setX(s->value());
        if (QAbstractSlider *s = area->verticalScrollBar())
            p.setY(s->value());
    }
    return p;
}

/*! \brief set the scroll position of the viewport.
    This function sets the position of the viewport to \a pos. If the given "pos" it outside the
    normal scrolling range it will move the viewport and return the additional movement in the
    overShootDist variable.

*/
void QMaemo5KineticScroller::setPosition(const QPoint &pos)
{
    Q_D(QMaemo5KineticScroller);

    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(d->widget)) {
        int overShootX = 0;
        int overShootY = 0;

        if (QAbstractSlider *s = area->horizontalScrollBar()) {
            s->setValue(pos.x());
            if (s->minimum() != s->maximum()) // can scroll
                overShootX = s->value()-pos.x();
        }
        if (QAbstractSlider *s = area->verticalScrollBar()) {
            s->setValue(pos.y());
            if (s->minimum() != s->maximum()) // can scroll
                overShootY = s->value()-pos.y();
        }

        qKSDebug() << "setPosition to " << pos << " viewportpos: " << area->viewport()->pos() << " range " << positionRange();

        QPoint oldOverShootDist = d->overShootDist;
        d->overShootDist.setX( qBound(-d->maxOverShoot.x(), overShootX, d->maxOverShoot.x()));
        d->overShootDist.setY( qBound(-d->maxOverShoot.y(), overShootY, d->maxOverShoot.y()));
        area->viewport()->move( area->viewport()->pos() + d->overShootDist - oldOverShootDist);
    }
}

void QMaemo5KineticScrollerPrivate::handleChildEvent(QChildEvent *e)
{
    if (e->child()->isWidgetType())
        static_cast<QWidget *>(e->child())->setAttribute(Qt::WA_TransparentForMouseEvents, e->added());
}

void QMaemo5KineticScrollerPrivate::handleScrollTimer()
{
    Q_Q(QMaemo5KineticScroller);

    qKSDebug("handleScrollTimer: %d/%d", motion.x(), motion.y());

    if (!motion.isNull())
        q->setPosition(q->position() - overShootDist - motion);
    q->killTimer(scrollTimerId);
    scrollTimerId = 0;
}

void QMaemo5KineticScrollerPrivate::handleIdleTimer()
{
    Q_Q(QMaemo5KineticScroller);

    if (mode == QMaemo5KineticScroller::PushMode) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
        return;
    }
    qKSDebug() << "idle timer - velocity: " << velocity << " overShoot: " << overShootDist;

    q->setPosition(q->position() - overShootDist - velocity.toPoint());

    if (!buttonPressed) {

        // Decelerate gradually when pointer is raised
        if (overShootDist.isNull()) {
            overShooting=0;

            // in case we move to a specific point do not decelerate when arriving
            if (scrollTo.x() != -1 || scrollTo.y() != -1) {

                // -- check if target was reached
                QPoint pos = q->position();
                QPointF  dist = QPointF(scrollTo - pos);

                qKSDebug() << "handleIdleTimer dist:" << dist << " scrollTo:" << scrollTo;

                // -- break if target was reached
                if ( (velocity.x() < 0.0 && dist.x() <= 0.0) ||
                        (velocity.x() > 0.0 && dist.x() >= 0.0) )
                    velocity.setX(0.0);

                if ( (velocity.y() < 0.0 && dist.y() <= 0.0) ||
                        (velocity.y() > 0.0 && dist.y() >= 0.0) )
                    velocity.setY(0.0);

                // -- break if we reached the borders
                /*
                QRect range = q->positionRange();
                if ( (range.left() > pos.x()-velocity.x() && velocity.x() > 0.0) ||
                        (range.right() < pos.x()-velocity.x() && velocity.x() < 0.0) )
                    velocity.setX(0.0);

                if ( (range.top() > pos.y()-velocity.y() && velocity.y() > 0.0) ||
                        (range.bottom() < pos.y()-velocity.y() && velocity.y() < 0.0) )
                    velocity.setY(0.0);
                    */


                if (velocity.x() == 0.0 && velocity.y() == 0.0) {
                    q->killTimer(idleTimerId);
                    idleTimerId = 0;
                    return;
                }

                // -- don't get too slow if target was not yet reached
                if (qAbs(velocity.x()) >= qreal(1.5))
                    velocity.rx() *= deceleration;
                if (qAbs(velocity.y()) >= qreal(1.5))
                    velocity.ry() *= deceleration;

            } else {
                if (!lowFrictionMode || (qAbs(velocity.x()) < qreal(0.8) * maxVelocity))
                    velocity.rx() *= deceleration;
                if (!lowFrictionMode || (qAbs(velocity.y()) < qreal(0.8) * maxVelocity))
                    velocity.ry() *= deceleration;

                if ((qAbs(velocity.x()) < qreal(1.0)) && (qAbs(velocity.y()) < qreal(1.0))) {
                    velocity = QPointF(0, 0);
                    q->killTimer(idleTimerId);
                    idleTimerId = 0;
                }
            }

        } else { // overShootDist != 0
            overShooting++;
            scrollTo = QPoint(-1, -1);

            /* When the overshoot has started we continue for
             * PROP_BOUNCE_STEPS more steps into the overshoot before we
             * reverse direction. The deceleration factor is calculated
             * based on the percentage distance from the first item with
             * each iteration, therefore always returning us to the
             * top/bottom most element
             */
            if (overShooting < bounceSteps) {
                velocity.setX( overShootDist.x() / maxOverShoot.x() * velocity.x() );
                velocity.setY( overShootDist.y() / maxOverShoot.y() * velocity.y() );
            } else {
                velocity.setX( -overShootDist.x() * 0.8 );
                velocity.setY( -overShootDist.y() * 0.8 );

                // ensure a minimum speed when scrolling back or else we might never return
                if (velocity.x() > -1.5 && velocity.x() < 0.0)
                    velocity.setX(-1.5);
                if (velocity.x() <  1.5 && velocity.x() > 0.0)
                    velocity.setX( 1.5);
                if (velocity.y() > -1.5 && velocity.y() < 0.0)
                    velocity.setY(-1.5);
                if (velocity.y() <  1.5 && velocity.y() > 0.0)
                    velocity.setY( 1.5);
            }

            velocity.setX( qBound((qreal)-vmaxOverShoot, velocity.x(), (qreal)vmaxOverShoot));
            velocity.setY( qBound((qreal)-vmaxOverShoot, velocity.y(), (qreal)vmaxOverShoot));
        } // overshoot

    } else if (mode == QMaemo5KineticScroller::AutoMode) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
    }
}


static QWidget *mouseTransparentChildAtGlobalPos(QWidget *parent, const QPoint &gp)
{
    foreach (QWidget *w, parent->findChildren<QWidget *>()) {
        if (w && !w->isWindow() && !w->isHidden() && (w->rect().contains(w->mapFromGlobal(gp)))) {
            if (QWidget *t = mouseTransparentChildAtGlobalPos(w, gp))
                return t;
            else
                return w;
        }
    }
    return 0;
}

bool QMaemo5KineticScrollerPrivate::handleMousePress(QMouseEvent *e)
{
    Q_Q(QMaemo5KineticScroller);
    qKSDebug("MP: start");
    if (e->button() != Qt::LeftButton)
        return false;

    QRect posRange = q->positionRange();
    bool canScrollX = (posRange.width() > 1);
    bool canScrollY = (posRange.height() > 1);
    if (!canScrollX && !canScrollY)
        return false;

    lastTime.start();
    lastPressTime.start();
    lastType = e->type();

    scrollTo = QPoint(-1, -1);

    /* could not happen?
    if (buttonPressed && childWidget) {
        // send leave event to child...
        if (childWidget) {
            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);
        }
    }
    */

    pos = e->globalPos();
    ipos = pos;

    childWidget = 0;

    // don't allow a click if we're still moving fast
    if ((qAbs(velocity.x()) <= (maxVelocity * fastVelocityFactor)) &&
        (qAbs(velocity.y()) <= (maxVelocity * fastVelocityFactor))) {
        if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(widget))
            childWidget = area->viewport();
        else
            childWidget = widget;

        if (QWidget *w = mouseTransparentChildAtGlobalPos(childWidget, e->globalPos()))
            childWidget = w;
    }
    buttonPressed = true;

    // stop scrolling on mouse press (so you can flick, then hold to stop)
    oldVelocity = velocity;
    velocity = QPointF(0, 0);

    if (idleTimerId) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
    }

    if (childWidget) {
        lastIn = true;

        // hack to remove the current selection as soon as we start to scroll
        // here we just remember the selection
        if (QAbstractItemView *view = qobject_cast<QAbstractItemView *>(childWidget->parentWidget())) {
            qKSDebug() << "remembering selection in item view";
            oldSelection = view->selectionModel()->selection();
        }

        if (childWidget) {
            QEvent ee(QEvent::Enter);
            sendEvent(childWidget, &ee);

            QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                           e->globalPos(), e->button(), e->buttons(), e->modifiers());
            sendEvent(childWidget, &me);
        }
    }
    qKSDebug("MP: end");
    return true;
}

bool QMaemo5KineticScrollerPrivate::handleMouseMove(QMouseEvent *e)
{
    qKSDebug() << "MM: pos: " << e->globalPos() << " - time: " << QTime::currentTime().msec();
    if (!buttonPressed)
        return false;
    if (moved && !lastTime.elapsed())
        return true;
    if (lastType == QEvent::MouseButtonPress)
        firstDrag = true;

    QPoint delta = e->globalPos() - pos;
    qKSDebug() << "Delta: " << delta << " Moved: " << moved;
    if (childWidget)
        qKSDebug() << "transparent?: " << childWidget->testAttribute(Qt::WA_TransparentForMouseEvents);

    if (!moved) {
        checkMove(e, delta);
        qKSDebug() << "After Check: " << delta << " Moved: " << moved;
    }
    if (moved) {
        if (childWidget) {
            // send leave and mouse up
            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);

            QMouseEvent me(e->type(), QPoint(-INT_MAX, -INT_MAX),
                           e->button(), e->buttons(), e->modifiers());
            sendEvent(childWidget, &me);

            // hack to remove the current selection as soon as we start to scroll
            // GtkTreeView apparently reacts on the Leave event, but Qt does not
            if (QAbstractItemView *view = qobject_cast<QAbstractItemView *>(childWidget->parentWidget())) {
                qKSDebug() << "Clearing selection in item view";
                view->selectionModel()->select(oldSelection, QItemSelectionModel::ClearAndSelect);
            }

            childWidget = 0;
        }
        handleMove(e, delta);
        qKSDebug() << "After Handle: " << delta << " Moved: " << moved;
    } else if (childWidget) {
        // send leave event to child according to lastIn and set lastIn
        if (childWidget) {
            bool in = childWidget->geometry().contains(childWidget->mapFromGlobal(e->globalPos()));

            if (in != lastIn) {
                QEvent ele(in ? QEvent::Enter : QEvent::Leave);
                sendEvent(childWidget, &ele);
                lastIn = in;
            }
        }
    }
    lastTime.restart();
    lastType = e->type();

    if (childWidget) {
        // send mouse move event to child
        QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                e->globalPos(), e->button(), e->buttons(), e->modifiers());
        sendEvent(childWidget, &me);
    }
    qKSDebug("MM: end");
    return true;
}

bool QMaemo5KineticScrollerPrivate::handleMouseRelease(QMouseEvent *e)
{
    qKSDebug() << "MR: pos: " << e->globalPos() << " - time: " << QTime::currentTime().msec();
    Q_Q(QMaemo5KineticScroller);

    if (!buttonPressed || e->button() != Qt::LeftButton)
        return false;

    // if last event was a motion-notify we have to check the
    // movement and launch the animation
    if (lastType == QEvent::MouseMove) {
        if (moved) {
            // move all the way to the last position now
            if (scrollTimerId) {
                q->killTimer(scrollTimerId);
                scrollTimerId = 0;
                q->setPosition(q->position() - overShootDist - motion);
                motion = QPoint(0, 0);
            }
        }
    }
    // If overshoot has been initiated with a finger down,
    // on release set max speed
    if (overShootDist.x()) {
        overShooting = bounceSteps; // Hack to stop a bounce in the finger down case
        velocity.setX(overShootDist.x() * qreal(0.9));

    }
    if (overShootDist.y()) {
        overShooting = bounceSteps; // Hack to stop a bounce in the finger down case
        velocity.setY(overShootDist.y() * qreal(0.9));
    }
    buttonPressed = false;

    bool forceFast = true;

    // if widget was moving fast in the panning, increase speed even more
    if ((lastPressTime.elapsed() < FastClick) &&
        ((qAbs(oldVelocity.x()) > minVelocity) ||
         (qAbs(oldVelocity.y()) > minVelocity)) &&
        ((qAbs(oldVelocity.x()) > MinimumAccelerationThreshold) ||
         (qAbs(oldVelocity.y()) > MinimumAccelerationThreshold))) {

        qKSDebug() << "FAST CLICK - using oldVelocity " << oldVelocity;
        int signX = 0, signY = 0;
        if (velocity.x())
            signX = (velocity.x() > 0) == (oldVelocity.x() > 0) ? 1 : -1;
        if (velocity.y())
            signY = (velocity.y() > 0) == (oldVelocity.y() > 0) ? 1 : -1;

        velocity.setX(signX * (oldVelocity.x() + (oldVelocity.x() > 0 ? accelerationVelocity.x() : -accelerationVelocity.x())));
        velocity.setY(signY * (oldVelocity.y() + (oldVelocity.y() > 0 ? accelerationVelocity.y() : -accelerationVelocity.y())));
        forceFast = false;
    }

    if ((qAbs(velocity.x()) >= minVelocity) ||
        (qAbs(velocity.y()) >= minVelocity)) {

        qKSDebug() << "over min velocity: " << velocity;
        // we have to move because we are in overshooting position
        if (!moved) {
            // (opt) emit panningStarted()
        }
        // (must) enable scrollbars

        if (forceFast) {
            if ((qAbs(velocity.x()) > MaximumVelocityThreshold) &&
                (accelerationVelocity.x() > MaximumVelocityThreshold)) {
                velocity.setX(velocity.x() > 0 ? accelerationVelocity.x() : -accelerationVelocity.x());
            }
            if ((qAbs(velocity.y()) > MaximumVelocityThreshold) &&
                (accelerationVelocity.y() > MaximumVelocityThreshold)) {
                velocity.setY(velocity.y() > 0 ? accelerationVelocity.y() : -accelerationVelocity.y());
            }
            qKSDebug() << "Force fast is on - velocity: " << velocity;

        }
        if (!idleTimerId)
            idleTimerId = q->startTimer(1000 / scrollsPerSecond);
    } else {
        if (centerOnChildFocusPending) {
            centerOnChildFocus();
        }

        if (moved) {
            // (opt) emit panningFinished()
        }
    }
    centerOnChildFocusPending = false;
    lastTime.restart();
    lastType = e->type();

    if (!childWidget) {
        bool wasMoved = moved;
        moved = false;
        return wasMoved; // when we did move we use the mouse release ourself

    } else {
        if (!childWidget->rect().contains(childWidget->mapFromGlobal(e->globalPos()))) {
            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);
        }

        QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                e->button(), e->buttons(), e->modifiers());
        sendEvent(childWidget, &me);

        childWidget = 0;
        moved = false;
        return true;
    }
}

void QMaemo5KineticScrollerPrivate::checkMove(QMouseEvent *me, QPoint &delta)
{
    Q_Q(QMaemo5KineticScroller);

    if (firstDrag && !moved && ((qAbs(delta.x()) > panningThreshold) || (qAbs(delta.y()) > panningThreshold))) {
        moved = true;
        delta = QPoint(0, 0);

        if (firstDrag) {
            int deltaXtoY = qAbs(ipos.x() - me->globalPos().x()) - qAbs(ipos.y() - me->globalPos().y());

            qKSDebug() << "First Drag with delta " << delta << ", greater than " << panningThreshold << " -- deltaXtoY: " << deltaXtoY;

            QRect posRange = q->positionRange();
            bool canScrollX = (posRange.width() > 1);
            bool canScrollY = (posRange.height() > 1);

            if (deltaXtoY < 0) {
                if (!canScrollY && (!canScrollX || (-deltaXtoY >= directionErrorMargin)))
                    moved = false;
            } else {
                if (!canScrollX && (!canScrollY || (deltaXtoY >= directionErrorMargin)))
                    moved = false;
            }
        }
        firstDrag = false;

        if (mode == QMaemo5KineticScroller::AccelerationMode) {
            if (!idleTimerId)
                idleTimerId = q->startTimer(1000 / scrollsPerSecond);
        }
    }
}

void QMaemo5KineticScrollerPrivate::handleMove(QMouseEvent *me, QPoint &delta)
{
    Q_Q(QMaemo5KineticScroller);

    switch (mode) {
    case QMaemo5KineticScroller::PushMode:
        // Scroll by the amount of pixels the cursor has moved
        // since the last motion event.
        scrollUpdate(delta);
        pos = me->globalPos();
        break;

    case QMaemo5KineticScroller::AccelerationMode:
        // Set acceleration relative to the initial click
        // epos = me->globalPos(); //TODO: what the heck is epos?
        velocity.setX(qreal(delta.x() < 0 ? -1 : 1) * ((qreal(qAbs(delta.x())) / qreal(widget->width()) * (maxVelocity - minVelocity)) + minVelocity));
        velocity.setY(qreal(delta.y() < 0 ? -1 : 1) * ((qreal(qAbs(delta.y())) / qreal(widget->height()) * (maxVelocity - minVelocity)) + minVelocity));
        break;

    case QMaemo5KineticScroller::AutoMode:
        QPointF newVelocity = calculateVelocity(me->globalPos() - pos, lastTime.elapsed());
        QRect posRange = q->positionRange();

        if (!posRange.width()) {
            delta.setX(0);
            newVelocity.setX(0);
        }
        if (!posRange.height()) {
            delta.setY(0);
            newVelocity.setY(0);
        }
        velocity = newVelocity;

        scrollUpdate(delta);

        if (posRange.width())
            pos.setX(me->globalPos().x());
        if (posRange.height())
            pos.setY(me->globalPos().y());
        break;
    }
}

QPointF QMaemo5KineticScrollerPrivate::calculateVelocity(const QPointF &dPixel, int dTime)
{
    qKSDebug() << "calculateVelocity(dP = " << dPixel << ", dT = " << dTime << ") -- velocity: " << velocity;

    QPointF newv = velocity;

    // fast than 15 pix / ms seems bogus (that's a screen height in 1/30 second)
    if ((dPixel / qreal(dTime)).manhattanLength() < 25) {
        QPointF rawv = dPixel / qreal(dTime) * qreal(force);
        newv = newv * (qreal(1) - dragInertia) + rawv * dragInertia;
    }

    qKSDebug() << " --> " << newv << " (before clamping)";

    newv.setX(dPixel.x() ? qBound(-maxVelocity, newv.x(), maxVelocity) : velocity.x());
    newv.setY(dPixel.y() ? qBound(-maxVelocity, newv.y(), maxVelocity) : velocity.y());
    return newv;
}

void QMaemo5KineticScrollerPrivate::scrollUpdate(const QPoint &delta)
{
    Q_Q(QMaemo5KineticScroller);

    if (scrollTimerId) {
        motion += delta;
    } else {
        // we do not delay the first event but the next ones
        q->setPosition(q->position() - overShootDist - delta);
        motion = QPoint(0, 0);
        scrollTimerId = q->startTimer(1000 / MotionEventsPerSecond);
    }
}


void QMaemo5KineticScrollerPrivate::centerOnChildFocus()
{
    //TODO:
}


/*!
    Returns the scrolling mode.

    \sa QMaemo5KineticScroller::setMode(), QMaemo5KineticScroller::Mode
*/
QMaemo5KineticScroller::Mode QMaemo5KineticScroller::mode() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->mode;
}

/*!
    Set the scrolling mode to \a mode

    \sa QMaemo5KineticScroller::mode(), QMaemo5KineticScroller::Mode
*/

void QMaemo5KineticScroller::setMode(Mode mode)
{
    Q_D(QMaemo5KineticScroller);
    d->mode = mode;
}


/*!
    Returns the value of the low-friction-mode.
    Low friction means that the scrolling will not be slowed down by the decelerationFactor.

    \sa decelerationFactor(), setLowFrictionEnabled()
*/
bool QMaemo5KineticScroller::isLowFrictionEnabled() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->lowFrictionMode;
}

/*!
    Sets the value of the low-friction-mode to \a b

    \sa decelerationFactor(), isLowFrictionEnabled()
*/
void QMaemo5KineticScroller::setLowFrictionEnabled(bool b)
{
    Q_D(QMaemo5KineticScroller);
    d->lowFrictionMode = b;
}


/*!
    Returns the value of the dragInertia.
    Percentage of the calculated speed in each moment we are are going to use to calculate
    the launch speed.

    \sa setDragInertia()
*/
qreal QMaemo5KineticScroller::dragInertia() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->dragInertia;
}

/*!
    Sets the value of the drag intertia to \a inertia

    \sa dragInertia()
*/
void QMaemo5KineticScroller::setDragInertia(qreal inertia)
{
    Q_D(QMaemo5KineticScroller);
    d->dragInertia = inertia;
}


/*!
    Returns the directionErrorMargin.
    This value influences whether a scroll is realized as such if the mouse is dragged
    diagonal.

    \sa setDirectionErrorMargin()
*/
int QMaemo5KineticScroller::directionErrorMargin() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->directionErrorMargin;
}

/*!
    Sets the directionErrorMargin to \a errorMargin.

    \sa directionErrorMargin()
*/
void QMaemo5KineticScroller::setDirectionErrorMargin(int errorMargin)
{
    Q_D(QMaemo5KineticScroller);
    d->directionErrorMargin = errorMargin;
}


/*!
    Returns the panningThreshold value.
    The amount in pixels the mouse must move until scrolling is started.

    \sa setPanningThreshold()
*/
int QMaemo5KineticScroller::panningThreshold() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->panningThreshold;
}

/*!
    Sets the panning threshold to \a threshold

    \sa panningThreshold()
*/
void QMaemo5KineticScroller::setPanningThreshold(int threshold)
{
    Q_D(QMaemo5KineticScroller);
    d->panningThreshold = threshold;
}

/*!
    Returns the deceleration factor which is the percentage of scrolling velocity
    remaining after every scroll step.

    \sa setDecelerationFactor()
*/
qreal QMaemo5KineticScroller::decelerationFactor() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->deceleration;
}

/*!
    Sets the deceleration factor to \a f.

    \sa decelerationFactor()
*/
void QMaemo5KineticScroller::setDecelerationFactor(qreal f)
{
    Q_D(QMaemo5KineticScroller);
    d->deceleration = f;
}


/*!
    Returns the fast velocity factor which is the factor that
    determines if the scrolling speed is too fast for a mouse click.

    If the current velocity is greater then maximumVelocity()*fastVelocityFactor()
    then the mouse click will just stop the scrolling.
    If the velocity is less then a click will stop the scrolling and transmitted to
    the scroll area.

    \sa setFastVelocityFactor()
*/
qreal QMaemo5KineticScroller::fastVelocityFactor() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->fastVelocityFactor;
}

/*!
    Sets the fast velocity factor to \a f.

    \sa fastVelocityFactor()
*/
void QMaemo5KineticScroller::setFastVelocityFactor(qreal f)
{
    Q_D(QMaemo5KineticScroller);
    d->fastVelocityFactor = f;
}

/*!
    Returns the minimumVelocity.
    The minimum velocity is the slowest speed in the acceleration mode.

    \sa setMinimumVelocity()
*/
qreal QMaemo5KineticScroller::minimumVelocity() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->minVelocity;
}

/*!
    Sets the miminum velocity to \a v.

    \sa minimumVelocity()
*/
void QMaemo5KineticScroller::setMinimumVelocity(qreal v)
{
    Q_D(QMaemo5KineticScroller);
    d->minVelocity = v;
}

/*!
    Returns the maximumVelocity.
    The maximum velocity is the slowest speed in the acceleration mode.

    \sa setMaximumVelocity()
*/
qreal QMaemo5KineticScroller::maximumVelocity() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->maxVelocity;
}

/*!
    Sets the maximum velocity to \a v.

    \sa maximumVelocity()
*/
void QMaemo5KineticScroller::setMaximumVelocity(qreal v)
{
    Q_D(QMaemo5KineticScroller);
    d->maxVelocity = v;
}


/*!
    Start scrolling to the position \a pos.
    The area will scroll in scrollTime milliseconds to the given position.
    The default for the private scrollTime is 1000 milliseconds.

    The speed will be calculated so that the scrolling reaches the given positon after
    this time. However the speed a the end position is not guaranteed to be zero.

    If a position outside the positionRange is given the area will overshoot and not end
    up at the invalid position.

    \sa positionRange()
*/
void QMaemo5KineticScroller::scrollTo(const QPoint &pos)
{
    Q_D(QMaemo5KineticScroller);

    if (pos == position())
        return;
    if (d->buttonPressed)
        return;

    // --- calculate the initial velocity to get to the point
    // -- calc the number of steps we have:
    int steps = d->scrollTime * d->scrollsPerSecond / 1000;

    // -- calc the distance we will move with a starting velocity of 1.0
    float dist = 0.0;
    float curVel = 1.0;
    for (int i = 0; i<steps; i++) {
        dist += curVel;
        curVel *= d->deceleration;
    }

    // --- start the scrolling
    d->scrollTo = pos;
    d->velocity = - QPointF(pos-position()) / dist;
    d->mode = QMaemo5KineticScroller::AutoMode;

    qKSDebug() << "QMaemo5KineticScroller::scrollTo new pos:" << pos << " velocity:"<<d->velocity;

    if (!d->idleTimerId)
        d->idleTimerId = startTimer(1000 / d->scrollsPerSecond);
}

/*!
    \enum QMaemo5KineticScroller::Mode
    \since 4.6

    This enum contains the different modes for the QMaemo5KineticScroller.

    \value AutoMode The mode will allow pushing and AccelerationMode.

    \value PushMode The area will be scrolled as long as the user drags it around with pressed mouse button.

    \value AccelerationMode The area will continue scrolling after the user releases the mouse button.
*/



QT_END_NAMESPACE
