/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the examples of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**
**
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QtGui>
#include <QtDBus>
#include <QtDebug>
#include <QMaemo5InformationBox>

#include <mce/mode-names.h>
#include <mce/dbus-names.h>

#include "window.h"

class RotatedLabel : public QWidget {
public:
    RotatedLabel(const QString &str, int rotation)
        : QWidget(), m_rotation(rotation), m_text(str)
    {
        QFontMetrics fm = fontMetrics();
        int w = fm.width(m_text);
        setMinimumSize(w + 8, w + 8);
    }

protected:
    void paintEvent(QPaintEvent *)
    {
        QPainter p(this);
        p.translate(rect().center());
        p.rotate(m_rotation);
        p.drawText(rect().translated(-rect().center()), Qt::AlignCenter, m_text);
    }

private:
    int m_rotation;
    QString m_text;
};

class ProgressBarAction : public QWidgetAction {
public:
    ProgressBarAction(QObject *parent)
        : QWidgetAction(parent)
    {
        setText(tr("Progress"));
    }

    QWidget * createWidget(QWidget *parent)
    {
        return new QProgressBar(parent);
    }
};


Window::Window(QWidget *parent, bool stacked)
    : QMainWindow(parent), m_level(-1), m_dialog(0)
{
#ifdef QT_ARCH_ARM
    // we should really test for X11 server == Xephyr here,
    // since Xephyr forgets to rotate the mouse coordinates.
    setAttribute(Qt::WA_Maemo5ForcePortraitOrientation);
#endif

    if (stacked) {
        m_level = 0;
        setAttribute(Qt::WA_Maemo5StackedWindow);
        if (Window *w = ::qobject_cast<Window *>(parent)) {
            m_level = w->m_level+1;
        }
    }

    menuBar()->addAction(tr("Open new window"), this, SLOT(openNewWindow()));
    menuBar()->addAction(tr("Close this window"), this, SLOT(close()));
    menuBar()->addAction(tr("Show a MessageBox"), this, SLOT(showMessageBox()));
    menuBar()->addAction(tr("Show a Banner"), this, SLOT(showBanner()));
    menuBar()->addAction(tr("Show a Note"), this, SLOT(showNote()));
    menuBar()->addAction(tr("Show a Expose"), this, SLOT(showExpose()));
    menuBar()->addAction(tr("Show a Complex Note"), this, SLOT(showComplexNote()));

    // add a checkbox action
    QAction *checkAction = new QAction(tr("Busy"), this);
    connect(checkAction, SIGNAL(triggered(bool)), this, SLOT(beBusy(bool)));
    checkAction->setCheckable(true);
    menuBar()->addAction(checkAction);

    // add a widget action
    menuBar()->addAction(new ProgressBarAction(this));

    // add some radio/exclusive actions (which will be converted to filter actions in the application menu)
    // The Maemo 5 UI specification limits this to up to four actions in up to one group
    QAction* filterAction1 = new QAction(tr("Filter 1"), menuBar());
    filterAction1->setCheckable(true);
    QAction* filterAction2 = new QAction(tr("Filter 2"), menuBar());
    filterAction2->setCheckable(true);
    QAction* filterAction3 = new QAction(tr("Filter 3"), menuBar());
    filterAction3->setCheckable(true);
    filterAction3->setChecked(true);

    QActionGroup* filterGroup = new QActionGroup(menuBar());
    filterGroup->addAction(filterAction1);
    filterGroup->addAction(filterAction2);
    filterGroup->addAction(filterAction3);

    menuBar()->addAction(filterAction1);
    menuBar()->addAction(filterAction2);
    menuBar()->addAction(filterAction3);

    QAction *fullscreen = new QAction(QIcon::fromTheme("general_fullsize"), QString(), this);
    fullscreen->setCheckable(true);
    QActionGroup *orientation = new QActionGroup(this);
    orientation->setExclusive(true);

    m_portrait = new QAction(QIcon::fromTheme("general_portrait", QIcon(QLatin1String(":/images/portrait"))), QString(), orientation);
    m_portrait->setCheckable(true);
    m_portrait->setChecked(true);
    m_landscape = new QAction(QIcon::fromTheme("general_landscape", QIcon(QLatin1String(":/images/landscape"))), QString(), orientation);
    m_landscape->setCheckable(true);

    connect(fullscreen, SIGNAL(toggled(bool)), this, SLOT(toggleFullScreen(bool)));
    connect(m_landscape, SIGNAL(toggled(bool)), this, SLOT(toggleLandscape(bool)));

    QToolBar *tb = new QToolBar(this);
    tb->addAction(fullscreen);
    tb->addAction(m_portrait);
    tb->addAction(m_landscape);
    addToolBar(tb);

    QWidget *central = new QWidget(this);
    QVBoxLayout *lay = new QVBoxLayout(central);

    lay->addWidget(new QLabel(tr("<qt>This is a Maemo5 Stacked Window. Current stacking depth is %1</qt>").arg(m_level)));

    QListWidget *list = new QListWidget();
    for (int i = 0; i < 100; ++i)
        list->addItem(tr("Item #%1").arg(i));
    lay->addWidget(list);

    QPushButton *pb = new QPushButton(tr("Open test dialog"));
    lay->addWidget(pb);
    connect(pb, SIGNAL(clicked()), this, SLOT(showDialog()));

    setCentralWidget(central);
    
    // The Maemo5 window manager mis-calculates the length of the window title
    // and won't show the 'v' indicator in Portrait mode if the title is "Stacked"
    setWindowTitle(tr("Stacke"));
}

Window::~Window()
{
    qWarning() << "Stacked Window [" << m_level << "] is being destroyed.";
}

void Window::openNewWindow()
{
    Window *w = new Window(this, true);
    w->show();
}

void Window::showDialog()
{
    if (!m_dialog) {
        m_dialog = new QDialog(this);
        m_dialog->setWindowTitle(tr("A Dialog"));

        QStringList onetwothree;
        onetwothree << tr("One") << tr("Two") << tr("Three");

        QVBoxLayout *lay = new QVBoxLayout(m_dialog);
        lay->addWidget(new QLabel(tr("<qt>This is a Maemo5 Stacked Dialog.")));
        lay->addWidget(new QLineEdit(tr("LineEdit")));
        lay->addWidget(new QSlider(Qt::Horizontal));
        QProgressBar *prb = new QProgressBar();
        prb->setValue(50);
        lay->addWidget(prb);
        lay->addWidget(new QCheckBox(tr("CheckBox")));
//        QComboBox *com = new QComboBox();
//        com->setEditable(false);
//        com->addItems(onethwothree);
//        lay->addWidget(com);
        QButtonGroup *radios = new QButtonGroup(this);
        QHBoxLayout *radiolay = new QHBoxLayout();
        radiolay->setSpacing(0);
        foreach(const QString &str, onetwothree) {
            QRadioButton *radio = new QRadioButton(str);
            radiolay->addWidget(radio);
            if (radios->buttons().isEmpty())
                radio->setChecked(true);
            radios->addButton(radio);
        }
        lay->addLayout(radiolay);
    }
    if (!m_dialog->isVisible())
        m_dialog->show();
}

void Window::showMessageBox()
{
    QMessageBox::information(this, tr("A MessageBox"), tr("This is a Maemo5 Message Box."), QMessageBox::Ok | QMessageBox::Cancel | QMessageBox::Apply);
}

void Window::showBanner()
{
    QMaemo5InformationBox::information(this, tr("This is a <b>Maemo5</b> <i>banner</i>"), QMaemo5InformationBox::DefaultTimeout);
}

void Window::showNote()
{
    QMaemo5InformationBox::information(this, tr("This<br>is<br>a<br><b>Maemo5</b><br><i>note</i>"), QMaemo5InformationBox::NoTimeout);
}

void Window::showComplexNote()
{
    QMaemo5InformationBox *box = new QMaemo5InformationBox(this);
    box->setWidget(new RotatedLabel(tr("Qt for Maemo5"), 45));
    box->setTimeout(QMaemo5InformationBox::NoTimeout);
    box->exec();
}

void Window::toggleLandscape(bool b)
{
    setAttribute(Qt::WA_Maemo5ForcePortraitOrientation, !b);
    setAttribute(Qt::WA_Maemo5ForceLandscapeOrientation, b);
    
    m_landscape->setChecked(b);
    m_portrait->setChecked(!b);
}

void Window::toggleFullScreen(bool b)
{
    setWindowState(b ? windowState() | Qt::WindowFullScreen
                     : windowState() & ~Qt::WindowFullScreen);
}

void Window::showExpose()
{
    QDBusConnection c = QDBusConnection::sessionBus();
    QDBusMessage m = QDBusMessage::createSignal("/","com.nokia.hildon_desktop","exit_app_view");
    c.send(m);
}

void Window::beBusy(bool on)
{
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, on);
}

void Window::showEvent(QShowEvent *se)
{
    QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                                                     MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ));
    QDBusConnection::systemBus().connect("", MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                                         MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString, QString, QString, int, int, int)));
    QMainWindow::showEvent(se);
}

void Window::hideEvent(QHideEvent *he)
{
    QDBusConnection::systemBus().disconnect("", MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                                            MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString, QString, QString, int, int, int)));
    QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                                                     MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ));
    QMainWindow::hideEvent(he);
}

void Window::orientationChanged(QString landscape, QString /*stand*/, QString /*facing*/, int /*x_mG*/, int /*y_mG*/, int /*z_mG*/)
{
    if (landscape == QLatin1String(MCE_ORIENTATION_LANDSCAPE))
        toggleLandscape(true);
    else if (landscape == QLatin1String(MCE_ORIENTATION_PORTRAIT))
        toggleLandscape(false);
}
