/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the Qt Mobility Components.
**
** $QT_BEGIN_LICENSE:LGPL$
** GNU Lesser General Public License Usage
** This file may be used under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation and
** appearing in the file LICENSE.LGPL included in the packaging of this
** file. Please review the following information to ensure the GNU Lesser
** General Public License version 2.1 requirements will be met:
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU General
** Public License version 3.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of this
** file. Please review the following information to ensure the GNU General
** Public License version 3.0 requirements will be met:
** http://www.gnu.org/copyleft/gpl.html.
**
** Other Usage
** Alternatively, this file may be used in accordance with the terms and
** conditions contained in a signed written agreement between you and Nokia.
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qorganizeritemfetchforexportrequest.h"
#include "qorganizeritemrequests_p.h"

QTM_BEGIN_NAMESPACE

/*!
  \class QOrganizerItemFetchForExportRequest
  \brief The QOrganizerItemFetchForExportRequest class allows a client to asynchronously
    request organizer items from an organizer item store manager.
  \inmodule QtOrganizer


  For a QOrganizerItemFetchForExportRequest, the resultsAvailable() signal will be emitted when the resultant
  organizer item (which may be retrieved by calling items()), are updated, as well as if
  the overall operation error (which may be retrieved by calling error()) is updated.

  \ingroup organizer-requests
 */

/*! Constructs a new organizer item fetch request whose parent is the specified \a parent */
QOrganizerItemFetchForExportRequest::QOrganizerItemFetchForExportRequest(QObject* parent)
    : QOrganizerAbstractRequest(new QOrganizerItemFetchForExportRequestPrivate, parent)
{
}

/*! Sets the organizer item filter used to determine which organizer items will be retrieved to \a filter */
void QOrganizerItemFetchForExportRequest::setFilter(const QOrganizerItemFilter& filter)
{
    Q_D(QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    d->m_filter = filter;
}

/*! Sets the sort order of the result to \a sorting.  Only has an effect if called prior to calling \c start() */
void QOrganizerItemFetchForExportRequest::setSorting(const QList<QOrganizerItemSortOrder>& sorting)
{
    Q_D(QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    d->m_sorting = sorting;
}

/*!
  Sets the fetch hint which may be used by the backend to optimize organizer item retrieval
  to \a fetchHint.  A client should not make changes to a organizer item which has been retrieved
  using a fetch hint other than the default fetch hint.  Doing so will result in information
  loss when saving the organizer item back to the manager (as the "new" restricted organizer item will
  replace the previously saved organizer item in the backend).
  \sa QOrganizerItemFetchHint
 */
void QOrganizerItemFetchForExportRequest::setFetchHint(const QOrganizerItemFetchHint &fetchHint)
{
    Q_D(QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    d->m_fetchHint = fetchHint;
}

/*! Sets the start period of the request to \a date. Only has an effect if called prior to calling \c start() */
void QOrganizerItemFetchForExportRequest::setStartDate(const QDateTime &date)
{
    Q_D(QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    d->m_startDate = date;
}

/*! Sets the end period of the request to \a date. Only has an effect if called prior to calling \c start() */
void QOrganizerItemFetchForExportRequest::setEndDate(const QDateTime &date)
{
    Q_D(QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    d->m_endDate = date;
}

/*! Returns the filter that will be used to select organizer items to be returned */
QOrganizerItemFilter QOrganizerItemFetchForExportRequest::filter() const
{
    Q_D(const QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    return d->m_filter;
}

/*! Returns the sort ordering that will be used sort the results of this request */
QList<QOrganizerItemSortOrder> QOrganizerItemFetchForExportRequest::sorting() const
{
    Q_D(const QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    return d->m_sorting;
}

/*!
  Returns the fetch hint which may be used by the backend to optimize organizer item retrieval.
  A client should not make changes to an organizer item which has been retrieved
  using a fetch hint other than the default fetch hint.  Doing so will result in information
  loss when saving the organizer item back to the manager (as the "new" restricted organizer item will
  replace the previously saved organizer item in the backend).
  \sa QOrganizerItemFetchHint
 */
QOrganizerItemFetchHint QOrganizerItemFetchForExportRequest::fetchHint() const
{
    Q_D(const QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    return d->m_fetchHint;
}

/*!
   Returns the date-time which is the lower bound for the range in which items will be returned.
   An invalid (default-constructed) date-time signifies that no lower bound is given (matches everything
   up to the end date).
   Note that an item matches if either it or any of its occurrences occur within the defined range.
 */
QDateTime QOrganizerItemFetchForExportRequest::startDate() const
{
    Q_D(const QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    return d->m_startDate;
}

/*!
   Returns the date-time which is the upper bound for the range in which items will be returned.
   An invalid (default-constructed) date-time signifies that no upper bound is given (matches everything
   after the start date).
   Note that an item matches if either it or any of its occurrences occur within the defined range.
 */
QDateTime QOrganizerItemFetchForExportRequest::endDate() const
{
    Q_D(const QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    return d->m_endDate;
}

/*! Returns the list of organizer items retrieved by this request */
QList<QOrganizerItem> QOrganizerItemFetchForExportRequest::items() const
{
    Q_D(const QOrganizerItemFetchForExportRequest);
    QMutexLocker ml(&d->m_mutex);
    return d->m_organizeritems;
}

#include "moc_qorganizeritemfetchforexportrequest.cpp"

QTM_END_NAMESPACE
