/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the Graphics Dojo project on Trolltech Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** sales department at sales@trolltech.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/
#include "context2d.h"

#include <QVariant>
#include <QDebug>

#include <math.h>
static const float Q_PI   = 3.141592;//65358979323846;   // pi

#define DEGREES(t) ((t) * 180.0 / Q_PI)

#define qClamp(val, min, max) qMin(qMax(val, min), max)
static QList<float> parseNumbersList(QString::const_iterator &itr)
{
    QList<float> points;
    QString temp;
    while ((*itr).isSpace())
        ++itr;
    while ((*itr).isNumber() ||
           (*itr) == '-' || (*itr) == '+' || (*itr) == '.') {
        temp = QString();

        if ((*itr) == '-')
            temp += *itr++;
        else if ((*itr) == '+')
            temp += *itr++;
        while ((*itr).isDigit())
            temp += *itr++;
        if ((*itr) == '.')
            temp += *itr++;
        while ((*itr).isDigit())
            temp += *itr++;
        while ((*itr).isSpace())
            ++itr;
        if ((*itr) == ',')
            ++itr;
        points.append(temp.toDouble());
        //eat spaces
        while ((*itr).isSpace())
            ++itr;
    }

    return points;
}

static QColor colorFromString(const QString &name)
{
    QString::const_iterator itr = name.constBegin();
    QList<float> compo;
    if (name.startsWith("rgba(")) {
        ++itr; ++itr; ++itr; ++itr; ++itr;
        compo = parseNumbersList(itr);
        if (compo.size() != 4) {
            return QColor();
        }
        //alpha seems to be always between 0-1
        compo[3] *= 255;
        //qDebug()<<compo;
        return QColor((int)compo[0], (int)compo[1],
                      (int)compo[2], (int)compo[3]);
    } else if (name.startsWith("rgb(")) {
        ++itr; ++itr; ++itr; ++itr;
        compo = parseNumbersList(itr);
        if (compo.size() != 3) {
            return QColor();
        }
        return QColor((int)qClamp(compo[0], (float)0.0, (float)255.0),
                      (int)qClamp(compo[1], (float)0.0, (float)255.0), 
                      (int)qClamp(compo[2], (float)0.0, (float)255.0));
    } else {
        //QRgb color;
        //CSSParser::parseColor(name, color);
        return QColor(name);
    }
}


static QPainter::CompositionMode compositeOperatorFromString(const QString &compositeOperator)
{
    if ( compositeOperator == "source-over" ) {
        return QPainter::CompositionMode_SourceOver;
    } else if ( compositeOperator == "source-out" ) {
        return QPainter::CompositionMode_SourceOut;
    } else if ( compositeOperator == "source-in" ) {
        return QPainter::CompositionMode_SourceIn;
    } else if ( compositeOperator == "source-atop" ) {
        return QPainter::CompositionMode_SourceAtop;
    } else if ( compositeOperator == "destination-atop" ) {
        return QPainter::CompositionMode_DestinationAtop;
    } else if ( compositeOperator == "destination-in" ) {
        return QPainter::CompositionMode_DestinationIn;
    } else if ( compositeOperator == "destination-out" ) {
        return QPainter::CompositionMode_DestinationOut;
    } else if ( compositeOperator == "destination-over" ) {
        return QPainter::CompositionMode_DestinationOver;
    } else if ( compositeOperator == "darker" ) {
        return QPainter::CompositionMode_SourceOver;
    } else if ( compositeOperator == "lighter" ) {
        return QPainter::CompositionMode_SourceOver;
    } else if ( compositeOperator == "copy" ) {
        return QPainter::CompositionMode_Source;
    } else if ( compositeOperator == "xor" ) {
        return QPainter::CompositionMode_Xor;
    }

    return QPainter::CompositionMode_SourceOver;
}

void Context2D::save()
{
    m_stateStack.push(m_state);
    m_painter.save();
}


void Context2D::restore()
{
    if (!m_stateStack.isEmpty())
        m_state = m_stateStack.pop();
    m_painter.restore();
}


void Context2D::scale(float x, float y)
{
    if (m_state.creatingShape)
        m_state.matrix.scale(x, y);
    m_painter.scale(x, y);
}


void Context2D::rotate(float angle)
{
    if (m_state.creatingShape)
        m_state.matrix.rotate(DEGREES(angle));
    m_painter.rotate(DEGREES(angle));
}


void Context2D::translate(float x, float y)
{
    if (m_state.creatingShape)
        m_state.matrix.translate(x, y);
    m_painter.translate(x, y);
}


void Context2D::transform(float m11, float m12, float m21, float m22,
                          float dx, float dy)
{
    QMatrix mat(m11, m12,
                m21, m22,
                dx, dy);
    if (m_state.creatingShape)
        m_state.matrix *= mat;
    m_painter.setMatrix(mat, true);
}


void Context2D::setTransform(float m11, float m12, float m21, float m22,
                             float dx, float dy)
{
    QMatrix mat(m11, m12,
                m21, m22,
                dx, dy);
    if (m_state.creatingShape)
        m_state.matrix = mat;
    m_painter.setMatrix(mat, false);
}

void Context2D::setGlobalAlpha(float alpha)
{
    m_painter.setOpacity(alpha);
}

void Context2D::setGlobalCompositeOperation(const QString &op)
{
    QPainter::CompositionMode mode =
        compositeOperatorFromString(op);

    m_painter.setCompositionMode(mode);
}

void Context2D::setStrokeStyle(const QString &style)
{
    QColor clr = colorFromString(style);
    QPen pen = m_painter.pen();
    pen.setColor(clr);
    if (pen.style() == Qt::NoPen)
        pen.setStyle(Qt::SolidLine);
    m_painter.setPen(pen);
}

void Context2D::setFillStyle(const QString &style)
{
    QColor clr = colorFromString(style);
    m_painter.setBrush(clr);
}

float Context2D::globalAlpha() const
{
    return m_painter.opacity();
}


QString Context2D::globalCompositeOperation() const
{
    return 0;
}


QString Context2D::strokeStyle() const
{
    return m_painter.pen().color().name();
}


QString Context2D::fillStyle() const
{
    return m_painter.brush().color().name();
}


void Context2D::setLineWidth(float w)
{
    QPen p = m_painter.pen();
    p.setWidthF(w);
    m_painter.setPen(p);
}

void Context2D::setLineCap(const QString &capString)
{
    QPen pen = m_painter.pen();
    if (capString == "round")
        pen.setCapStyle(Qt::RoundCap);
    else if (capString == "square")
        pen.setCapStyle(Qt::SquareCap);
    else
        pen.setCapStyle(Qt::FlatCap);
    m_painter.setPen(pen);
}

void Context2D::setLineJoin(const QString &joinString)
{
    QPen pen = m_painter.pen();
    if (joinString == "round")
        pen.setJoinStyle(Qt::RoundJoin);
    else if (joinString == "bevel")
        pen.setJoinStyle(Qt::BevelJoin);
    else
        pen.setJoinStyle(Qt::MiterJoin);
    m_painter.setPen(pen);
}

void Context2D::setMiterLimit(float m)
{
    QPen pen = m_painter.pen();
    pen.setMiterLimit(m);
    m_painter.setPen(pen);
}

float Context2D::lineWidth() const
{
    return m_painter.pen().widthF();
}

QString Context2D::lineCap() const
{
    return QString();
}


QString Context2D::lineJoin() const
{
    return QString();
}

float Context2D::miterLimit() const
{
    return 0;
}

void Context2D::setShadowOffsetX(float x)
{
    Q_UNUSED(x);
}

void Context2D::setShadowOffsetY(float y)
{
    Q_UNUSED(y);
}

void Context2D::setShadowBlur(float b)
{
    Q_UNUSED(b);
}

void Context2D::setShadowColor(const QColor &c)
{
    Q_UNUSED(c);
}

float Context2D::shadowOffsetX() const
{
    return 0;
}

float Context2D::shadowOffsetY() const
{
    return 0;
}


float Context2D::shadowBlur() const
{
    return 0;
}


QColor Context2D::shadowColor() const
{
    return QColor();
}


void Context2D::clearRect(float x, float y, float w, float h)
{
    m_painter.save();
    m_painter.setCompositionMode(QPainter::CompositionMode_Source);
    m_painter.fillRect(QRectF(x, y, w, h), Qt::white);
    m_painter.restore();
}


void Context2D::fillRect(float x, float y, float w, float h)
{
    m_painter.fillRect(QRectF(x, y, w, h), m_painter.brush());
}


void Context2D::strokeRect(float x, float y, float w, float h)
{
    QPainterPath path; path.addRect(x, y, w, h);
    m_painter.strokePath(path, m_painter.pen());
}


void Context2D::beginPath()
{
    m_path = QPainterPath();
    m_state.creatingShape = true;
}


void Context2D::closePath()
{
    m_path.closeSubpath();
    m_state.creatingShape = false;
}


void Context2D::moveTo(float x, float y)
{
    QPointF pt = m_state.matrix.map(QPointF(x, y));
    m_path.moveTo(pt);
}


void Context2D::lineTo(float x, float y)
{
    QPointF pt = m_state.matrix.map(QPointF(x, y));
    m_path.lineTo(pt);
}


void Context2D::quadraticCurveTo(float cpx, float cpy, float x, float y)
{
    QPointF cp = m_state.matrix.map(QPointF(cpx, cpy));
    QPointF xy = m_state.matrix.map(QPointF(x, y));
    m_path.quadTo(cp, xy);
}


void Context2D::bezierCurveTo(float cp1x, float cp1y,
                              float cp2x, float cp2y, float x, float y)
{
    QPointF cp1 = m_state.matrix.map(QPointF(cp1x, cp1y));
    QPointF cp2 = m_state.matrix.map(QPointF(cp2x, cp2y));
    QPointF end = m_state.matrix.map(QPointF(x, y));
    m_path.cubicTo(cp1, cp2, end);
}


void Context2D::arcTo(float x1, float y1, float x2, float y2, float radius)
{
    //FIXME: this is surely busted
    QPointF st  = m_state.matrix.map(QPointF(x1, y1));
    QPointF end = m_state.matrix.map(QPointF(x2, y2));
    m_path.arcTo(st.x(), st.y(),
                 end.x()-st.x(), end.y()-st.y(),
                 radius, 90);
}


void Context2D::rect(float x, float y, float w, float h)
{
    QPainterPath path; path.addRect(x, y, w, h);
    path = m_state.matrix.map(path);
    m_path.addPath(path);
}

void Context2D::arc(float xc, float yc, float radius,
                    float sar, float ear,
                    bool anticlockwise)
{
    //### HACK
    // In Qt we don't switch the coordinate system for degrees
    // and still use the 0,0 as bottom left for degrees so we need
    // to switch
    sar = -sar;
    ear = -ear;
    anticlockwise = !anticlockwise;
    //end hack

    float sa = DEGREES(sar);
    float ea = DEGREES(ear);

    float span = 0;

    float xs     = xc - radius;
    float ys     = yc - radius;
    float width  = radius*2;
    float height = radius*2;

    if (!anticlockwise && (ea < sa)) {
        span += 360;
    } else if (anticlockwise && (sa < ea)) {
        span -= 360;
    }

    //### this is also due to switched coordinate system
    // we would end up with a 0 span instead of 360
    if (!(qFuzzyCompare(span + (ea - sa), (float)0.0) &&
          qFuzzyCompare(qAbs(span), (float)360.0))) {
        span   += ea - sa;
    }

    QPainterPath path;
    path.moveTo(QPointF(xc + radius  * cos(sar),
                                yc - radius  * sin(sar)));

    path.arcTo(xs, ys, width, height, sa, span);
    path = m_state.matrix.map(path);
    m_path.addPath(path);
}


void Context2D::fill()
{
    m_painter.fillPath(m_path, m_painter.brush());
    m_state.creatingShape = false;
}


void Context2D::stroke()
{
    m_painter.strokePath(m_path, m_painter.pen());
    m_state.creatingShape = false;
}


void Context2D::clip()
{
    m_painter.setClipPath(m_path);
}


bool Context2D::isPointInPath(float x, float y) const
{
    return m_path.contains(QPointF(x, y));
}

Context2D::Context2D(QWidget *parent)
    : QObject(parent),
      m_cache(parent->size())
{
    //m_cache.fill(Qt::transparent);

    begin();
}

const QPixmap & Context2D::end()
{
    m_painter.end();
    m_state.creatingShape = false;
    return m_cache;
}

void Context2D::begin()
{
    if (!m_painter.isActive()) {
        m_painter.begin(&m_cache);
        m_painter.setRenderHint(QPainter::Antialiasing);
        m_painter.setBrush(Qt::black);
    }
}

void Context2D::setSize(int w, int h)
{
    if (m_painter.isActive())
        end();
    QPixmap newp(w, h);
    newp.fill(Qt::transparent);
    QPainter p(&newp);
    p.drawPixmap(0, 0, m_cache);
    p.end();
    m_cache = newp;
    begin();
}
// (setq show-trailing-whitespace t)
