/*!
 * Copyright (C) 2017 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 1.0

Item {
    id: root

    /*!
     * \internal
     */
    default property alias items: items.data

    /*!
     * type:int
     * \brief The number of items in the splitter.
     */
    property alias count: items.children.length
    
    /*!
     * \brief The orientation of the splitter.
     *
     * Possible values are Qt.Horizontal (the default) or Qt.Vertical.
     */
    property int orientation: Qt.Horizontal
    
    /*!
     * \brief Indicates whether items are current being resized.
     */
    property bool resizing: false
    
    /*!
     * type:SplitterStyle
     * \brief Provides styling properties for the splitter.
     */
    property QtObject style: SplitterStyle {}
    
    /*!
     * \brief Adds the specified \a item to the splitter.
     * @param type:Item item
     */
    function addItem(item) {
        item.parent = item;
        internal.updateGeometry();
        var h = handle.createObject(handles);
        h.index = handles.children.length - 1;
        item.xChanged.connect(internal.updateHandles);
        item.yChanged.connect(internal.updateHandles);
        item.widthChanged.connect(internal.updateHandles);
        item.heightChanged.connect(internal.updateHandles);
    }

    /*!
     * \brief Removes the specified \a item from the splitter.
     * @param type:Item item
     */
    function removeItem(item) {
        for (var i = 0; i < items.children.length; i++) {
            if (items.children[i] === item) {
                item.xChanged.disconnect(internal.updateHandles);
                item.yChanged.disconnect(internal.updateHandles);
                item.widthChanged.disconnect(internal.updateHandles);
                item.heightChanged.disconnect(internal.updateHandles);
                item.parent = null;
                var h = handles.children[i];

                if (h) {
                    h.destroy();
                }

                return;
            }
        }
    }
    
    Item {
        id: items
        
        anchors.fill: parent
    }
    
    Item {
        id: handles
        
        anchors.fill: parent
    }
    
    QtObject {
        id: internal
        
        property bool complete: false
        
        function init() {            
            for (var i = 0; i < items.children.length - 1; i++) {
                var h = handle.createObject(handles);
                h.index = i;
            }
            
            complete = true;
            updateGeometry();
            updateHandles();
        }

        function updateGeometry() {
            if (!complete) {
                return;
            }

            var count = items.children.length;

            if (!count) {
                return;
            }

            if (root.orientation == Qt.Vertical) {
                var maxHeight = items.height - handles.children.length * root.style.handleWidth;

                for (var i = 0; i < count; i++) {
                    var child = items.children[i];
                    child.width = items.width;
                    child.height = (i == count - 1 ? maxHeight : Math.min(child.height, maxHeight));
                    maxHeight -= child.height;
                }
            }
            else {
                var maxWidth = items.width - handles.children.length * root.style.handleWidth;

                for (var i = 0; i < count; i++) {
                    var child = items.children[i];
                    child.height = items.height;
                    child.width = (i == count - 1 ? maxWidth : Math.min(child.width, maxWidth));
                    maxWidth -= child.width;
                }
            }
        }
        
        function updateHandles() {
            if ((!complete) || (root.resizing)) {
                return;
            }
            
            for (var i = 0; i < handles.children.length; i++) {
                updateHandle(handles.children[i]);
            }
        }
        
        function updateHandle(h) {
            if ((!complete) || (root.resizing)) {
                return;
            }
            
            root.resizing = true;
            var left = items.children[h.index];
            var right = items.children[h.index + 1];

            if (root.orientation == Qt.Vertical) {
                var topDiff = (left ? left.y + left.height - h.y : 0);
                
                if (topDiff) {
                    if (h.pressed) {
                        left.height -= topDiff;
                    }
                    else {
                        h.y += topDiff;
                    }
                }
                
                var bottomDiff = (right ? right.y - h.y - h.height : 0);
                
                if (bottomDiff) {
                    right.y -= bottomDiff;
                    right.height += bottomDiff;
                }
            }
            else {
                var leftDiff = (left ? left.x + left.width - h.x : 0);
                
                if (leftDiff) {
                    if (h.pressed) {
                        left.width -= leftDiff;
                    }
                    else {
                        h.x += leftDiff;
                    }
                }

                
                var rightDiff = (right ? right.x - h.x - h.width : 0);
                
                if (rightDiff) {
                    right.x -= rightDiff;
                    right.width += rightDiff;
                }
            }

            root.resizing = false;
        }
    }
    
    Component {
        id: handle
        
        Item {
            id: container
            
            property alias pressed: mouseArea.pressed
            property int index
            
            width: root.orientation == Qt.Vertical ? handles.width : root.style.handleWidth
            height: root.orientation == Qt.Vertical ? root.style.handleWidth : handles.height
            
            Image {
                id: image
                
                anchors.centerIn: parent
                source: root.orientation == Qt.Vertical ? root.style.verticalHandle : root.style.horizontalHandle
            }
            
            MouseArea {
                id: mouseArea
                
                anchors {
                    fill: parent
                    margins: -platformStyle.paddingSmall
                }
                drag.target: container
                drag.axis: root.orientation == Qt.Vertical ? Drag.YAxis : Drag.XAxis
            }
            
            onXChanged: internal.updateHandle(container)
            onYChanged: internal.updateHandle(container)
            onWidthChanged: internal.updateHandle(container)
            onHeightChanged: internal.updateHandle(container)
            Component.onCompleted: internal.updateHandle(container)
        }
    }
    
    onWidthChanged: internal.updateGeometry()
    onHeightChanged: internal.updateGeometry()
    onOrientationChanged: {
        internal.updateGeometry();
        internal.updateHandles();
    }
    Component.onCompleted: internal.init()
}
