/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#include "qchvideowindow.h"
#include "metadatawatcher.h"
#include "missioncontrol.h"
#include "mafw/mafwregistryadapter.h"
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <QApplication>
#include <QDBusConnection>
#include <QDBusMessage>
#include <QDeclarativeInfo>

class QchVideoWindowPrivate : public QchWindowPrivate
{

public:
    QchVideoWindowPrivate(QchVideoWindow *parent) :
        QchWindowPrivate(parent),
        mafwRegistry(0),
        mafwRenderer(0),
        mafwPlaylist(0),
        mafwTrackerSource(0),
        metadataWatcher(0),
        missionControl(0),
        autoLoad(true),
        bufferProgress(0.0),
        seekable(true),
        position(0),
        duration(0),
        status(QchMediaStatus::Stopped),
        muted(false),
        volume(0),
        muteVolume(0),
        tickInterval(1000),
        positionTimerId(-1),
        sourceLoaded(true),
        readyToPlay(false),
        playWhenReady(false)
    {
    }
    
    void loadSource() {
        QString uriToPlay = source.startsWith("/") ? "file://" + source : source;
        QString objectId = mafwTrackerSource->createObjectId(uriToPlay);
        
        if (uriToPlay.startsWith("file://")) {
            QString mime(gnome_vfs_get_mime_type_for_name(uriToPlay.toUtf8()));
            objectId = objectId.remove(0, 18) // "urisource::file://"
                               .replace("/", "%2F")
                               .prepend(QString("localtagfs::%1/")
                               .arg(mime.startsWith("video") ? "videos" : "music/songs"));
        }
        
        mafwPlaylist->assignVideoPlaylist();
        mafwPlaylist->clear();
        mafwPlaylist->appendItem(objectId);
        sourceLoaded = true;
    }
    
    void startPositionTimer() {
        if (positionTimerId == -1) {
            Q_Q(QchVideoWindow);
            positionTimerId = q->startTimer(tickInterval);
        }
        
        mafwRenderer->getPosition();
    }
    
    void stopPositionTimer() {
        if (positionTimerId != -1) {
            Q_Q(QchVideoWindow);
            q->killTimer(positionTimerId);
            positionTimerId = -1;
        }
    }
    
    void _q_onStatusReady(MafwPlaylist*, uint index, MafwPlayState state, const char*, const QString &error) {
        Q_Q(QchVideoWindow);
        q->disconnect(mafwRenderer, SIGNAL(signalGetStatus(MafwPlaylist*,uint,MafwPlayState,const char*,QString)),
                      q, SLOT(_q_onStatusReady(MafwPlaylist*,uint,MafwPlayState,const char*,QString)));

        q->connect(mafwRenderer, SIGNAL(stateChanged(int)), q, SLOT(_q_onStateChanged(int)));
        
        _q_onStateChanged(state);
        
        if (!error.isEmpty()) {
            qmlInfo(q) << error;
        }
    }
    
    void _q_onMetaDataChanged() {
        Q_Q(QchVideoWindow);
        int dur = metadataWatcher->metadata().value(MAFW_METADATA_KEY_DURATION).toInt();
        bool seek = metadataWatcher->metadata().value(MAFW_METADATA_KEY_IS_SEEKABLE).toBool();
        QString uri = metadataWatcher->metadata().value(MAFW_METADATA_KEY_URI).toString();
        
        if (dur != duration) {
            duration = dur;
            emit q->durationChanged();
        }
        
        if (seek != seekable) {
            seekable = seek;
            emit q->seekableChanged();
        }
        
        if (uri != source) {
            source = uri;
            sourceLoaded = true;
            emit q->sourceChanged();
        }
        
        emit q->hasAudioChanged();
        emit q->hasVideoChanged();
    }
    
    void _q_onBufferProgressChanged(float progress) {
        Q_Q(QchVideoWindow);
        bufferProgress = progress;
        emit q->bufferProgressChanged();
    }
    
    void _q_onPositionChanged(int pos) {
        Q_Q(QchVideoWindow);
        position = pos;
        emit q->positionChanged();
    }
    
    void _q_onVolumeChanged(int vol) {
        Q_Q(QchVideoWindow);
        volume = vol;
        emit q->volumeChanged();
    }
    
    void _q_onStateChanged(int state) {
        Q_Q(QchVideoWindow);
        QchMediaStatus::Status oldStatus = status;
        
        switch (state) {
        case Transitioning:
            readyToPlay = false;
            status = QchMediaStatus::Loading;
            break;
        case Playing:
            readyToPlay = false;
            status = QchMediaStatus::Playing;
            
            if (tickInterval > 0) {
                startPositionTimer();
            }
            
            if (oldStatus == QchMediaStatus::Paused) {
                emit q->resumed();
            }
            else {
                emit q->started();
            }
            
            break;
        case Paused:
            readyToPlay = true;
            
            if (playWhenReady) {
                playWhenReady = false;
                q->play();
            }
            else {
                status = QchMediaStatus::Paused;
                mafwRenderer->getPosition();
                stopPositionTimer();
                emit q->paused();
            }
            
            break;
        case Stopped:
            readyToPlay = true;
            
            if (playWhenReady) {
                playWhenReady = false;
                q->play();
            }
            else {
                if ((duration > 0) && (position >= duration)) {
                    status = QchMediaStatus::EndOfMedia;
                }
                else {
                    status = QchMediaStatus::Stopped;
                }
                
                position = 0;
                stopPositionTimer();
                emit q->positionChanged();
                emit q->stopped();
            }
            
            break;
        default:
            readyToPlay = false;
            break;
        }
        
        emit q->statusChanged();
    }
    
    void _q_onPropertyChanged(const QDBusMessage &msg) {
        if (msg.arguments()[0].toString() == MAFW_PROPERTY_RENDERER_VOLUME) {
            Q_Q(QchVideoWindow);
            volume = qdbus_cast<QVariant>(msg.arguments()[1]).toInt();
            emit q->volumeChanged();
        }
    }
    
    void _q_onError(const QDBusMessage &msg) {
        if (msg.arguments()[0] == "com.nokia.mafw.error.renderer") {
            status = QchMediaStatus::Error;
            errorString.clear();
            errorString.append(QchVideoWindow::tr("Unable to play media"));
            errorString.append("\n");

            if (msg.arguments()[1] == MAFW_RENDERER_ERROR_NO_MEDIA) {
                errorString.append(QchVideoWindow::tr("Media not found"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_URI_NOT_AVAILABLE) {
                errorString.append(QchVideoWindow::tr("URI not available"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_INVALID_URI) {
                errorString.append(QchVideoWindow::tr("Invalid URI"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_MEDIA_NOT_FOUND) {
                errorString.append(QchVideoWindow::tr("Unable to open media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_STREAM_DISCONNECTED) {
                errorString.append(QchVideoWindow::tr("Playback stream no longer available"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_TYPE_NOT_AVAILABLE) {
                errorString.append(QchVideoWindow::tr("Could not determine MIME-type"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_PLAYBACK) {
                errorString.append(QchVideoWindow::tr("General error occured, unable to continue playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNABLE_TO_PERFORM) {
                errorString.append(QchVideoWindow::tr("General error occured"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNSUPPORTED_TYPE) {
                errorString.append(QchVideoWindow::tr("Unsupported media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNSUPPORTED_RESOLUTION) {
                errorString.append(QchVideoWindow::tr("Unsupported resolution"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_UNSUPPORTED_FPS) {
                errorString.append(QchVideoWindow::tr("Unsupported framerate"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_DRM) {
                errorString.append(QchVideoWindow::tr("Media is protected by DRM"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_DEVICE_UNAVAILABLE) {
                errorString.append(QchVideoWindow::tr("System sound device is unavailable"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CORRUPTED_FILE) {
                errorString.append(QchVideoWindow::tr("Media corrupted"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_PLAYLIST_PARSING) {
                errorString.append(QchVideoWindow::tr("Error while parsing playlist"));
                errorString.append(QchVideoWindow::tr("Playlist may be corrupt or empty"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CODEC_NOT_FOUND) {
                errorString.append(QchVideoWindow::tr("Codec not found:") + "\n");
                errorString.append(msg.arguments()[2].toString());
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_VIDEO_CODEC_NOT_FOUND) {
                errorString.append(QchVideoWindow::tr("Video codec not found:") + "\n");
                errorString.append(msg.arguments()[2].toString());
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_AUDIO_CODEC_NOT_FOUND) {
                errorString.append(QchVideoWindow::tr("Video codec not found:") + "\n");
                errorString.append(msg.arguments()[2].toString());
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_NO_PLAYLIST) {
                errorString.append(QchVideoWindow::tr("No playlist assigned"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_INDEX_OUT_OF_BOUNDS) {
                errorString.append(QchVideoWindow::tr("Media index is not in bound with playlist items"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_PLAY) {
                errorString.append(QchVideoWindow::tr("Unable to start playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_STOP) {
                errorString.append(QchVideoWindow::tr("Unable to stop playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_PAUSE) {
                errorString.append(QchVideoWindow::tr("Unable to pause playback"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_SET_POSITION) {
                errorString.append(QchVideoWindow::tr("Unable to seek position in media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_GET_POSITION) {
                errorString.append(QchVideoWindow::tr("Unable to retrieve current position in media"));
            }
            else if (msg.arguments()[1] == MAFW_RENDERER_ERROR_CANNOT_GET_STATUS) {
                errorString.append(QchVideoWindow::tr("Unable to get current playback status"));
            }
                
            Q_Q(QchVideoWindow);
            emit q->statusChanged();
            emit q->error(errorString);
        }
    }
    
    MafwRegistryAdapter *mafwRegistry;
    MafwRendererAdapter *mafwRenderer;
    MafwPlaylistAdapter *mafwPlaylist;
    MafwSourceAdapter *mafwTrackerSource;
    MetadataWatcher *metadataWatcher;
    MissionControl *missionControl;
    
    bool autoLoad;
    
    qreal bufferProgress;
    
    QString errorString;
    
    bool seekable;
    
    int position;
    int duration;
    
    QString source;
    
    QchMediaStatus::Status status;
    
    bool muted;
    
    int volume;
    int muteVolume;
    
    int tickInterval;
    
    int positionTimerId;
    
    bool sourceLoaded;
    
    bool readyToPlay;
    bool playWhenReady;
    
    Q_DECLARE_PUBLIC(QchVideoWindow)
};

QchVideoWindow::QchVideoWindow(QWidget *parent) :
    QWidget(parent),
    d_ptr(new QchVideoWindowPrivate(this))
{
    Q_D(QchVideoWindow);
    setAttribute(Qt::WA_OpaquePaintEvent);
    
    d->mafwRegistry = MafwRegistryAdapter::get();
    d->mafwRenderer = d->mafwRegistry->renderer();
    d->mafwPlaylist = d->mafwRegistry->playlist();
    d->mafwTrackerSource = d->mafwRegistry->source(MafwRegistryAdapter::Tracker);
    d->metadataWatcher = MetadataWatcher::acquire();
    d->missionControl = MissionControl::acquire();
    
    d->mafwRenderer->setErrorPolicy(MAFW_RENDERER_ERROR_POLICY_STOP);
    
    connect(d->metadataWatcher, SIGNAL(metadataChanged()), this, SLOT(_q_onMetaDataChanged()));
    connect(d->mafwRenderer, SIGNAL(signalGetStatus(MafwPlaylist*,uint,MafwPlayState,const char*,QString)),
            this, SLOT(_q_onStatusReady(MafwPlaylist*,uint,MafwPlayState,const char*,QString)));
    connect(d->mafwRenderer, SIGNAL(signalGetPosition(int,QString)), this, SLOT(_q_onPositionChanged(int)));
    connect(d->mafwRenderer, SIGNAL(signalGetVolume(int)), this, SLOT(_q_onVolumeChanged(int)));
    connect(d->mafwRenderer, SIGNAL(bufferingInfo(float)), this, SLOT(_q_onBufferProgressChanged(float)));
            
    QDBusConnection::sessionBus().connect("com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer",
                                          "/com/nokia/mafw/renderer/gstrenderer",
                                          "com.nokia.mafw.extension",
                                          "property_changed",
                                          this, SLOT(_q_onPropertyChanged(const QDBusMessage &)));
                                          
    QDBusConnection::sessionBus().connect("",
                                          "/com/nokia/mafw/renderer/gstrenderer",
                                          "com.nokia.mafw.extension",
                                          "error",
                                          this, SLOT(_q_onError(const QDBusMessage &)));
}

QchVideoWindow::~QchVideoWindow() {
    Q_D(QchVideoWindow);
    d->mafwRenderer->setErrorPolicy(MAFW_RENDERER_ERROR_POLICY_CONTINUE);
    d->mafwRenderer->enablePlayback(false);
    d->mafwRenderer->stop();
}

bool QchVideoWindow::autoLoad() const {
    Q_D(const QchVideoWindow);
    return d->autoLoad;
}

void QchVideoWindow::setAutoLoad(bool enable) {
    if (enable != autoLoad()) {
        Q_D(QchVideoWindow);
        d->autoLoad = enable;
        emit autoLoadChanged();
    }
}

qreal QchVideoWindow::bufferProgress() const {
    Q_D(const QchVideoWindow);
    return d->bufferProgress;
}

QString QchVideoWindow::errorString() const {
    Q_D(const QchVideoWindow);
    return d->errorString;
}

bool QchVideoWindow::hasAudio() const {
    return !metaData()->audioCodec().isEmpty();
}

bool QchVideoWindow::hasVideo() const {
    return !metaData()->videoCodec().isEmpty();
}

MetadataWatcher* QchVideoWindow::metaData() const {
    Q_D(const QchVideoWindow);
    return d->metadataWatcher;
}

bool QchVideoWindow::isMuted() const {
    Q_D(const QchVideoWindow);
    return d->muted;
}

void QchVideoWindow::setMuted(bool muted) {
    if (muted != isMuted()) {
        Q_D(QchVideoWindow);
        d->muted = muted;
        emit mutedChanged();
        
        if (muted) {
            d->muteVolume = volume();
            setVolume(0);
        }
        else {
            setVolume(d->muteVolume);
        }
    }
}

bool QchVideoWindow::isPaused() const {
    return status() == QchMediaStatus::Paused;
}

void QchVideoWindow::setPaused(bool paused) {
    if (paused) {
        pause();
    }
    else if (isPaused()) {
        play();
    }
}

bool QchVideoWindow::isPlaying() const {
    switch (status()) {
    case QchMediaStatus::Loading:
    case QchMediaStatus::Playing:
        return true;
    default:
        return false;
    }
}

void QchVideoWindow::setPlaying(bool playing) {
    if (playing) {
        play();
    }
    else if (isPlaying()) {
        pause();
    }
}

bool QchVideoWindow::isSeekable() const {
    Q_D(const QchVideoWindow);
    return d->seekable;
}

int QchVideoWindow::position() const {
    Q_D(const QchVideoWindow);
    return d->position;
}

void QchVideoWindow::setPosition(int pos) {
    if (pos != position()) {
        Q_D(QchVideoWindow);
        d->mafwRenderer->setPosition(SeekAbsolute, pos);
        d->mafwRenderer->getPosition();
    }
}

int QchVideoWindow::duration() const {
    Q_D(const QchVideoWindow);
    return d->duration;
}

QString QchVideoWindow::source() const {
    Q_D(const QchVideoWindow);
    return d->source;
}

void QchVideoWindow::setSource(const QString &uri) {
    if (uri != source()) {
        Q_D(QchVideoWindow);
        d->source = uri;
        
        if (autoLoad()) {
            d->loadSource();
        }
        else {
            d->sourceLoaded = false;
        }
    }
}

QchMediaStatus::Status QchVideoWindow::status() const {
    Q_D(const QchVideoWindow);
    return d->status;
}

int QchVideoWindow::volume() const {
    Q_D(const QchVideoWindow);
    return d->volume;
}

void QchVideoWindow::setVolume(int vol) {
    if (vol != volume()) {
        Q_D(QchVideoWindow);
        d->volume = qBound(0, vol, 100);
        d->mafwRenderer->setVolume(d->volume);
        d->mafwRenderer->getVolume();
    }
}

int QchVideoWindow::tickInterval() const {
    Q_D(const QchVideoWindow);
    return d->tickInterval;
}

void QchVideoWindow::setTickInterval(int interval) {
    if (interval != tickInterval()) {
        Q_D(QchVideoWindow);
        d->tickInterval = qMax(0, interval);
        emit tickIntervalChanged();
        
        d->stopPositionTimer();
        
        if ((interval > 0) && (isPlaying())) {
            d->startPositionTimer();
        }
    }
}

void QchVideoWindow::play() {
    Q_D(QchVideoWindow);
    
    if (isPaused()) {
        d->mafwRenderer->resume();
    }
    else {
        if (!d->sourceLoaded) {
            d->loadSource();
        }
        
        if (d->readyToPlay) {
            d->mafwRenderer->play();
        }
        else {
            d->playWhenReady = true;
        }
    }
}

void QchVideoWindow::pause() {
    if (!isPlaying()) {
        Q_D(QchVideoWindow);
        d->mafwRenderer->pause();
    }
}

void QchVideoWindow::stop() {
    if ((isPlaying()) || (isPaused())) {
        Q_D(QchVideoWindow);
        d->mafwRenderer->stop();
    }
}

void QchVideoWindow::timerEvent(QTimerEvent *) {
    Q_D(QchVideoWindow);
    d->mafwRenderer->getPosition();
}

void QchVideoWindow::classBegin() {}

void QchVideoWindow::componentComplete() {
    Q_D(QchVideoWindow);
    QApplication::syncX();
    d->mafwRenderer->setColorKey(QColor(3, 13, 3).rgb() & 0xffffff);
    d->mafwRenderer->setWindowXid(winId());
    
    if (d->mafwRenderer->isRendererReady()) {
        d->mafwRenderer->getStatus();
        d->mafwRenderer->getVolume();
    }
    else {
        connect(d->mafwRenderer, SIGNAL(rendererReady()), d->mafwRenderer, SLOT(getStatus()));
        connect(d->mafwRenderer, SIGNAL(rendererReady()), d->mafwRenderer, SLOT(getVolume()));
    }
}

#include "moc_qchvideowindow.cpp"
