//QueueView.cpp: Implementation of the QueueView class (view containing the current play queue items, playback controls etc).

#include "QueueView.h"


QueueView::QueueView(Spotify::Session* pSession, OssoWrapper* pOsso, DBusWrapper *pDBusWrapper, QWidget* parent)
    : QMainWindow(parent)
{
    //construct the QueueView UI

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5StackedWindow); //window is part of window stack
#endif

    setWindowTitle(tr("Now Playing"));

    m_pSession = pSession; //set the Spotify session instance
    m_playing = false; //set initial playback state
    m_userInitiatedPause = false;
    m_pauseOnHsetUnplugged = false;
    m_update = false;
    m_lastTimeTrackActivated = QTime::currentTime(); //set to current time
    m_CallerID = QUEUEVIEW_ID;

    m_pAlbumCoverContainer = new Spotify::ImageContainer(this); //allocate coverart imagecontainer instance
    connect(m_pAlbumCoverContainer,SIGNAL(loaded()),SLOT(OnAlbumCoverLoaded()));

    m_pOssoWrapper = pOsso; //set OssoWrapper instance pointer
    if(m_pOssoWrapper)
    {
        connect(m_pOssoWrapper,SIGNAL(playbackStateChangeRequest(pb_state_e)),SLOT(OnPlaybackStateChangeRequest(pb_state_e)));
    }

    m_pDBusWrapper = pDBusWrapper; //set D-bus wrapper instance pointer
    if(m_pDBusWrapper)
    {
       connect(m_pDBusWrapper,SIGNAL(onDisplayStateChange(bool)),SLOT(OnDisplayStateChanged(bool)));
       connect(m_pDBusWrapper,SIGNAL(onHalButtonPressed(const QString&)),SLOT(OnHalButtonPressed(const QString&)));
       connect(m_pDBusWrapper,SIGNAL(onHeadsetRemoved()),SLOT(OnHeadsetRemoved()));
    }

    m_pCurrTrack = NULL;
    m_currQueuePos = 0;
    m_currShufflePos = 0;
    m_currQueueBuffPos = 0;
    m_shuffle = false;
    m_shuffleStateChanged = false;

    //define slider used for displaying buffering and track progress
    m_currSliderWidth = SLIDER_WIDTH_LANDSCAPE;
    m_pProgressSlider = new CustomSlider(this,m_currSliderWidth,SLIDER_HEIGHT);
    m_pProgressSlider->setBgColor(QColor(51,51,51)); //background color of progress slider
    m_pProgressSlider->setP1Color(QColor(8,117,189)); //color of buffering progress bar
    m_pProgressSlider->setP2Color(QColor(41,166,241)); //color of track progress bar
    m_pProgressSlider->setTextColor(QColor(255,255,255)); //set text color to white
    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);

    m_pSpectrum = new SpectrumAnalyzer(this,50,40);
    m_pSpectrum->setColor(QColor(255,255,255)); //set spectrum color to white

    //init the trackinfo listview instance and associated item delegate
    m_pTrackInfoView = new QListView(this);
    m_pTrackInfoModel = new QStandardItemModel(this);
    m_pTrackInfoDelegate = new ListViewDelegate(this);
    m_pTrackInfoDelegate->setHeaderFontPointSize(15);
    m_pTrackInfoDelegate->setSubHeaderFontPointSize(14);
    m_pTrackInfoDelegate->setItemTextHighlightColor(QColor(41,166,241));

    m_pTrackInfoView->setItemDelegate(m_pTrackInfoDelegate); //connect the delegate to view
    m_pTrackInfoView->setModel(m_pTrackInfoModel); //connect the model to view
    m_pTrackInfoView->setProperty("FingerScrollable",true); //make sure that the current item is not accidentally de-selected when scrolling

    setupContextMenu(); //setup the listview contextmenu

    setupMenubar(); //setup the view menubar

    //set up signal and slot connections
    connect(m_pSession,SIGNAL(trackBufferingAmountUpdated(qint32)),SLOT(OnTrackBufferedAmountUpdated(qint32)));
    connect(m_pSession,SIGNAL(playbackPositionUpdated(qint64)),SLOT(OnPlaybackPositionUpdated(qint64)));
    connect(m_pSession,SIGNAL(albumBrowserReady(Spotify::AlbumBrowser*,qint32)),SLOT(OnAlbumBrowseReady(Spotify::AlbumBrowser*,qint32)));
    connect(m_pSession,SIGNAL(playing(bool)),SLOT(OnPlayingState(bool)));
    connect(m_pSession,SIGNAL(playbackFinished()),SLOT(OnTrackPlaybackFinished()));
    connect(m_pSession,SIGNAL(bufferNextTrack()),SLOT(OnBufferNextTrack()));
    connect(m_pTrackInfoView,SIGNAL(activated(QModelIndex)),SLOT(OnTrackActivated(QModelIndex)));
    connect(&m_PauseResumeBtn,SIGNAL(clicked()),SLOT(OnPauseResumeToggle()));
    connect(&m_ShuffleBtn,SIGNAL(clicked()),SLOT(OnShuffleToggle()));
    connect(&m_PrevBtn,SIGNAL(clicked()),SLOT(OnPrevTrackBtn()));
    connect(&m_NextBtn,SIGNAL(clicked()),SLOT(OnNextTrackBtn()));
    connect(menuBar(),SIGNAL(triggered(QAction*)),SLOT(OnMenubarAction(QAction*)));
    connect(QApplication::desktop(), SIGNAL(resized(int)),SLOT(OnOrientationChanged()));
   // connect(m_pProgressSlider,SIGNAL(positionChanged(int)),SLOT(OnProgressSliderClicked(int)));

    m_AlbumCover.setFixedSize(QUEUEVIEW_COVERART_WIDTH_LANDSCAPE,QUEUEVIEW_COVERART_HEIGHT_LANDSCAPE); //set fixed size of the album coverart

    //style the player control buttons
    m_PauseResumeBtn.setCheckable(true); //toggle button (between pause / resume states)
    m_PauseResumeBtn.setEnabled(false);

    m_PauseResumeBtn.setStyleSheet(
            "QPushButton {max-height: 44px;}"
            "QPushButton {min-height: 44px;}"
            "QPushButton {max-width: 100px;}"
            "QPushButton {min-width: 100px;}"
            "QPushButton {border-style: none;}"
            "QPushButton {background-image: url(/opt/qspot/images/play.png);}"
            "QPushButton:on {background-image: url(/opt/qspot/images/pause.png);}"
            "QPushButton:pressed {background-image: url(/opt/qspot/images/play-on.png);}");

    m_PrevBtn.setStyleSheet("QPushButton {min-width: 80px; min-height: 44px; max-width: 80px; max-height: 44px; border-style: none; background-image: url(/opt/qspot/images/previous.png);}"
                            "QPushButton:pressed {background-image: url(/opt/qspot/images/previous-on.png);}");

    m_NextBtn.setStyleSheet("QPushButton {min-width: 80px; min-height: 44px; max-width: 80px; max-height: 44px;border-style: none; background-image: url(/opt/qspot/images/next.png);}"
                            "QPushButton:pressed {background-image: url(/opt/qspot/images/next-on.png);}");

    m_ShuffleBtn.setCheckable(true);

    m_ShuffleBtn.setStyleSheet(
            "QPushButton {max-height: 44px;}"
            "QPushButton {min-height: 44px;}"
            "QPushButton {max-width: 60px;}"
            "QPushButton {min-width: 60px;}"
            "QPushButton {border-style: none;}"
            "QPushButton {background-image: url(/opt/qspot/images/shuffle.png);}"
            "QPushButton:on {background-image: url(/opt/qspot/images/shuffle-down.png);}");

    m_pPlayerControlsHoriz = new QGridLayout();

    m_pPlayerControlsHoriz->setSpacing(0);
    m_pPlayerControlsHoriz->addWidget(&m_PrevBtn,0,0,1,1,Qt::AlignRight);
    m_pPlayerControlsHoriz->addWidget(&m_PauseResumeBtn,0,1,1,1,Qt::AlignLeft);
    m_pPlayerControlsHoriz->addWidget(&m_NextBtn,0,2,Qt::AlignLeft);
    m_pPlayerControlsHoriz->setColumnStretch(2,6);
    m_pPlayerControlsHoriz->addWidget(&m_ShuffleBtn,0,3,1,1,Qt::AlignLeft);
    m_pPlayerControlsHoriz->setColumnStretch(3,5);
    m_pPlayerControlsHoriz->addWidget(m_pProgressSlider,0,4,1,1);
    m_pPlayerControlsHoriz->setColumnMinimumWidth(4,370);
    m_pPlayerControlsHoriz->setColumnStretch(4,10);
    m_pPlayerControlsHoriz->addWidget(m_pSpectrum,0,5,1,1,Qt::AlignRight);
    m_pPlayerControlsHoriz->setContentsMargins(10,5,10,0);

    m_pHorizGridLayout = new QGridLayout();

    //set up the QueueView UI landscape mode layout

    m_pHorizGridLayout->addWidget(&m_AlbumCover,0,0,7,1,Qt::AlignTop);
    m_pHorizGridLayout->addWidget(m_pTrackInfoView,0,1,7,1);
    m_pHorizGridLayout->addLayout(m_pPlayerControlsHoriz,7,0,1,-1);
    m_pHorizGridLayout->setContentsMargins(0,5,0,10); //set layout margin

    m_pPlayerControlsVert = new QGridLayout();
    m_pProgressLayoutVert = new QGridLayout();

    //allocate the QueueView UI portrait mode layout
    m_pVertLayout = new QGridLayout();
    m_pVertLayout->addLayout(m_pPlayerControlsVert,2,0,1,-1);
    m_pVertLayout->addLayout(m_pProgressLayoutVert,3,0,1,-1);

    m_MainHorizLayout.setLayout(m_pHorizGridLayout);
    m_MainVertLayout.setLayout(m_pVertLayout);

    m_MainLayout.addWidget(&m_MainHorizLayout); //assign landscape mode grid layout
    m_MainLayout.addWidget(&m_MainVertLayout); //add portrait mode layout

    setCentralWidget(&m_MainLayout);
}

QueueView::~QueueView()
{
}

void QueueView::setupContextMenu()
{
    //set up the listview context menu

    m_pTrackInfoView->setContextMenuPolicy(Qt::ActionsContextMenu); //actions context menu associated with listview

    //set up the listview contextmenu options
    QAction* pPlayAlbumAct = new QAction(tr("Play Album"),this);
    connect(pPlayAlbumAct,SIGNAL(triggered()),SLOT(OnPlayAlbumAction()));
    m_pTrackInfoView->addAction(pPlayAlbumAct);
    QAction* pViewAlbumAct = new QAction(tr("Browse Album"),this);
    connect(pViewAlbumAct,SIGNAL(triggered()),SLOT(OnViewAlbumAction()));
    m_pTrackInfoView->addAction(pViewAlbumAct);
    QAction* pViewArtistAct = new QAction(tr("Browse Artist"),this);
    connect(pViewArtistAct,SIGNAL(triggered()),SLOT(OnViewArtistAction()));
    m_pTrackInfoView->addAction(pViewArtistAct);
}

void QueueView::setPauseOnHSUnplugged(bool state)
{
    //public method for setting the headset unplugged state (i.e., whether playback should be paused when headset is unplugged)
    m_pauseOnHsetUnplugged = state;
}

void QueueView::setupMenubar()
{
    //setup the view menubar
    menuBar()->addAction(tr("FM Transmitter")); //add FM transmitter button to the menubar
}

void QueueView::OnMenubarAction(QAction* action)
{
    //Some action in the QueueView menubar has been selected

    if(action->text()==tr("FM Transmitter"))
    {
        //Request to display the standard FM transmitter configuration dialog
        //(i.e., requests to execute the control panel FM Transmitter plugin)
        emit showFMTransmitterDialog();
    }
}

void QueueView::show()
{
    //overriden show method (handles selection of current playing track etc).

    setListViewState();
    QMainWindow::show(); //call base
}

void QueueView::setListViewState()
{
    //set current listview state
    QModelIndex index;
    QStandardItem* currTrackItem = NULL;

    if(m_currQueuePos>=0)
    {
        currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        if(currTrackItem)
        {
            currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        }

        index = m_pTrackInfoModel->index(m_currQueuePos,0);
        if(index.isValid())
        {
            m_pTrackInfoView->scrollTo(index,QAbstractItemView::PositionAtTop); //center the current item at the top of viewport
            m_pTrackInfoView->setCurrentIndex(index); //set the current row in the queue list
        }
    }
}

void QueueView::OnPlaybackStateChangeRequest(enum pb_state_e requestedState)
{
   //request received via libplayback asking for a playback state change
    qDebug() << "Got here..";
    if(((requestedState==PB_STATE_STOP)||(requestedState==PB_STATE_PLAY)) && !m_userInitiatedPause)
    { //request to move to stopped state, pause current playback
        qDebug() << "Request to pause / resume .. (state:" << requestedState << "), pause state: " << m_userInitiatedPause;
        if(m_pCurrTrack)
        {
            pauseResumePlayback(); //pause playback
            m_playbackStateChangeReq = true; //indicate that playback state has been changed
        }
    }
}

void QueueView::setQueueInitialState(QList<Spotify::Track*> queue, qint32 currQueuePos)
{
    //set the initial state of the play queue view (assign new playqueue, reset members etc.)
    //Param. currQueuePos specifies the current position within the playqueue.

    m_PlayQueue = queue;

    addTracks(m_PlayQueue); //add track data to track list

    //set the queue initial state

    m_currQueuePos = currQueuePos; //initial playback pos
    m_currQueueBuffPos = currQueuePos; //initial buffering track pos
    m_shuffleStateChanged = false;
    m_currTrackScrobbled = false;
    m_userInitiatedPause = false;

    m_lastTimeTrackActivated.start();

    if(m_shuffle)
    { //shuffle mode active; shuffle the current play queue
        shuffleQueue();
    }

    m_pCurrTrack = m_PlayQueue.at(m_currQueuePos); //set current track to first in queue

    emit playQueuePosUpdated(m_currQueuePos);

    if(m_pCurrTrack)
    {
        if(m_pOssoWrapper)
            m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_PLAY); //request to move into playing state
        playTrack(m_pCurrTrack); //initiate track playback
        m_PauseResumeBtn.setEnabled(true);
        m_PauseResumeBtn.setChecked(true);
    }
    else
    {
        clearViewSelection();
    }

    //set the initial state of the progress indicators
    OnTrackBufferedAmountUpdated(0);
    OnPlaybackPositionUpdated(0);
}

void QueueView::clearViewSelection()
{
    //clear the current view selection (i.e., set to state when no track is active)

    QStandardItem* currTrackItem;

    m_AlbumCover.clear();

    m_pTrackInfoView->clearSelection();

    if(m_currQueuePos!=-1)
    {
        currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        if(currTrackItem)
        {
            currTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight
        }
    }

    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);
    m_pProgressSlider->setText("");

    m_pCurrTrack = NULL;
    m_currQueuePos = -1;

    m_PauseResumeBtn.setChecked(false);
    m_PauseResumeBtn.setEnabled(false);
}

void QueueView::OnPlayingState(bool state)
{
    if(state)
    { //playing state
        m_playing = true;
        if(m_update)
            m_pSpectrum->Start();
    }
    else
    { //paused state
        m_playing = false;
        m_pSpectrum->Stop();
    }
}

void QueueView::OnBufferNextTrack()
{
    //invoked when ready to start buffering the following track (track pre-buffering mode)

    Spotify::Error error;
    Spotify::Track* track;

    if(m_shuffle)
    { //shuffle mode active; get next track position from shuffle queue (if queue is not empty)
        if(m_currShuffleBuffPos < m_ShuffleQueue.size()-1)
        {
            m_currShuffleBuffPos++;
            qDebug() << "currShuffleBuffPos: " << m_currShuffleBuffPos;
            m_currQueueBuffPos = m_ShuffleQueue.at(m_currShuffleBuffPos);
            qDebug() << "currQueueBuffPos: " << m_currQueueBuffPos;
        }
        else
        {
           m_currQueueBuffPos = m_PlayQueue.size();
        }
    }

    if((!m_shuffle && (m_currQueueBuffPos < m_PlayQueue.size()-1))||(m_shuffle && (m_currQueueBuffPos < m_PlayQueue.size())))
    {
        if(!m_shuffle)
        {
            m_currQueueBuffPos++;
        }
        track = m_PlayQueue.at(m_currQueueBuffPos);
        if(track)
        {
            m_pSession->setNewTrackStarted(true); //indicate start of new track audio buffer data (in playback thread FIFO)
            error = m_pSession->play(track,true); //initiate buffering of track
            if(error.isError())
            {
                QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
            }
        }
    }
    else
    {
        //end of playqueue reached, set next track pos to end of current track
        m_currQueueBuffPos = -1;
        m_pSession->setNewTrackStarted(false);
        m_pSession->setNextTrackStartPos();
    }
}

void QueueView::loadTrackData(Spotify::Track* track)
{
    //Load track data for specified track (e.g., album cover art)

    QString strCurrTrackDuration, strCurrTrackProgressInfo;
    QTextStream outStream(&strCurrTrackProgressInfo);

    if(track)
    {
        if(track->getAlbum())
        {
            m_pSession->loadImage(track->getAlbum()->getCover(),m_pAlbumCoverContainer); //request to load album coverart
        }
        m_currTrackDuration = track->getDuration().hour()*3600+track->getDuration().minute()*60+track->getDuration().second(); //current track duration (in secs)
        strCurrTrackDuration = track->getDurationAsString();
        outStream << "0:00" << " / " << strCurrTrackDuration;
        m_pProgressSlider->setText(strCurrTrackProgressInfo);
    }
}

void QueueView::addTracks(QList<Spotify::Track*> trackList)
{
    //add tracks from specified list to main listview.
    //Overriden method.

    QPixmap iconMap;
    QString strSubHeader;

    iconMap.load("/opt/qspot/images/icon-type-song-color.png"); //load song (track) type icon from file

    QIcon trackTypeIcon(iconMap);

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    //m_TrackInfo.clear();
    m_pTrackInfoModel->clear();

     for(int trackPos = 0; trackPos < trackList.size(); trackPos++)
     {
             Spotify::Track* currTrack = trackList.at(trackPos);
             QStandardItem* trackInfo = new QStandardItem();
             trackInfo->setEditable(false);
             if(currTrack)
             {
                 strSubHeader = "";
                 trackInfo->setData(trackTypeIcon,ListViewDelegate::iconRole);
                 trackInfo->setData(currTrack->getName(),ListViewDelegate::headerRole);
                 trackInfo->setData(currTrack->getDurationAsString(),ListViewDelegate::subHeaderRightJustifyRole);
                 if(currTrack->getArtist())
                 {
                     strSubHeader = currTrack->getArtist()->getName();
                 }
                 trackInfo->setData(strSubHeader,ListViewDelegate::subHeaderLeftJustifyRole);
                 m_pTrackInfoModel->appendRow(trackInfo);
             }
     }

     setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void QueueView::OnTrackActivated(QModelIndex index)
{
    //A track has been activated (e.g., selected) in the track list
    //Display artist and album cover pic (if available)

    if(m_lastTimeTrackActivated.restart()>=400) //ignore successive (rapid) activations (within 400msecs since last activation)
    {
        if(m_shuffleStateChanged)
            m_shuffleStateChanged=false;

        if(m_shuffle)
        { //re-shuffle the playqueue
            shuffleQueue();
        }

        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(m_currQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item

        QStandardItem* currTrackItem = m_pTrackInfoModel->itemFromIndex(index); //highlight new item text
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole);

        m_currQueuePos = index.row(); //get corresponding position in playqueue
        m_currQueueBuffPos = m_currQueuePos; //set queue buffering position to current queue pos

        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);

        if(m_pCurrTrack) //we have got a valid track instance
        {
            playTrack(m_pCurrTrack);
            emit playQueuePosUpdated(m_currQueuePos);
        }
    }
}

void QueueView::playTrack(Spotify::Track* track)
{
    //request to play (stream) specified track (and load associated track data)

    Spotify::Error error;
    QString strArtist, strAlbum, strTrack, strAlbumArtist;

    if(m_playbackStateChangeReq)
    { //ask for state change
        m_playbackStateChangeReq = false;
        m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_PLAY);
    }
    m_currTrackScrobbled = false;
    m_pSession->resetPlaybackMembers();
    loadTrackData(track);
    m_PauseResumeBtn.setChecked(true); //toggle playing state
    m_userInitiatedPause = false;
    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
    QMaemo5InformationBox::information(this,tr("Buffering"));
    error = m_pSession->play(track); //request to start playback of track
    if(error.isError())
    { //error occured
        QMaemo5InformationBox::information(this,tr("Error starting track playback: ")+error.toString());
    }
    else
    { //set state of pause / resume button
        m_PauseResumeBtn.setChecked(true);
        m_PauseResumeBtn.setEnabled(true);
        strTrack = track->getName();
        if(track->getArtist())
            strArtist = track->getArtist()->getName();
        if(track->getAlbum())
        {
            strAlbum = track->getAlbum()->getName();
            if(track->getAlbum()->getArtist())
                strAlbumArtist = track->getAlbum()->getArtist()->getName();
        }
        emit updateNowPlaying(strTrack,strArtist,strAlbum,strAlbumArtist); //send update now playing request to Last.fm
    }
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
}

void QueueView::OnTrackBufferedAmountUpdated(qint32 bufferedAmount)
{
    //Amount of buffered of the currently loaded / playing track has been updated
    int currPos;

    if(m_update)
    {
        if(m_currQueuePos != m_currQueueBuffPos)
        {
            currPos = m_currSliderWidth; //current track has been completely buffered
        }
        else
        {
            currPos = (bufferedAmount*m_currSliderWidth) / (m_currTrackDuration*1000); //current track has partly been buffered
        }

        m_pProgressSlider->setPos1(currPos); //set curr. slider position
    }
}

void QueueView::OnPlaybackPositionUpdated(qint64 playbackPos)
{
    //Current track playback position updated

    QString strCurrTrackProgressInfo, strCurrPlaybackPos, strCurrTrackDuration;

    if(m_update)
    {

        QTextStream outStream(&strCurrTrackProgressInfo);

        int seconds = ( playbackPos / 1000);
        QTime currPlaybackPos = QTime( 0, 0, 0, 0 );
        currPlaybackPos = currPlaybackPos.addSecs( seconds );

        if(!m_currTrackScrobbled && playbackPos>=240000)
        {//scrobble track in case not already scrobbled (and track has played for at least 240 secs)
           emit scrobbleTrack();
           m_currTrackScrobbled = true;
        }

        if(m_pCurrTrack)
        { //set track progress info in format current playback position / total track duration
            if(currPlaybackPos.hour()>0)
                strCurrPlaybackPos = currPlaybackPos.toString("h:mm:ss"); //position exceeds 60 mins
            else
                strCurrPlaybackPos = currPlaybackPos.toString("m:ss");
            if(m_pCurrTrack->getDuration().hour()>0) //duration longer than 60 mins
                strCurrTrackDuration = m_pCurrTrack->getDuration().toString("h:mm:ss");
            else
                strCurrTrackDuration = m_pCurrTrack->getDuration().toString("m:ss");
            outStream << strCurrPlaybackPos << " / " << strCurrTrackDuration;
            m_pProgressSlider->setText(strCurrTrackProgressInfo);
        }

        int currPos = (playbackPos*m_currSliderWidth) / (m_currTrackDuration*1000);


        m_pProgressSlider->setPos2(currPos); //set curr. slider position
    }
}

void QueueView::OnPauseResumeToggle()
{
    //Play / Pause button has bee clicked (toggled)

    if(m_pCurrTrack) //some track has been activated
    {
        if(m_playbackStateChangeReq)
        { //ask for state change
            m_playbackStateChangeReq = false;
            m_pOssoWrapper->requestPlaybackStateChange(PB_STATE_PLAY); //notify via libplayback that we are entering playing state
        }
        if(!m_PauseResumeBtn.isChecked()) //entering paused state
        {
            qDebug() << "User paused playback..";
            m_userInitiatedPause = true; //indicate that pausing of current track has been requested by user (used to ensure playback is not autoresumed by system upon notification from libplayback
        }
        else //entering playing state
        {
            m_userInitiatedPause = false;
        }
        m_pSession->pauseResumePlayback(); //pause / resume playback of current track
    }
}

void QueueView::pauseResumePlayback()
{
    //request to pause / resume music playback (and set the UI to corresponding state)

    m_pSession->pauseResumePlayback(); //pause / resume playback

    if(m_playing)
    {
        m_PauseResumeBtn.setChecked(false); //set to paused state
    }
    else
    {
        m_PauseResumeBtn.setChecked(true); //set to playing state
    }
}


void QueueView::OnNextTrackBtn()
{
    //move to next track in play queue

    Spotify::Error error;

    qint32 oldQueuePos = m_currQueuePos;

    if(m_shuffleStateChanged)
        m_shuffleStateChanged=false;

    if(m_shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos < m_ShuffleQueue.size()-1)
        {
            m_currShufflePos++;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
        else
        { //no tracks left in shuffle queue
            m_currQueuePos = m_PlayQueue.size()-1;
            m_currQueueBuffPos = m_PlayQueue.size()-1;
        }
    }

    if((!m_shuffle && (m_currQueuePos < m_PlayQueue.size()-1))||(m_shuffle && (m_currQueuePos < m_PlayQueue.size())))
    {
        if(!m_shuffle)
        {
            m_currQueuePos++;
        }
        m_currQueueBuffPos = m_currQueuePos;
        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(oldQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item
        QStandardItem* currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
              m_pTrackInfoView->setCurrentIndex(index);
            playTrack(m_pCurrTrack);
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
}

void QueueView::OnPrevTrackBtn()
{
    //move to previous track in play queue

    Spotify::Error error;

    qint32 oldQueuePos = m_currQueuePos;

    if(m_shuffleStateChanged)
        m_shuffleStateChanged=false;

    if(m_shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos > 0)
        {
            m_currShufflePos--;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
    }

    if((!m_shuffle && (m_currQueuePos > 0))||(m_shuffle && (m_currQueuePos>0)))
    {
        if(!m_shuffle)
        {
            m_currQueuePos--;
        }
        m_currQueueBuffPos = m_currQueuePos;
        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(oldQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item
        QStandardItem* currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
              m_pTrackInfoView->setCurrentIndex(index);
            playTrack(m_pCurrTrack);
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
}

void QueueView::OnTrackPlaybackFinished()
{
   //Playback of current track has finished
   //move to next track in play queue, if possible

    qDebug() << "Playback finished..";

    QString strTrack, strArtist, strAlbum, strAlbumArtist;

    Spotify::Error error;

    qint32 oldQueuePos = m_currQueuePos; //save current (old) queue position

    if(!m_currTrackScrobbled && m_currTrackDuration>=30)
    {//scrobble track in case not already scrobbled (and track duration is at least 30secs)
       emit scrobbleTrack();
    }
    else
    {
        m_currTrackScrobbled = false; //reset current track scrobbled status
    }

    if(m_shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos<m_ShuffleQueue.size()-1)
        {
            m_currShufflePos++;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
        else
        { //no tracks left in shuffle queue
            m_currQueuePos = m_PlayQueue.size();
        }
    }

    if((!m_shuffle && (m_currQueuePos < m_PlayQueue.size()-1))||(m_shuffle && (m_currQueuePos < m_PlayQueue.size())))
    {
        if(!m_shuffle) //normal playback mode
            m_currQueuePos++;
        QStandardItem* prevTrackItem = m_pTrackInfoModel->item(oldQueuePos); //get previously selected item
        if(prevTrackItem)
            prevTrackItem->setData(false,ListViewDelegate::highlightItemTextRole); //remove highlight from prev. selected item
        QStandardItem* currTrackItem = m_pTrackInfoModel->item(m_currQueuePos);
        currTrackItem->setData(true,ListViewDelegate::highlightItemTextRole); //highlight new item text
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
                m_pTrackInfoView->setCurrentIndex(index);
            loadTrackData(m_pCurrTrack); //load track info
            strTrack = m_pCurrTrack->getName();
            if(m_pCurrTrack->getArtist())
                strArtist = m_pCurrTrack->getArtist()->getName();
            if(m_pCurrTrack->getAlbum())
            {
                strAlbum = m_pCurrTrack->getAlbum()->getName();
                if(m_pCurrTrack->getAlbum()->getArtist())
                    strAlbumArtist = m_pCurrTrack->getAlbum()->getArtist()->getName();
            }
            emit updateNowPlaying(strTrack,strArtist,strAlbum,strAlbumArtist); //send update now playing request to Last.fm
            if(!m_pSession->getPreBufferingActive()) //pre-buffering is not active, buffering / playback should be initiated
            {
                m_pSession->resetPlaybackMembers();
                error = m_pSession->play(m_pCurrTrack); //initiate track playback
                if(error.isError())
                {
                    QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
                }
            }
            else
            { //track pre-buffering mode active; reset playback position and playing state
                m_pSession->setPlaybackPos(0); //reset current playback position
                if(m_currQueueBuffPos==-1)
                    m_pProgressSlider->setPos1(m_currSliderWidth); //set buffering progress to 100% (final track buffered completely)
                if(m_currQueueBuffPos != oldQueuePos)
                {
                    if(m_shuffleStateChanged)
                    {//shuffle state has been changed (enabled / disabled), and possible pre-buffered track data should be discarded
                        m_shuffleStateChanged = false;
                        m_currQueueBuffPos = m_currQueuePos; //buffering the currently playing track
                        m_currShuffleBuffPos = m_currShufflePos;
                        m_pSession->resetPlaybackMembers();
                        error = m_pSession->play(m_pCurrTrack); //initiate track playback
                        if(error.isError())
                        {
                            QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
                        }
                    }
                }
            }
            emit playQueuePosUpdated(m_currQueuePos); //emit pos updated
        }
    }
    else
    {
        //end of playback queue reached
        clearViewSelection();
        emit playQueuePosUpdated(-1); //emit pos updated (indicates that playqueue has reached the end)
    }
}

void QueueView::OnAlbumCoverLoaded()
{
    //Album coverart has been loaded (for currently selected track),
    //assign it to the album coverart container.

    if(m_pAlbumCoverContainer)
        m_AlbumCover.setPixmap(QPixmap::fromImage(m_pAlbumCoverContainer->getImage()).scaled(QUEUEVIEW_COVERART_WIDTH_LANDSCAPE,QUEUEVIEW_COVERART_HEIGHT_LANDSCAPE));
}

void QueueView::OnShuffleToggle()
{
    //toggle the shuffle state (on / off)

    if(m_currQueuePos!=m_currQueueBuffPos)
        m_shuffleStateChanged = true; //indicate that current shuffle state has changed (enabled / disabled)

    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry

    if(m_shuffle)
    {
        m_shuffle = false;
        setWindowTitle(tr("Now Playing"));
    }
    else
    {
        shuffleQueue(); //shuffle the current playqueue
        m_shuffle = true;
        if (screenGeometry.width() > screenGeometry.height())
        {
            //landscape mode
            setWindowTitle(tr("Now Playing") + " - " + tr("Shuffle"));
        }
        else
        { //portrait mode
            setWindowTitle(tr("Shuffle"));
        }
    }
}

void QueueView::shuffleQueue()
{
    //Shuffle the current playback queue (the m_ShuffleQueue list contains the shuffled queue positions).
    //The shuffle is implemented using the Fisher-Yates (Knuth's) algorithm.

    qsrand(QTime::currentTime().msec()); //initialise the random number seed

    m_ShuffleQueue.clear();

    //initialise the position indexes in the shuffle queue
    for (int pos=0; pos<m_PlayQueue.size(); pos++)
    {
        m_ShuffleQueue.append(pos);
    }

    for(int i = m_ShuffleQueue.size()-1; i>=0; i--)
    {
        qint32 j = qrand() % (i+1);
        if(i==j)
            continue;
        m_ShuffleQueue.swap(j,i); //swap list elements
    }

    m_currShufflePos = -1;
    m_currShuffleBuffPos = -1;
}

void QueueView::keyPressEvent(QKeyEvent *evt)
{
    //Overriden keyPressEvent handler

    if(evt->key()==Qt::Key_Left)
    {
        close(); //close the view
        evt->accept();
    }
    else
    {
        QMainWindow::keyPressEvent(evt);
    }
}


void QueueView::OnPlayAlbumAction()
{
    //request to play the album of currently selected track

    Spotify::Track* track = m_PlayQueue.at(m_pTrackInfoView->currentIndex().row());

    if(track)
    {
        if(track->getAlbum())
        {
            if(m_pSession->browse(track->getAlbum(),&m_CallerID)) //request to load album tracks (browse album)
            {
                m_pTrackInfoView->setDisabled(true);
                QMaemo5InformationBox::information(this,tr("Loading album tracks"));
                setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true);
            }
        }
    }
}

void QueueView::OnAlbumBrowseReady(Spotify::AlbumBrowser* browser, qint32 callerID)
{
    //notification that album browse request completed

    Spotify::Error error;

    if(callerID==m_CallerID)
    {
        if(browser)
        {
            error = sp_albumbrowse_error(browser->getAlbumBrowser());
            if(error.isError())
            {
                //error occured (album browse request failed)
                QMaemo5InformationBox::information(this,tr("Fetching of album tracks failed: ") + error.toString());
            }
            else
            {
                if(!browser->load())
                {
                    QMaemo5InformationBox::information(this,tr("Failed to load tracks!"));
                }
                browser->sortByDiscIndex(); //sort tracks in album order
                emit playQueueSet(browser->getTracks(false),0); //only add available (playable) album tracks to the playqueue
                setListViewState();
            }
        }
    }
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
    m_pTrackInfoView->setDisabled(false);
}

void QueueView::OnViewAlbumAction()
{
    //Request to show Album View for currently selected album.

    Spotify::Track* track = m_PlayQueue.at(m_pTrackInfoView->currentIndex().row());

    if(track)
    {
        if(track->getAlbum())
        {
            emit showAlbumView(track->getAlbum());
        }
    }
}

void QueueView::OnViewArtistAction()
{
    //Request to show Artist View for currently selected artist.

    Spotify::Track* track = m_PlayQueue.at(m_pTrackInfoView->currentIndex().row());

    if(track)
    {
        if(track->getArtist())
        {
            emit showArtistView(track->getArtist());
        }
    }
}

void QueueView::OnOrientationChanged()
{
    //Device orientation changed (display rotation) notification

    QRect screenGeometry = QApplication::desktop()->screenGeometry(); // get current screen geometry
    if (screenGeometry.width() > screenGeometry.height())
    {
        //switch to landscape mode
        if(m_shuffle)
        {
            setWindowTitle(tr("Now Playing") + " - " + tr("Shuffle"));
        }
        m_pVertLayout->removeWidget(&m_AlbumCover);
        m_pVertLayout->removeWidget(m_pTrackInfoView);
        m_pPlayerControlsVert->removeWidget(&m_PrevBtn);
        m_pPlayerControlsVert->removeWidget(&m_PauseResumeBtn);
        m_pPlayerControlsVert->removeWidget(&m_NextBtn);
        m_pPlayerControlsVert->removeWidget(&m_ShuffleBtn);
        m_pPlayerControlsVert->removeWidget(m_pSpectrum);
        m_pProgressLayoutVert->removeWidget(m_pProgressSlider);

        m_pHorizGridLayout->addWidget(&m_AlbumCover,0,0,7,1,Qt::AlignTop);
        m_pHorizGridLayout->addWidget(m_pTrackInfoView,0,1,7,1);
        m_pPlayerControlsHoriz->setSpacing(0);
        m_pPlayerControlsHoriz->addWidget(&m_PrevBtn,0,0,1,1,Qt::AlignRight);
        m_pPlayerControlsHoriz->addWidget(&m_PauseResumeBtn,0,1,1,1,Qt::AlignLeft);
        m_pPlayerControlsHoriz->addWidget(&m_NextBtn,0,2,Qt::AlignLeft);
        m_pPlayerControlsHoriz->setColumnStretch(2,6);
        m_pPlayerControlsHoriz->addWidget(&m_ShuffleBtn,0,3,1,1,Qt::AlignLeft);
        m_pPlayerControlsHoriz->setColumnStretch(3,5);
        m_pPlayerControlsHoriz->addWidget(m_pProgressSlider,0,4,1,1);
        m_currSliderWidth = SLIDER_WIDTH_LANDSCAPE;
        m_pProgressSlider->setSize(m_currSliderWidth,SLIDER_HEIGHT);
        m_pPlayerControlsHoriz->setColumnMinimumWidth(4,370);
        m_pPlayerControlsHoriz->setColumnStretch(4,10);
        m_pPlayerControlsHoriz->addWidget(m_pSpectrum,0,5,1,1,Qt::AlignRight);
        m_MainLayout.setCurrentIndex(LANDSCAPE_ORIENTATION); //active landscape layout
    }
    else
    {
        //switch to portrait mode
        if(m_shuffle)
        {
            setWindowTitle(tr("Shuffle"));
        }
        m_pHorizGridLayout->removeWidget(&m_AlbumCover);
        m_pHorizGridLayout->removeWidget(m_pTrackInfoView);
        m_pPlayerControlsHoriz->removeWidget(&m_PrevBtn);
        m_pPlayerControlsHoriz->removeWidget(&m_PauseResumeBtn);
        m_pPlayerControlsHoriz->removeWidget(&m_NextBtn);
        m_pPlayerControlsHoriz->removeWidget(&m_ShuffleBtn);
        m_pPlayerControlsHoriz->removeWidget(m_pProgressSlider);
        m_pPlayerControlsHoriz->removeWidget(m_pSpectrum);

        m_pVertLayout->addWidget(&m_AlbumCover,0,0,1,1,Qt::AlignCenter);
        m_pVertLayout->addWidget(m_pTrackInfoView,1,0,1,1);
        m_pPlayerControlsVert->setSpacing(0);
        m_pPlayerControlsVert->addWidget(&m_PrevBtn,0,0,1,1,Qt::AlignRight);
        m_pPlayerControlsVert->addWidget(&m_PauseResumeBtn,0,1,1,1,Qt::AlignLeft);
        m_pPlayerControlsVert->addWidget(&m_NextBtn,0,2,Qt::AlignLeft);
        m_pPlayerControlsVert->setColumnStretch(2,6);
        m_pPlayerControlsVert->addWidget(&m_ShuffleBtn,0,3,1,1,Qt::AlignLeft);
        m_pPlayerControlsVert->setColumnStretch(3,5);
        m_pPlayerControlsVert->addWidget(m_pSpectrum,0,4,1,1,Qt::AlignRight);
        m_currSliderWidth = SLIDER_WIDTH_PORTRAIT;
        m_pProgressSlider->setSize(m_currSliderWidth,SLIDER_HEIGHT);
        m_pProgressLayoutVert->addWidget(m_pProgressSlider,0,0,1,-1);
        m_MainLayout.setCurrentIndex(PORTRAIT_ORIENTATION); //activate portrait layout
    }

    if(m_pCurrTrack)
    {
        OnTrackBufferedAmountUpdated(m_pSession->getBufferingPos()); //force update of track buffering position
        OnPlaybackPositionUpdated(m_pSession->getPlaybackPos()); //force update of playback position
    }
}

void QueueView::OnDisplayStateChanged(bool state)
{
    //display state changed notification (true=ON, false=OFF)
    if(state)
    { //display on
        m_update = true; //allow UI updates
        if(m_pCurrTrack)
        {
            if(m_playing)
                m_pSpectrum->Start();
            OnTrackBufferedAmountUpdated(m_pSession->getBufferingPos()); //force update of track buffering position
            OnPlaybackPositionUpdated(m_pSession->getPlaybackPos()); //force update of playback position
        }
    }
    else
    { //display off
        m_update = false; //no UI updates
        m_pSpectrum->Stop();
    }
}

void QueueView::OnHalButtonPressed(const QString& value)
{
    //button pressed notification (condition) received from HAL (device)

    if(m_pCurrTrack) //track currently loaded
    {
        if(value=="pause-cd"||value=="play-cd")
        {
            //pause / resume current playback (request from e.g., BT headset)
            pauseResumePlayback();
        }
        else if(value=="next-song")
        {
            //skip to next track
            OnNextTrackBtn();
        }
        else if(value=="previous-song")
        {
            //skip to previous track
            OnPrevTrackBtn();
        }
    }
}

void QueueView::OnHeadsetRemoved()
{
    //Headset removed notification received from Hal device manager    
    if(m_pauseOnHsetUnplugged && m_playing)
    { //in case music playback is in progress (and option to pause on headset unplugged active); pause music playback
        pauseResumePlayback();
    }
}

/*void QueueView::OnProgressSliderClicked(int pos)
{
    //progress slider has been clicked (pos=clicked position)
    double seekPos;

    if(m_pCurrTrack)
    {
        //we have a track loaded, calculate corresponding position within the track (in msecs)
        seekPos = ((pos*m_currTrackDuration*1000) / m_currSliderWidth);
        qDebug() << "Request to seek to position " << seekPos << " msecs within track";
        if(m_currQueuePos != m_currQueueBuffPos)
        {
            //we are currently buffering and playing different tracks
              m_currQueueBuffPos = m_currQueuePos;
              playTrack(m_pCurrTrack);
        }
        m_pSession->seek(seekPos); //seek to specified position within the stream
        m_pProgressSlider->setPos1(pos);
        m_pProgressSlider->setPos2(pos);
    }
}
*/

