//PlaybackThread.cpp: Implementation of the PlaybackThread class.

#include "PlaybackThread.h"
#include <QDebug>
#include <QMutex>
#include <QByteArray>

#define SAMPLE_RATE 44100
#define CHANNELS    2

PlaybackThread::PlaybackThread(QObject* parent)
    : QThread(parent)
{

    //init members
    m_exitThread = false;
    m_FIFO.clear();
    m_FIFOSize = 0;
    m_isPaused = false;
    m_playbackPaused = false;
    m_endOfTrack = false;
    m_minBufferSize = 30;
    m_preBufActive = true;
    m_nextTrackStartPos = -1;
    m_trackStartPosList.clear();
    m_newTrackStarted = false;

    //init pulseaudio interface

    pa_sample_spec ss;
    pa_channel_map map;
    pa_buffer_attr buffer;

    ss.format = PA_SAMPLE_S16NE;
    ss.channels = CHANNELS;
    ss.rate = SAMPLE_RATE;

    m_pAudioStream = NULL;

    buffer.maxlength = pa_usec_to_bytes(750000, &ss);
    buffer.tlength = -1;
    buffer.prebuf = -1;
    buffer.minreq = -1;

    pa_channel_map_init_stereo(&map);

    m_pAudioStream = pa_simple_new(NULL,"QSpot", PA_STREAM_PLAYBACK, NULL, "QSpot",&ss,&map, &buffer,NULL);
}

PlaybackThread::~PlaybackThread()
{
    //cleanup
    int err=0;
    m_audio_mutex.lock(); //lock mutex
    pa_simple_flush(m_pAudioStream,&err); //flush playback buffer
    pa_simple_free(m_pAudioStream); //close and free pulseaudio connection
    m_audio_mutex.unlock(); //release lock
}

void PlaybackThread::stop()
{
    //stop the thread execution
    m_audio_mutex.lock(); //lock mutex
    m_exitThread = true;
    m_audio_mutex.unlock(); //release lock
}

void PlaybackThread::run()
{
   //thread main loop

    QByteArray buffer;
    qint32 nextTrackStartPos;
    int error = 0;

    forever
    {
        m_audio_mutex.lock(); //lock mutex

        if(m_exitThread)
        {
            m_exitThread = false;
            m_audio_mutex.unlock(); //release lock
            break;
        }

        if(m_preBufActive)
        { //track pre-buffering mode is active
            if(!m_trackStartPosList.isEmpty())
            {
                if(m_trackStartPosList.first()==0)
                {
                    m_trackStartPosList.removeFirst();
                    emit playbackFinished(); //playback of the current track has finished
                }
            }
        }
        else
        {
            if(m_endOfTrack && m_FIFO.isEmpty())
            { //end of track reached (track pre-buffering not active), emit notification signal
                m_endOfTrack = false;
                emit playbackFinished();
            }
        }

        if(!m_FIFO.isEmpty() && !m_playbackPaused)
        { //data in audio FIFO and playback has not been paused
            emit playing(true);
            buffer = m_FIFO.takeFirst();
            m_FIFOSize -= buffer.size();
            if(m_preBufActive && !m_trackStartPosList.isEmpty())
            {
                if(m_trackStartPosList.first()>0)
                {
                    nextTrackStartPos = m_trackStartPosList.takeFirst();
                    nextTrackStartPos--;
                    m_trackStartPosList.prepend(nextTrackStartPos);
                }
            }

            if(m_FIFOSize<=m_minBufferSize*SAMPLE_RATE*sizeof(int16_t)*CHANNELS && m_isPaused)
            { //request to resume buffering (less than m_minBufferSize secs of data in buffer)
                m_isPaused = false;
                emit resumeBuffering();
            }

            m_audio_mutex.unlock();

            qint64 duration = (buffer.size()*1000)/(SAMPLE_RATE*sizeof(int16_t)*CHANNELS); //buffer duration (msecs)

            pa_simple_write(m_pAudioStream, buffer.data(),buffer.size(),&error); //write audio stream data to pulseaudio server

            emit playbackPositionUpdated(duration);

        }
        else
        {
          emit playing(false);
          m_audio_mutex.unlock(); //release lock
        }

        msleep(20); //sleep for a while
    }
}


QMutex* PlaybackThread::getMutex()
{
    return &m_audio_mutex;
}

void PlaybackThread::FIFOAppend(const QByteArray& buffer)
{
    m_FIFO.append(buffer);
}

void PlaybackThread::clearFIFO()
{
    m_FIFO.clear();
}

qint32 PlaybackThread::getFIFOSize()
{
    return m_FIFOSize;
}

void PlaybackThread::setFIFOSize(qint32 size)
{
    //set FIFO size in bytes
    m_FIFOSize = size;
}

void PlaybackThread::incFIFOSize(qint32 size)
{
    m_FIFOSize += size;
}

void PlaybackThread::setBufferingPaused(bool paused)
{
    m_isPaused = paused;
}

void PlaybackThread::setPlaybackPaused(bool paused)
{
    m_playbackPaused = paused;
}

bool PlaybackThread::getPlaybackPaused()
{
    return m_playbackPaused;
}

void PlaybackThread::setEndOfTrack(bool endOfTrack)
{
    m_endOfTrack = endOfTrack;
}

void PlaybackThread::setMinBufferSize(qint32 secs)
{
    //set min. FIFO buffer size (in secs)
    m_minBufferSize = secs;
}

qint32 PlaybackThread::getMinBufferSize()
{
   return m_minBufferSize;
}

void PlaybackThread::setPreBufActive(bool active)
{
    m_preBufActive = active;
}

void PlaybackThread::setNextTrackStartPos()
{
    m_trackStartPosList.append(m_FIFO.count());
}

void PlaybackThread::setNextTrackStartPos(qint32 pos)
{
    //Overriden method.
    //Assigns the starting position of new track audio data in the buffer.
    //If the pos param is -1, the list is cleared.
    if(pos!=-1)
        m_trackStartPosList.append(pos);
    else
        m_trackStartPosList.clear();
}

void PlaybackThread::setNewTrackStarted(bool started)
{
    m_newTrackStarted = started;
}

bool PlaybackThread::getNewTrackStarted()
{
    return m_newTrackStarted;
}
