//TrackView.cpp: Implementation of the TrackView class (default playlist / search result / album view).

#include "TrackView.h"

TrackView::TrackView(Spotify::Session* pSession, QWidget* parent) :
        QMainWindow(parent)
{
        //constructs the UI track view

	setAttribute(Qt::WA_Maemo5StackedWindow);

        m_pSession = pSession; //set the Spotify session instance

        m_pQueueView = new QueueView(m_pSession,this);

        connect(m_pQueueView,SIGNAL(playQueuePosUpdated(qint32)),SLOT(OnPlayQueuePosUpdated(qint32)));
        connect(&m_TrackInfo,SIGNAL(itemActivated(QTableWidgetItem*)),SLOT(OnTrackActivated(QTableWidgetItem*)));
        connect(m_TrackInfo.horizontalHeader(),SIGNAL(sectionClicked(int)),SLOT(OnHeaderClicked(int)));

        //set table view properties
        m_TrackInfo.setColumnCount(3);
        m_TrackInfo.setColumnWidth(0,300);
        m_TrackInfo.setColumnWidth(1,200);
        m_TrackInfo.setColumnWidth(2,200);
        m_TrackInfo.setSelectionMode(QAbstractItemView::SingleSelection);
        m_TrackInfo.setShowGrid(false); //no grid should be displayed
        m_TrackInfo.verticalHeader()->hide(); //hide default vertical header
        m_TrackInfo.horizontalScrollBar()->hide();
        m_TrackInfo.horizontalHeader()->setStretchLastSection(true);
        m_TrackInfo.horizontalHeader()->setResizeMode(0, QHeaderView::Stretch); //  column 0
        m_TrackInfo.horizontalHeader()->setClickable(true);


        m_Layout.addWidget(&m_TrackInfo);

        m_MainLayout.setLayout(&m_Layout);

	setCentralWidget(&m_MainLayout);
}


TrackView::~TrackView()
{

}

void TrackView::setPlaylist(Spotify::Playlist* pPlaylist)
{
    //set the playlist for which tracks should be displayed


	if(pPlaylist)
        {
            m_pPlaylist = pPlaylist;

            m_currSortOrder = SORT_ORIGINAL;
            m_currSortCol = -1;

            setWindowTitle(pPlaylist->getName());

            addTracks(m_pPlaylist); //add track data to track list
	}
}

void TrackView::setSearchResults(Spotify::Search* pSearch)
{
    //Set the search results to be displayed

    m_pPlaylist = NULL;

    if(pSearch)
    {
        setWindowTitle(tr("Search results: ") + pSearch->getSearchQuery());

        m_currSortOrder = SORT_ORIGINAL;
        m_currSortCol = -1;

        m_TrackList = pSearch->getTracks();
	//sort track instance first by artist, then by album, and finally by track
	qSort(m_TrackList.begin(),m_TrackList.end(),trackLessThanCaseInsensitive); 
        addTracks(m_TrackList);
    }
}


void TrackView::OnTrackActivated(QTableWidgetItem *trackItem)
{
    //A track has been activated (e.g., selected) in the track list

    Spotify::Error error;
    Spotify::Track* track;

    if(trackItem!=NULL)
    {
        int trackPos = trackItem->data(Qt::UserRole).toInt();

        if(m_pPlaylist) //track from playlist
            track = m_pPlaylist->getTrack(trackPos);
        else //search result
        {
            qDebug() << trackPos;
            track = m_TrackList.at(trackPos);
        }

        if(track) //we have got a valid track instance
        {
            m_TrackInfo.setEnabled(false); //disable table temporarily
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(track); //request to start playback of track
            if(error.isError())
            { //error occured
                QMaemo5InformationBox::information(this,tr("Error starting track playback: ") + error.toString());
            }
            else
            {
                setPlayQueue(); //set new playqueue, and move to play queue view
            }
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            m_TrackInfo.setEnabled(true); //re-enable the table view
        }
    }
}


void TrackView::setPlayQueue()
{
    //Set playqueue (i.e., add all tracks from selected track to last track in list to playqueue)

    QList<Spotify::Track*> PlayQueue;
    Spotify::Track* track;

    for(int pos = m_TrackInfo.currentRow(); pos < m_TrackInfo.rowCount(); pos++)
    {
        QTableWidgetItem* currTrackItem = m_TrackInfo.item(pos,0);

         int trackPos = currTrackItem->data(Qt::UserRole).toInt();
         if(m_pPlaylist) //track from playlist
             track = m_pPlaylist->getTrack(trackPos);
         else //search result etc.
             track = m_TrackList.at(trackPos);
         if(track)
         {
             PlayQueue.append(track);
         }
    }

    emit playQueueSet(PlayQueue); //signal that playqueue has been set / updated

    if(m_pQueueView)
    {
        m_pQueueView->setNowPlaying(PlayQueue,0,true);
        m_pQueueView->show();
    }
}

void TrackView::OnPlayQueuePosUpdated(qint32 pos)
{
    //emit notification that position in playqueue has been updated
    emit playQueuePosUpdated(pos);
}

void TrackView::OnHeaderClicked(int logicalIndex)
{
    //A header has been clicked, perform sorting of tracks
    //based on selected column and sort order

    if(logicalIndex!=m_currSortCol)
    {
        m_currSortOrder = SORT_ASCENDING;
        m_currSortCol = logicalIndex;
        if(m_pPlaylist) //playlist mode
        {
            if(m_currSortCol == COLUMN_TRACK)
                m_pPlaylist->sortByTracks(true);
            else if(m_currSortCol == COLUMN_ARTIST)
                m_pPlaylist->sortByArtists(true);
            else if(m_currSortCol == COLUMN_ALBUM)
                m_pPlaylist->sortByAlbums(true);
            addTracks(m_pPlaylist); //refresh playlist view
        }
        else
        {
            //search mode
            if(m_currSortCol == COLUMN_TRACK)
                qSort(m_TrackList.begin(),m_TrackList.end(),tracksByNameAsc);
            else if(m_currSortCol == COLUMN_ARTIST)
                qSort(m_TrackList.begin(),m_TrackList.end(),tracksByArtistNameAsc);
            else if(m_currSortCol == COLUMN_ALBUM)
                qSort(m_TrackList.begin(),m_TrackList.end(),tracksByAlbumNameAsc);
            addTracks(m_TrackList); //refresh track view
        }
    }
    else
    { //same column clicked, change the sort order
        if(m_currSortOrder==SORT_ORIGINAL)
            m_currSortOrder=SORT_ASCENDING;
        else if(m_currSortOrder==SORT_ASCENDING)
            m_currSortOrder=SORT_DECENDING;
        else if(m_currSortOrder==SORT_DECENDING)
            m_currSortOrder=SORT_ORIGINAL;
        if(m_pPlaylist) //playlist mode
        {
            if(m_currSortOrder==SORT_ORIGINAL)
            {
                m_pPlaylist->restoreOrignalCollection();
            }
            else
            {
                if(m_currSortCol==COLUMN_TRACK)
                {
                    if(m_currSortOrder==SORT_ASCENDING)
                        m_pPlaylist->sortByTracks(true);
                    else
                        m_pPlaylist->sortByTracks(false);
                }
                else if(m_currSortCol==COLUMN_ARTIST)
                {
                    if(m_currSortOrder==SORT_ASCENDING)
                        m_pPlaylist->sortByArtists(true);
                    else
                        m_pPlaylist->sortByArtists(false);
                }
                else if(m_currSortCol==COLUMN_ALBUM)
                {
                    if(m_currSortOrder==SORT_ASCENDING)
                        m_pPlaylist->sortByAlbums(true);
                    else
                        m_pPlaylist->sortByAlbums(false);
                }
            }
            addTracks(m_pPlaylist);
        }
        else
        { //search mode
            if(m_currSortOrder==SORT_ORIGINAL)
            {
                qSort(m_TrackList.begin(),m_TrackList.end(),trackLessThanCaseInsensitive);
            }
            else
            {
                if(m_currSortCol==COLUMN_TRACK)
                {
                    if(m_currSortOrder==SORT_ASCENDING)
                       qSort(m_TrackList.begin(),m_TrackList.end(),tracksByNameAsc);
                    else
                        qSort(m_TrackList.begin(),m_TrackList.end(),tracksByNameDesc);
                }
                else if(m_currSortCol==COLUMN_ARTIST)
                {
                    if(m_currSortOrder==SORT_ASCENDING)
                        qSort(m_TrackList.begin(),m_TrackList.end(),tracksByArtistNameAsc);
                    else
                        qSort(m_TrackList.begin(),m_TrackList.end(),tracksByArtistNameDesc);
                }
                else if(m_currSortCol==COLUMN_ALBUM)
                {
                    if(m_currSortOrder==SORT_ASCENDING)
                       qSort(m_TrackList.begin(),m_TrackList.end(),tracksByAlbumNameAsc);
                    else
                        qSort(m_TrackList.begin(),m_TrackList.end(),tracksByAlbumNameDesc);
                }
            }
            addTracks(m_TrackList);
        }
    }
}

void TrackView::addTracks(Spotify::Playlist* playlist)
{
    //add playlist tracks to main tableview

    QStringList headers;

    headers << tr("Track") << tr("Artist") << tr("Album");

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    m_TrackInfo.clear();

    m_TrackInfo.setHorizontalHeaderLabels(headers);

    int row=0;

    if(playlist)
    {
        m_TrackInfo.setRowCount(playlist->getTrackCount(false)); //exclude unavailable tracks

        for(int trackPos = 0; trackPos < playlist->getNumTracks(); trackPos++)
        {
                    Spotify::Track* currTrack = playlist->getTrack(trackPos);
                    if(currTrack)
                    {
                            if(currTrack->isAvailable()) //track is available (playable)
                            {
                                QTableWidgetItem* trackName = new QTableWidgetItem();
                                QTableWidgetItem* artistName = new QTableWidgetItem();
                                QTableWidgetItem* albumName = new QTableWidgetItem();

                                trackName->setText(currTrack->getName());
                                trackName->setFlags(Qt::ItemIsSelectable|Qt::ItemIsEnabled);
                                trackName->setData(Qt::UserRole,trackPos); //associate position in track instances list with list item
                                if(currTrack->getArtist())
                                    artistName->setText(currTrack->getArtist()->getName());
                                artistName->setFlags(Qt::ItemIsSelectable|Qt::ItemIsEnabled);
                                artistName->setData(Qt::UserRole,trackPos);
                                if(currTrack->getAlbum())
                                    albumName->setText(currTrack->getAlbum()->getName());
                                albumName->setFlags(Qt::ItemIsSelectable|Qt::ItemIsEnabled);
                                albumName->setData(Qt::UserRole,trackPos);
                                m_TrackInfo.setItem(row,0,trackName);
                                m_TrackInfo.setItem(row,1,artistName);
                                m_TrackInfo.setItem(row,2,albumName);
                                m_TrackInfo.setRowHeight(row,70); //set default row size to 70px
                                row++;
                            }
                    }
            }

        if(m_TrackInfo.rowCount()>0)
            m_TrackInfo.setCurrentCell(0,0);
    }


    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void TrackView::addTracks(QList<Spotify::Track*> trackList)
{
    //add tracks from specified list to main tableview.
    //Overriden method.

    QStringList headers;

    headers << "Track" << "Artist" << "Album";

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    m_TrackInfo.clear();

    m_TrackInfo.setHorizontalHeaderLabels(headers);

    int row=0;

     m_TrackInfo.setRowCount(trackList.size()); //exclude unavailable tracks

     for(int trackPos = 0; trackPos < trackList.size(); trackPos++)
     {
             Spotify::Track* currTrack = trackList.at(trackPos);
             if(currTrack)
             {
                 QTableWidgetItem* trackName = new QTableWidgetItem();
                 QTableWidgetItem* artistName = new QTableWidgetItem();
                 QTableWidgetItem* albumName = new QTableWidgetItem();

                 trackName->setText(currTrack->getName());
                 trackName->setData(Qt::UserRole,trackPos); //associate position in track instances list with list item
                 if(currTrack->getArtist())
                     artistName->setText(currTrack->getArtist()->getName());
                  artistName->setData(Qt::UserRole,trackPos);
                 if(currTrack->getAlbum())
                     albumName->setText(currTrack->getAlbum()->getName());
                  albumName->setData(Qt::UserRole,trackPos);
                 m_TrackInfo.setItem(row,0,trackName);
                 m_TrackInfo.setItem(row,1,artistName);
                 m_TrackInfo.setItem(row,2,albumName);
                 m_TrackInfo.setRowHeight(row,70); //set default row height to 70px
                 row++;
             }
     }
	 
	  if(m_TrackInfo.rowCount()>0)
            m_TrackInfo.setCurrentCell(0,0);

     setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

