#include "mainwindow.h"
#include "ui_mainwindow.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    radio=0;
    isFullScreen=false;
    metaDataTimer = new QTimer;
    connect(metaDataTimer, SIGNAL(timeout()), SLOT(getMetaData()));

    alternateBoolean = true;

    readSettings();

    ui->setupUi(this);
    populateListWidget();
    createActionGroups();
    createQuickAccessButtons();
    setQuickAccessButtonsText();
    setButtonsIcons();
    changeFont();

    addAction(ui->actionGetMetaData);
    addAction(ui->actionGetStereoMode);
    addAction(ui->actionSetForceStereoMode);
    addAction(ui->actionGoFullScreen);

    setWidgetsEnabled(false);

    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call",
                                         "com.nokia.csd.Call",
                                         "Coming",
                                         this, SLOT(callIncoming()));

    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call",
                                         "com.nokia.csd.Call",
                                         "Created",
                                         this, SLOT(callIncoming()));

    QDBusConnection::systemBus().connect("com.nokia.csd.Call",
                                         "/com/nokia/csd/call/1",
                                         "com.nokia.csd.Call.Instance",
                                         "Terminated",
                                         this, SLOT(callEnded()));

    QDBusConnection::systemBus().connect(QString(),
                                         "/org/freedesktop/Hal/devices/platform_cam_focus",
                                         "org.freedesktop.Hal.Device",
                                         "PropertyModified",
                                         this, SLOT(focusButtonPressed()));

    if (inizializeAtStartUp)
    {
        qDebug()<<"Inizialize at start up is enabled, starting radio.";
        startRadio();
    }

    fileSystemWatcher = new QFileSystemWatcher;
    fileSystemWatcher->addPath("/sys/devices/platform/gpio-switch/cam_focus/disable");
    connect(fileSystemWatcher, SIGNAL(fileChanged(QString)), SLOT(setFocusButtonActive()));

}

MainWindow::~MainWindow()
{

    if (!radio==0) stopRadio();
    delete ui;

}

void MainWindow::setButtonsIcons()
{
    ui->prevStationButton->setIcon(QIcon("/usr/share/icons/hicolor/64x64/hildon/general_back.png"));
    ui->nextStationButton->setIcon(QIcon("/usr/share/icons/hicolor/64x64/hildon/general_forward.png"));
    ui->addStationButton->setIcon(QIcon("/usr/share/icons/hicolor/64x64/hildon/general_add.png"));
}

void MainWindow::startRadio()
{
    QFile file("/sys/devices/platform/gpio-switch/headphone/state");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream t( &file );
    QString headphoneStatus;
    headphoneStatus = t.readLine();
    qDebug()<<"Headphone status: "<<headphoneStatus;
    file.close();

    if (headphoneStatus == "disconnected")
    {
        return;
    }

    radio = new QRadioTuner;
    connect(radio,SIGNAL(frequencyChanged(int)),this,SLOT(freqChanged(int)));
    connect(radio, SIGNAL(metaDataChanged()), SLOT(getMetaData()));
    connect(radio, SIGNAL(metaDataAvailableChanged(bool)), SLOT(getMetaData()));
    connect(radio, SIGNAL(signalStrengthChanged(int)), SLOT(signalChanged()));
    if (radio->isBandSupported(QRadioTuner::FM))
    {
        radio->setBand(QRadioTuner::FM);
        radio->setFrequency(lastFreq);
        radio->setVolume(ui->volumeSlider->value());
        radio->start();
    }

    metaDataTimer->start(10000);
    setWidgetsEnabled(true);
}

void MainWindow::stopRadio()
{
    metaDataTimer->stop();
    saveSettings();
    setWidgetsEnabled(false);
    delete radio;
    radio=0;
    QProcess::execute("sh /opt/qradio/restoremicrophone.sh");
}

void MainWindow::freqChanged(int)
{
    double freqI=radio->frequency()/1000000.0;
    ui->freqLabel->setText(QString::number(freqI, 'f', 1));
}

void MainWindow::on_actionStart_Radio_triggered()
{
    if (radio==0) startRadio();
}

void MainWindow::on_actionStop_Radio_triggered()
{
    if (!radio==0) stopRadio();
}

void MainWindow::on_freqUpButton_clicked()
{
    int f = radio->frequency();
    f = f + radio->frequencyStep(QRadioTuner::FM);
    radio->setFrequency(f);
}

void MainWindow::on_freqDownButton_clicked()
{
    int f = radio->frequency();
    f = f - radio->frequencyStep(QRadioTuner::FM);
    radio->setFrequency(f);
}

void MainWindow::getMetaData()
{
    QFile file("/sys/class/i2c-adapter/i2c-3/3-0022/rds_ps");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QString line;
    QTextStream t( &file );
    line = t.readLine();
    ui->rdsRadioLabel->setText(line);
    file.close();
    QFile file2("/sys/class/i2c-adapter/i2c-3/3-0022/rds_rt");
    file2.open(QIODevice::ReadOnly | QIODevice::Text);

    QTextStream t2( &file2 );
    line = t2.readLine();
    ui->rdsProgramLabel->setText(line);
    file.close();
}

void MainWindow::signalChanged()
{
    qDebug()<<"Signal changed: "<<radio->signalStrength();
    if(radio->signalStrength() > 1) radio->cancelSearch();
    ui->signalLabel->setText(QString::number(radio->signalStrength()).append('%'));
}

void MainWindow::on_actionGetMetaData_triggered()
{
    getMetaData();

}

void MainWindow::on_seekUpButton_clicked()
{
    radio->searchForward();
}

void MainWindow::on_seekDownButton_clicked()
{
    radio->searchBackward();
}

void MainWindow::changeFont()
{
    QFont freqLabelFont(fontName, freqLabelFontSize);
    ui->freqLabel->setFont(freqLabelFont);

    QFont rdsLabelsFont(fontName, rdsLabelsFontSize);
    ui->rdsProgramLabel->setFont(rdsLabelsFont);
    ui->signalLabel->setFont(rdsLabelsFont);
    rdsLabelsFont.setBold(true);
    ui->rdsRadioLabel->setFont(rdsLabelsFont);

    QFont quickAccessFont(fontName, quickAccessButtonsFontSize);
    for (int i=0; i<4; i++)
    {
        quickAccessButton[i]->setFont(quickAccessFont);
    }

    QFont stationListFont(fontName, stationListFontSize);
    ui->stationsListWidget->setFont(stationListFont);

    QFont buttonsFont(fontName, buttonsFontSize);
    ui->freqDownButton->setFont(buttonsFont);
    ui->freqUpButton->setFont(buttonsFont);
    ui->seekDownButton->setFont(buttonsFont);
    ui->seekUpButton->setFont(buttonsFont);
}

void MainWindow::readSettings()
{
    QSettings settings("qradio", "settings");
    fontName=settings.value("fontName", "Digital-7").toString();
    freqLabelFontSize=settings.value("currentFreqFontSize", 140).toInt();
    rdsLabelsFontSize=settings.value("rdsLabelsFontSize", 16).toInt();
    quickAccessButtonsFontSize=settings.value("quickAccessButtonsFontSize", 20).toInt();
    buttonsFontSize=settings.value("buttonsFontSize", 20).toInt();
    stationListFontSize=settings.value("stationListFontSize", 20).toInt();
    lastFreq=settings.value("lastFreq").toInt();
    currentSet=settings.value("currentSet", 1).toInt();
    resumeAfterCall=settings.value("resumeAfterCall", false).toBool();
    inizializeAtStartUp=settings.value("inizializeAtStartUp", false).toBool();
    cameraFocusButtonUsed=settings.value("cameraFocusButtonUsed", true).toBool();

    QSettings setSettings("qradio", "set" + QString::number(currentSet));
    quickAccessButtonsTextList = setSettings.value("quickAccessButtonsTextList", QStringList()<<"87.6"<< "101.0"<< "99.5"<< "102.3").toStringList();
    stationsList = setSettings.value("stationsList", QStringList()<<"RMC 101-101.0"<<"Radio1-88.4"<<"Virgin Radio-87.6"<<"102.3-102.3").toStringList();

}

void MainWindow::on_stationsListWidget_itemActivated(QListWidgetItem* item)
{

    int n;
    n=item->toolTip().toDouble()*1000000;
    radio->setFrequency(n);
}

void MainWindow::on_stationsListWidget_customContextMenuRequested(QPoint pos)
{
    qDebug()<<"Custom Context Menu Requested, coords: "<<pos.rx()<<" "<<pos.ry();

    if (ui->stationsListWidget->itemAt(pos - ui->stationsListWidget->pos()) ==0) qDebug()<<"No station is in this position";
    else qDebug()<<"Station selected is: "<< ui->stationsListWidget->itemAt(pos - ui->stationsListWidget->pos())->text();

    listWidgetPosition = (pos - ui->stationsListWidget->pos());
    QMenu menu(this);
    menu.addAction(ui->actionInsert_here);
    if (ui->stationsListWidget->itemAt(pos - ui->stationsListWidget->pos()) !=0 ) menu.addAction(ui->actionDelete_Station);
    menu.addAction(ui->actionSort_Alphabetically);
    menu.addAction(ui->actionClear_List);
    menu.addAction(ui->actionToggle_Drag_Enabled);
    menu.exec(pos);



}

void MainWindow::on_addStationButton_clicked()
{
    QString radioName;
    QString radioFreq;
    bool ok;
    radioName = QInputDialog::getText(this, tr("Station name"), tr("Insert Name:"), QLineEdit::Normal, ui->rdsRadioLabel->text(), &ok , 0);
    if (!ok) return;
    if (radioName.isEmpty()) return;

    double freq;
    freq = radio->frequency();
    freq /= 1000000;
    radioFreq = QString::number(freq, 'f', 1);

    QListWidgetItem *listWidgetItem = new QListWidgetItem;
    listWidgetItem->setText(radioName);
    listWidgetItem->setToolTip(radioFreq);
    listWidgetItem->setTextAlignment(Qt::AlignCenter);

    ui->stationsListWidget->addItem(listWidgetItem);



}

void MainWindow::on_actionToggle_Drag_Enabled_triggered()
{
    qDebug()<<"Toggle drag e drop mode triggered";
    if (ui->stationsListWidget->dragDropMode() == QListWidget::NoDragDrop)
    {
        qDebug()<<"Drag e Drop mode is: NoDragDrop";
        ui->stationsListWidget->setDragDropMode(QListWidget::InternalMove);
    }
    else
    {
        qDebug()<<"Drag e Drop mode is: InternalMove";
        ui->stationsListWidget->setDragDropMode(QListWidget::NoDragDrop);
    }
}

void MainWindow::createQuickAccessButtons()
{
    for (int i=0; i<4; i++)
    {
        quickAccessButton[i]=new EmittingButton(this, 87.6, i);
        quickAccessButton[i]->setFixedHeight(50);
        connect((quickAccessButton[i]), SIGNAL(buttonClicked(int)), SLOT(quickAccessClicked(int)));
        connect((quickAccessButton[i]), SIGNAL(buttonIsHold(int)), SLOT(quickAccessHold(int)));
        ui->quickAccessLayout->addWidget(quickAccessButton[i]);
    }
}

void MainWindow::on_volumeSlider_valueChanged(int value)
{
    if (ui->actionHeadphones->isChecked())  radio->setVolume(value);
    if (ui->actionSpeakers->isChecked())
    {
        int volume = ui->volumeSlider->value();
        //volume = (volume/100.0)*118;         //Needed to get max volume for speakers, hurts ears.
        QString program = "sh /opt/qradio/speakerVolume.sh ";
        program.append(QString::number(volume));

        QProcess::execute(program);


    }
}

void MainWindow::setWidgetsEnabled(bool enable)
{
 ui->volumeSlider->setEnabled(enable);

 ui->seekDownButton->setEnabled(enable);
 ui->seekUpButton->setEnabled(enable);
 ui->prevStationButton->setEnabled(enable);
 ui->nextStationButton->setEnabled(enable);

 ui->addStationButton->setEnabled(enable);
 ui->freqDownButton->setEnabled(enable);
 ui->freqUpButton->setEnabled(enable);
 ui->stationsListWidget->setEnabled(enable);

 ui->actionManual->setEnabled(enable);
 ui->actionAuto_Scan->setEnabled(enable);
 ui->actionHeadphones->setEnabled(enable);
 ui->actionSpeakers->setEnabled(enable);

 for (int i=0; i<4; i++)
 {
     quickAccessButton[i]->setEnabled(enable);
 }


}

void MainWindow::quickAccessClicked(int n)
{
    radio->setFrequency(n);
}

void MainWindow::quickAccessHold(int i)
{
    double freqz=(radio->frequency())/1000000.0;
    quickAccessButton[i]->setText(QString::number(freqz, 'f', 1 ));

}

void MainWindow::createActionGroups()
{
    QActionGroup *audioOutputActionGroup = new QActionGroup(this);
    audioOutputActionGroup->addAction(ui->actionHeadphones);
    audioOutputActionGroup->addAction(ui->actionSpeakers);

    ui->menuMenu->addActions(audioOutputActionGroup->actions());
}

void MainWindow::on_actionClear_List_triggered()
{
    ui->stationsListWidget->clear();
}

void MainWindow::on_actionDelete_Station_triggered()
{
    qDebug()<<"Deleting radio station in row "<<ui->stationsListWidget->row(ui->stationsListWidget->currentItem());
    ui->stationsListWidget->takeItem(ui->stationsListWidget->row(ui->stationsListWidget->currentItem()));

}

void MainWindow::on_actionInsert_here_triggered()
{
    QString radioName;
    QString radioFreq;
    bool ok;
    radioName = QInputDialog::getText(this, tr("Station name"), tr("Insert Name:"), QLineEdit::Normal, ui->rdsRadioLabel->text(), &ok , 0);
    if (!ok) return;
    if (radioName.isEmpty()) return;

    double freq;
    freq = radio->frequency();
    freq /= 1000000;
    radioFreq = QString::number(freq, 'f', 1);

    QListWidgetItem *listWidgetItem = new QListWidgetItem;
    listWidgetItem->setText(radioName);
    listWidgetItem->setToolTip(radioFreq);
    listWidgetItem->setTextAlignment(Qt::AlignCenter);

    int row = ui->stationsListWidget->row(ui->stationsListWidget->currentItem());

    ui->stationsListWidget->insertItem(row, listWidgetItem);

}

void MainWindow::on_prevStationButton_clicked()
{
    int currentItemRow = ui->stationsListWidget->row(ui->stationsListWidget->currentItem());
    int itemCount = ui->stationsListWidget->count();
    qDebug()<<"Current Row is: "<<currentItemRow;

    if (itemCount == 0)
    {
        qDebug()<<"No stations in list, returning";
        return;
    }

    switch(currentItemRow)
    {
    case (-1): qDebug()<<"No station currently selected";
                ui->stationsListWidget->setCurrentRow(0); break;
    case (0): ui->stationsListWidget->setCurrentRow(itemCount - 1); break;
    default:  ui->stationsListWidget->setCurrentRow(currentItemRow - 1); break;
    }

    int n;
    n=ui->stationsListWidget->currentItem()->toolTip().toDouble()*1000000;
    radio->setFrequency(n);

}

void MainWindow::on_nextStationButton_clicked()
{
    int currentItemRow = ui->stationsListWidget->row(ui->stationsListWidget->currentItem());
    int itemCount = ui->stationsListWidget->count();
    qDebug()<<"Current Row is: "<<currentItemRow;

    if (itemCount == 0)
    {
        qDebug()<<"No stations in list, returning";
        return;
    }


    if (currentItemRow == -1)
    {
        qDebug()<<"No station currently selected";
        ui->stationsListWidget->setCurrentRow(0);

    }
    if (currentItemRow == (itemCount -1))
    {
        ui->stationsListWidget->setCurrentRow(0);
    }

    if ((currentItemRow != -1) && (currentItemRow != (itemCount -1))) ui->stationsListWidget->setCurrentRow(currentItemRow + 1);

    int n;
    n=ui->stationsListWidget->currentItem()->toolTip().toDouble()*1000000;
    radio->setFrequency(n);

}

void MainWindow::saveSettings()
{
    QSettings settings("qradio", "settings");
    settings.setValue("lastFreq", radio->frequency());

    QStringList tempQuickAccessList;
    for (int i=0; i<4; i++)
    {
        tempQuickAccessList.append(quickAccessButton[i]->text());
    }

    QStringList tempStationsList;
    QString tempString;
    QListWidgetItem *tempItem;


    for (int i=0; i<ui->stationsListWidget->count(); i++)
    {
        tempItem = ui->stationsListWidget->item(i);
        tempString = tempItem->text() + "-" + tempItem->toolTip();
        tempStationsList<<tempString;
    }

    QSettings setSettings("qradio", "set" + QString::number(currentSet));
    setSettings.setValue("quickAccessButtonsTextList", tempQuickAccessList);
    setSettings.setValue("stationsList", tempStationsList);

}

void MainWindow::setQuickAccessButtonsText()
{
    for (int i=0; i<4; i++)
    {
        quickAccessButton[i]->setText(quickAccessButtonsTextList.at(i));
    }
}

void MainWindow::populateListWidget()
{
    ui->stationsListWidget->clear();

    QString temp1;
    QString temp2;
    QStringList tempList;

    for (int i=0; i<stationsList.count(); i++)
    {
        QListWidgetItem *item = new QListWidgetItem(ui->stationsListWidget);
        tempList = stationsList.at(i).split("-");
        temp1=tempList.at(0);
        temp2=tempList.at(1);
        item->setText(temp1);
        item->setToolTip(temp2);

        item->setTextAlignment(Qt::AlignCenter);



    }
}

void MainWindow::on_actionSort_Alphabetically_triggered()
{
    ui->stationsListWidget->sortItems();
}

void MainWindow::on_actionGetStereoMode_triggered()
{
    if (radio==0) return;
    qDebug()<<radio->stereoMode();
}

void MainWindow::on_actionSetForceStereoMode_triggered()
{
    if (radio==0) return;
    radio->setStereoMode(QRadioTuner::ForceStereo);
    qDebug()<<"Force Stereo mode set";
}

void MainWindow::on_actionManual_triggered()
{
    bool ok;
    double manualFreq = QInputDialog::getDouble(this, tr("Manual Freq"), tr("Frequence"), 87.50, 87.5, 108.0, 2, &ok, 0);
    if (ok) radio->setFrequency(manualFreq*1000000);
}

void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::about(this, tr("About QRadio"),
                    tr("<h2>QRadio 0.4.7</h2>"
                       "<p>"
                       "<p>Written by Giovanni Grammatico (ggiovanni88@yahoo.it)"));
}

void MainWindow::on_actionGoFullScreen_triggered()
{
    if (isFullScreen)
        {
            isFullScreen=false;
            this->showNormal();;
        }
        else
        {
            isFullScreen = true;
            this->showFullScreen();
        }
}

void MainWindow::on_actionSpeakers_triggered()
{
    QProcess::execute("sh /opt/qradio/output.sh speaker");
    ui->volumeSlider->setValue(50);
    radio->setVolume(100);
}

void MainWindow::on_actionHeadphones_triggered()
{
    QProcess::execute("sh /opt/qradio/output.sh headphone");
    ui->volumeSlider->setValue(100);
    radio->setVolume(100);
}

void MainWindow::on_actionSettings_triggered()
{
    if (radio != 0) saveSettings();
    SettingsDialog settingsdialog;
    connect(&settingsdialog, SIGNAL(saveButtonClicked()), SLOT(settingsChanged()));
    settingsdialog.exec();
}

void MainWindow::settingsChanged()
{
    readSettings();
    populateListWidget();
    setQuickAccessButtonsText();
    changeFont();
}

void MainWindow::on_actionAuto_Scan_triggered()
{
    int tempFreq = radio->frequency();
        if (tempFreq<87500000) tempFreq = 87500000;


        do
        {
            radio->setFrequency(tempFreq);
            if (radio->signalStrength()>0)
            {
                QString radioFreq;

                double freq;
                freq = radio->frequency();
                freq /= 1000000;
                radioFreq = QString::number(freq, 'f', 1);

                QListWidgetItem *listWidgetItem = new QListWidgetItem;
                listWidgetItem->setText(radioFreq);
                listWidgetItem->setToolTip(radioFreq);

                ui->stationsListWidget->addItem(listWidgetItem);

            }
            tempFreq+=100000;

        }

        while (tempFreq<108000000);

}

void MainWindow::callIncoming()
{
    qDebug()<<"Call incoming/created";

    if (radio==0) return;

    qDebug()<<"Stopping radio.";

    metaDataTimer->stop();
    saveSettings();
    setWidgetsEnabled(false);
    delete radio;
    radio=0;
    QProcess::execute("sh /opt/qradio/restoremicrophone.sh");
}

void MainWindow::callEnded()
{
    qDebug()<<"Call ended";

    if (!resumeAfterCall)
    {
        qDebug()<<"Resuming after call is not enabled";
        return;
    }

    qDebug()<<"Resuming after call enabled, restarting radio.";

    QFile file("/sys/devices/platform/gpio-switch/headphone/state");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream t( &file );
    QString headphoneStatus;
    headphoneStatus = t.readLine();
    qDebug()<<"Headphone status: "<<headphoneStatus;
    file.close();

    if (headphoneStatus == "disconnected")
    {
        return;
    }

    radio = new QRadioTuner;
    connect(radio,SIGNAL(frequencyChanged(int)),this,SLOT(freqChanged(int)));
    connect(radio, SIGNAL(metaDataChanged()), SLOT(getMetaData()));
    connect(radio, SIGNAL(metaDataAvailableChanged(bool)), SLOT(getMetaData()));
    connect(radio, SIGNAL(signalStrengthChanged(int)), SLOT(signalChanged()));
    if (radio->isBandSupported(QRadioTuner::FM))
    {
        radio->setBand(QRadioTuner::FM);
        radio->setFrequency(lastFreq);
        radio->setVolume(ui->volumeSlider->value());
        radio->start();
    }

    metaDataTimer->start(10000);
    setWidgetsEnabled(true);
}

void MainWindow::focusButtonPressed()
{
    qDebug()<<"Focus button pressed";

    if (radio==0)
    {
        return;
    }

    if (!cameraFocusButtonUsed)
    {
        qDebug()<<"Camera Focus Button option not enabled";
        return;
    }

    if (alternateBoolean == true)
    {
        system("cat /sys/devices/platform/gpio-switch/cam_focus/state");
        on_nextStationButton_clicked();
        alternateBoolean = false;
    }
    else
    {
        alternateBoolean = true;
    }

}

void MainWindow::setFocusButtonActive()
{
    if (!cameraFocusButtonUsed) return;

    qDebug()<<"Forcing focus button active";

    QFile file("/sys/devices/platform/gpio-switch/cam_focus/disable");
    file.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream t( &file );
    QString value;
    value = t.readLine();
    qDebug()<<"Disable status: "<<value;
    file.close();

    if (value == "0")
    {
        return;
    }
    system("echo 'echo 0 > /sys/devices/platform/gpio-switch/cam_focus/disable' | root");
}
