import Qt 4.7
import QtMultimediaKit 1.2
import "scripts/settings.js" as Settings

Rectangle {
    id: window

    property variant currentVideo
    property int playlistPosition : 0
    property bool gettingVideoUrl : false
    property string playbackQuality

    signal playbackStopped

    function setPlaylist(videoList) {
        for (var i = 0; i < videoList.length; i++) {
            playbackModel.append(videoList[i]);
        }
        currentVideo = playbackModel.get(0);
        if (currentVideo.filePath) {
            videoPlayer.setVideo(currentVideo.filePath);
            if (!((currentVideo.quality == "audio") || (currentVideo.filePath.slice(-4) == ".m4a"))) {
                Controller.doNotDisturb(true);
            }
        }
        else {
            gettingVideoUrl = true;
            YouTube.getVideoUrl(currentVideo.videoId);
            Controller.doNotDisturb(true);
        }
    }

    function getTime(msecs) {
        /* Convert seconds to HH:MM:SS format */

        var secs = Math.floor(msecs / 1000);
        var hours = Math.floor(secs / 3600);
        var minutes = Math.floor(secs / 60) - (hours * 60);
        var seconds = secs - (hours * 3600) - (minutes * 60);
        if (seconds < 10) {
            seconds = "0" + seconds;
        }
        var duration = minutes + ":" + seconds;
        if (hours > 0) {
            duration = hours + ":" + duration;
        }
        return duration;
    }

    function previous() {
        /* Play the previous item in the playlist */

        if (playlistPosition > 0) {
            playlistPosition--;
        }
    }

    function next() {
        /* Play the next item in the playlist */

        if (playlistPosition < playbackModel.count - 1) {
            playlistPosition++;
        }
    }

    color: "black"
    onPlaylistPositionChanged: {
        if (playlistPosition < playbackModel.count) {
            var nextVideo = playbackModel.get(playlistPosition);
            if (nextVideo.filePath) {
                videoPlayer.setVideo(nextVideo.filePath);
            }
            else if (nextVideo.videoUrl) {
                videoPlayer.setVideo(nextVideo.videoUrl);
            }
            else {
                gettingVideoUrl = true;
                YouTube.getVideoUrl(nextVideo.videoId);
            }
            currentVideo = nextVideo;
        }
    }
    onStateChanged: {
        if (window.state == "audio") {
            Controller.doNotDisturb(false);
        }
        else {
            Controller.doNotDisturb(true);
        }
    }

    Component.onCompleted: {
        playbackQuality = Settings.getSetting("playbackQuality");
    }

    Connections {
        target: YouTube
        onGotVideoUrl: {
            gettingVideoUrl = false;
            videoPlayer.setVideo(videoUrl);
            var cv = currentVideo;
            cv["videoUrl"] = videoUrl;
            currentVideo = cv;
        }
        onVideoUrlError: {
            gettingVideoUrl = false;
            playlistPosition++;
        }
        onVideoRated: {
            var cv = currentVideo;
            cv["rating"] = ytBar.likeOrDislike;
            currentVideo = cv;
        }
        onAddedToFavourites: {
            var cv = currentVideo;
            cv["favourite"] = true;
            currentVideo = cv;
        }
    }

    Timer {
        id: controlsTimer

        running: (window.state == "") && (controls.showControls) && (!controls.showExtraControls)
        interval: 3000
        onTriggered: controls.showControls = false
    }

    Video {
        id: videoPlayer

        function setVideo(videoUrl) {
            videoPlayer.source = videoUrl;
            videoPlayer.play()
            if (currentVideo.filePath) {
                archiveModel.markItemAsOld(currentVideo.filePath);
            }
        }

        z: (currentVideo) && !(currentVideo.quality == "audio") && (seekBar.position > 0) ? 0 : -1
        anchors.fill: window
        onPositionChanged: {
            if (videoPlayer.position > 0) {
                seekBar.position = videoPlayer.position;
            }
            if ((videoPlayer.duration > 0) && ((videoPlayer.duration - videoPlayer.position) < 500)) {
                if (playlistPosition == (playbackModel.count - 1)) {
                    videoPlayer.stop();
                    videoPlayer.source = "";
                    playbackStopped();
                }
                else {
                    playlistPosition++;
                }
            }
        }

        Behavior on z { NumberAnimation { duration: 300 } }
    }

    Item {
        id: controls

        property bool showControls : false
        property bool showExtraControls : false
        property bool audioMode : false

        anchors.fill: window
        onShowControlsChanged: {
            if (!controls.showControls) {
                controls.showExtraControls = false;
            }
        }

        MouseArea {
            id: controlsMouseArea

            property real xPos

            anchors.fill: controls
            onClicked: {
                if (window.state == "") {
                    controls.showControls = !controls.showControls;
                }
            }
            onPressAndHold: videoPlayer.paused = !videoPlayer.paused
            onPressed: xPos = mouseX
            onReleased: {
                if (xPos - mouseX > 100) {
                    next();
                }
                else if (mouseX - xPos > 100) {
                    previous();
                }
            }
        }

        Image {
            id: pauseIcon

            width: 80
            height: 80
            anchors.centerIn: controls
            source: "ui-images/pauseicon.png"
            smooth: true
            visible: (videoPlayer.paused) && (playlistView.opacity == 0)
        }

        Text {
            id: loadingText

            anchors.centerIn: controls
            font.pixelSize: _LARGE_FONT_SIZE
            color: "white"
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            smooth: true
            text: qsTr("Loading...")
            visible: (!videoPlayer.paused) && ((gettingVideoUrl) || (videoPlayer.status == Video.Loading) || (videoPlayer.status == Video.Buffering) || (videoPlayer.status == Video.Stalled))
        }

        Item {
            id: ytBar

            property string likeOrDislike

            height: 50
            anchors { top: playlistView.top; left: playlistView.right; leftMargin: -1; right: titleBar.right; bottom: undefined }
            opacity: (controls.showExtraControls) && (currentVideo) && (currentVideo.videoId) ? 1 : 0

            Rectangle {
                anchors.fill: ytBar
                color: "black"
                smooth: true
                opacity: (window.state == "audio") ? 0 : 1
            }

            Row {
                anchors.fill: ytBar
                spacing: 10

                Image {
                    id: likeButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (likeMouseArea.pressed) || ((currentVideo) && (currentVideo.rating == "like")) ? "ui-images/likeiconblue.png" : "ui-images/likeicon.png"
                    sourceSize.width: likeButton.width
                    sourceSize.height: likeButton.height
                    smooth: true

                    MouseArea {
                        id: likeMouseArea

                        anchors.fill: likeButton
                        onClicked: {
                            if (window.state == "") {
                                controls.showControls = false;
                            }
                            if ((!currentVideo.rating) && (userIsSignedIn())) {
                                ytBar.likeOrDislike = "like";
                                YouTube.rateVideo(currentVideo.videoId, "like");
                            }
                        }
                    }
                }

                Image {
                    id: dislikeButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (dislikeMouseArea.pressed) || ((currentVideo) && (currentVideo.rating == "dislike")) ? "ui-images/dislikeiconblue.png" : "ui-images/dislikeicon.png"
                    sourceSize.width: dislikeButton.width
                    sourceSize.height: dislikeButton.height
                    smooth: true

                    MouseArea {
                        id: dislikeMouseArea

                        anchors.fill: dislikeButton
                        onClicked: {
                            if (window.state == "") {
                                controls.showControls = false;
                            }
                            if ((!currentVideo.rating) && (userIsSignedIn())) {
                                ytBar.likeOrDislike = "dislike";
                                YouTube.rateVideo(currentVideo.videoId, "dislike");
                            }
                        }
                    }
                }

                Image {
                    id: favButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (favMouseArea.pressed) || ((currentVideo) && (currentVideo.favourite)) ? "ui-images/favouritesiconblue.png" : "ui-images/favouritesicon.png"
                    sourceSize.width: favButton.width
                    sourceSize.height: favButton.height
                    smooth: true

                    MouseArea {
                        id: favMouseArea

                        anchors.fill: favButton
                        onClicked: {
                            if (window.state == "") {
                                controls.showControls = false;
                            }
                            if ((!currentVideo.favourite) && (userIsSignedIn())) {
                                YouTube.addToFavourites(currentVideo.videoId);
                            }
                        }
                    }
                }

                Image {
                    id: videoDownloadButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (videoDownloadMouseArea.pressed) || ((currentVideo) && (currentVideo.videoDownload)) ? "ui-images/videodownloadiconblue.png" : "ui-images/videodownloadicon.png"
                    sourceSize.width: videoDownloadButton.width
                    sourceSize.height: videoDownloadButton.height
                    smooth: true

                    MouseArea {
                        id: videoDownloadMouseArea

                        anchors.fill: videoDownloadButton
                        onClicked: {
                            if (window.state == "") {
                                controls.showControls = false;
                            }
                            if (!currentVideo.videoDownload) {
                                var cv = currentVideo;
                                cv["videoDownload"] = true;
                                cv["status"] = "paused";
                                addDownload(cv);
                                currentVideo = cv;
                            }
                        }
                    }
                }

                Image {
                    id: audioDownloadButton

                    width: ytBar.height
                    height: ytBar.height
                    source: (audioDownloadMouseArea.pressed) || ((currentVideo) && (currentVideo.audioDownload)) ? "ui-images/audiodownloadiconblue.png" : "ui-images/audiodownloadicon.png"
                    sourceSize.width: audioDownloadButton.width
                    sourceSize.height: audioDownloadButton.height
                    smooth: true

                    MouseArea {
                        id: audioDownloadMouseArea

                        anchors.fill: audioDownloadButton
                        onClicked: {
                            if (window.state == "") {
                                controls.showControls = false;
                            }
                            if (!currentVideo.audioDownload) {
                                var cv = currentVideo;
                                cv["audioDownload"] = true;
                                cv["status"] = "paused";
                                addAudioDownload(cv);
                                currentVideo = cv;
                            }
                        }
                    }
                }
            }
        }

        Rectangle {
            id: frame

            width: Math.floor(controls.width / 2.5)
            height: Math.floor(frame.width / (4 / 3))
            anchors { left: controls.left; leftMargin: 10; top: controls.top; topMargin: 100 }
            color: "black"
            border.width: 2
            border.color: "white"
            smooth: true
            visible: false

            Image {
                id: coverArt

                anchors { fill: frame; margins: 2 }
                sourceSize.width: coverArt.width
                sourceSize.height: coverArt.height
                smooth: true
            }

            Text {
                anchors.centerIn: frame
                font.pixelSize: _STANDARD_FONT_SIZE
                color: "white"
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                smooth: true
                text: qsTr("Loading...")
                visible: (!videoPlayer.paused) && ((gettingVideoUrl) || (videoPlayer.status == Video.Loading) || (videoPlayer.status == Video.Buffering) || (videoPlayer.status == Video.Stalled) || (coverArt.status == Image.Loading))
            }
        }

        Row {
            id: buttonRow

            spacing: 10
            anchors { top: frame.bottom; topMargin: 10; horizontalCenter: frame.horizontalCenter }
            visible: false

            PushButton {
                id: prevButton

                width: Math.floor((frame.width / 3) - 7)
                useTheme: false
                icon: "ui-images/previousicon.png"
                onButtonClicked: {
                    if (videoPlayer.position > 5000) {
                        videoPlayer.position = 1;
                    }
                    else {
                        previous();
                    }
                }
            }

            PushButton {
                id: playButton

                width: prevButton.width
                useTheme: false
                icon: videoPlayer.paused ? "ui-images/playicon.png" : "ui-images/pauseicon.png"
                onButtonClicked: videoPlayer.paused = !videoPlayer.paused
            }

            PushButton {
                id: nextButton

                width: prevButton.width
                useTheme: false
                icon: "ui-images/nexticon.png"
                onButtonClicked: next()
            }
        }


        ListView {
            id: playlistView

            width: titleText.width
            height: (playlistView.count <= 4) ? (playlistView.count * 50) : 200
            anchors { top: titleBar.bottom; topMargin: -1; left: controls.left; right: undefined }
            clip: true
            interactive: playlistView.count > 4
            boundsBehavior: Flickable.StopAtBounds
            snapMode: ListView.SnapToItem
            opacity: (controls.showExtraControls) && (playbackModel.count > 1) ? 1 : 0
            onOpacityChanged: playlistView.positionViewAtIndex(playlistPosition, ListView.Beginning)
            delegate: Rectangle {
                id: delegate

                width: delegate.parent.width
                height: 50
                color: "black"

                ListHighlight {
                    visible: (delegateMouseArea.pressed) || (playlistPosition == index)
                }

                Text {
                    anchors { fill: delegate; leftMargin: 10 }
                    font.pixelSize: _STANDARD_FONT_SIZE
                    elide: Text.ElideRight
                    verticalAlignment: Text.AlignVCenter
                    color: "white"
                    smooth: true
                    text: (index + 1).toString() + " - " + title
                }

                MouseArea {
                    id: delegateMouseArea

                    z: 100
                    anchors.fill: delegate
                    onClicked: {
                        if (window.state == "") {
                            controls.showControls = false;
                        }
                        playlistPosition = index;
                    }
                }
            }

            model: ListModel {
                id: playbackModel
            }
        }

        Item {
            id: titleBar

            anchors.top: controls.top
            width: controls.width
            height: 50
            opacity: controls.showControls ? 1 : 0

            Rectangle {
                anchors.fill: titleBar
                color: "black"
                smooth: true
                opacity: (window.state == "audio") ? 0.8 : 1
            }

            ToolButton {
                id: minimizeButton

                anchors { left: titleBar.left; leftMargin: 10; verticalCenter: titleBar.verticalCenter }
                useTheme: false
                icon: "ui-images/minimizeicon.png"
                onButtonClicked: Controller.minimize()
            }

            ToolButton {
                id: modeButton

                anchors { left: titleBar.left; leftMargin: 60; verticalCenter: titleBar.verticalCenter }
                useTheme: false
                icon: controls.audioMode ? "ui-images/videosicon.png" : "ui-images/musicicon.png"
                onButtonClicked: controls.audioMode = !controls.audioMode
                visible: !currentVideo ? false : ((currentVideo.quality == "audio") || ((currentVideo.filePath) && (currentVideo.filePath.slice(-4) == ".m4a"))) ? false : true
            }

            Text {
                id: titleText

                anchors { left: (currentVideo) && ((currentVideo.quality == "audio") || ((currentVideo.filePath) && (currentVideo.filePath.slice(-4) == ".m4a"))) ? minimizeButton.right : modeButton.right;
                    leftMargin: 10; right: titleBar.right; rightMargin: 200; verticalCenter: titleBar.verticalCenter }
                font.pixelSize: _STANDARD_FONT_SIZE
                elide: Text.ElideRight
                verticalAlignment: Text.AlignVCenter
                color: "white"
                smooth: true
                text: !currentVideo ? "" : currentVideo.title

                MouseArea {
                    id: titleMouseArea
                    z: 100
                    anchors.fill: titleText
                    onClicked: controls.showExtraControls = !controls.showExtraControls
                }
            }

            CloseButton {
                id: closeButton

                anchors { right: titleBar.right; rightMargin: 10; verticalCenter: titleBar.verticalCenter }
                useTheme: false
                onButtonClicked: {
                    videoPlayer.stop();
                    videoPlayer.source = "";
                    playbackStopped();
                }
            }
        }

        Text {
            id: time

            anchors { right: titleBar.right; rightMargin: 70; verticalCenter: titleBar.verticalCenter; top: undefined }
            height: titleBar.height
            font.pixelSize: _STANDARD_FONT_SIZE
            verticalAlignment: Text.AlignVCenter
            color: "white"
            smooth: true
            opacity: titleBar.opacity
            text: !currentVideo ? "0:00/0:00" : getTime(seekBar.position) + "/" + getTime(videoPlayer.duration)
        }

        Rectangle {
            id: seekRect

            height: 20
            anchors { bottom: controls.bottom; left: controls.left; right: controls.right; verticalCenter: undefined }
            color: "black"
            opacity: controls.showControls ? 0.7 : 0
            smooth: true

            Rectangle {
                id: seekBar

                property int position : 0

                width: !currentVideo ? 0 : Math.floor(seekRect.width * (seekBar.position / videoPlayer.duration))
                height: seekRect.height
                anchors.bottom: seekRect.bottom
                color: _ACTIVE_COLOR_LOW
                smooth: true

                Behavior on width { SmoothedAnimation { velocity: 1200 } }
            }

            MouseArea {
                id: seekMouseArea

                width: parent.width
                height: 50
                anchors.bottom: parent.bottom
                onClicked: {
                    if (!((playbackQuality == "mobile") ||(currentVideo.quality == "mobile"))) {
                        videoPlayer.position = Math.floor((mouseX / seekRect.width) * videoPlayer.duration);
                    }
                }
            }
        }
    }

    states: State {
        name: "audio"
        when: (controls.audioMode) || ((currentVideo) && (currentVideo.quality == "audio") || (currentVideo.filePath.slice(-4) == ".m4a"))
        PropertyChanges { target: videoPlayer; z: -1 }
        PropertyChanges { target: coverArt; source: !currentVideo ? "" : currentVideo.videoId ?  currentVideo.largeThumbnail : currentVideo.thumbnail.replace("default", "hqdefault") }
        PropertyChanges { target: controlsMouseArea; enabled: false }
        PropertyChanges { target: titleMouseArea; enabled: false }
        PropertyChanges { target: controls; showControls: true; showExtraControls: true }
        PropertyChanges { target: frame; visible: true }
        PropertyChanges { target: buttonRow; visible: true }
        PropertyChanges { target: loadingText; visible: false }
        PropertyChanges { target: pauseIcon; visible: false }
        AnchorChanges { target: playlistView; anchors { left: frame.right; top: frame.top; right: controls.right } }
        PropertyChanges { target: playlistView; opacity: 1; height: frame.height; anchors { leftMargin: 10; topMargin: 0; rightMargin: 10 } }
        AnchorChanges { target: ytBar; anchors { bottom: playlistView.top; right: playlistView.right; left: undefined; top: undefined } }
        AnchorChanges { target: seekRect; anchors { left: playlistView.left; right: playlistView.right; verticalCenter: buttonRow.verticalCenter; bottom: undefined } }
        AnchorChanges { target: time; anchors { right: controls.right; top: seekRect.bottom; verticalCenter: undefined } }
        PropertyChanges { target: time; anchors { topMargin: 10; rightMargin: 10 } }
        PropertyChanges { target: titleText; anchors.rightMargin: 80 }
    }
}

