/*******************************************************************************
**
** vkservice.cpp - implement VkService and VkServicePlugin, main of plug-in.
**                       Inherited by QMailServiceConfiguration.
** This file is part of the QMF-Vkontakte plug-in.
**
** Copyright (C) 2010 PetrSU. Pavel Shiryaev <shiryaev AT cs.karelia.ru>
** Contact: QMF-vkontakte team (qmf-vkontakte AT cs.karelia.ru)
**
** QMF-Vkontakte plug-in is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** QMF-Vkontakte plug-in is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with QMF-Vkontakte plug-in; if not, write to the Free Software
** Foundation, Inc., 51 Franklin St, Fifth Floor,
** Boston, MA  02110-1301  USA
**
*******************************************************************************/

#include "vkservice.h"
#include "vkconfiguration.h"
#ifndef QMF_NO_MESSAGE_SERVICE_EDITOR
#include "vksettingssource.h"
#include "vksettingssink.h"
#endif
#include <QTimer>
#include <QtPlugin>
#include <QCoreApplication>

namespace { const QString serviceKey("vk"); }

class VkService::Source : public QMailMessageSource
{
    Q_OBJECT

public:
    Source(VkService *service)
        : QMailMessageSource(service),
          _service(service),
          _deleting(false),
          _unavailable(false),
          _mailCheckQueued(false),
          _queuedMailCheckInProgress(false)
    {
        connect(&_service->_client, SIGNAL(allMessagesReceived()),
                this, SIGNAL(newMessagesAvailable()));
        connect(&_service->_client, SIGNAL(messageActionCompleted(QString)),
                this, SLOT(messageActionCompleted(QString)));
        connect(&_service->_client, SIGNAL(retrievalCompleted()),
                this, SLOT(retrievalCompleted()));
        connect(&_intervalTimer, SIGNAL(timeout()),
                this, SLOT(queueMailCheck()));
    }

    void setIntervalTimer(int interval)
    {
        _intervalTimer.stop();
        if (interval > 0)
            _intervalTimer.start(interval*1000*60); // interval minutes
    }

    virtual QMailStore::MessageRemovalOption messageRemovalOption()
              const { return QMailStore::CreateRemovalRecord; }

public slots:
    virtual bool retrieveFolderList(const QMailAccountId &accountId,
                                    const QMailFolderId &folderId,
                                    bool descending);
    virtual bool retrieveMessageList(const QMailAccountId &accountId,
                                     const QMailFolderId &folderId,
                                     uint minimum,
                                     const QMailMessageSortKey &sort);

    virtual bool retrieveMessages(
                               const QMailMessageIdList &messageIds,
                               QMailRetrievalAction::RetrievalSpecification spec
                                  );

    virtual bool retrieveAll(const QMailAccountId &accountId);
    virtual bool exportUpdates(const QMailAccountId &accountId);

    virtual bool synchronize(const QMailAccountId &accountId);

    virtual bool deleteMessages(const QMailMessageIdList &ids);

    void messageActionCompleted(const QString &uid);
    void retrievalCompleted();
    void retrievalTerminated();
    void queueMailCheck();

private:
    VkService *_service;
    bool _deleting;
    bool _unavailable;
    bool _mailCheckQueued;
    bool _queuedMailCheckInProgress;
    QTimer _intervalTimer;
};

bool VkService::Source::retrieveFolderList(const QMailAccountId &accountId,
                                           const QMailFolderId &folderId,
                                           bool descending)
{
    qDebug() << Q_FUNC_INFO;

    if (!accountId.isValid()) {
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No account specified"));
        return false;
    }

    if (folderId.isValid()) {
        // Folders don't make sense for VK
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No account specified"));
        return false;
    }

    QTimer::singleShot(0, &(_service->_client), SLOT(updateProfile()));
    return true;

    Q_UNUSED(descending)
}

bool VkService::Source::retrieveMessageList(const QMailAccountId &accountId,
                                            const QMailFolderId &folderId,
                                            uint minimum,
                                            const QMailMessageSortKey &sort)
{
    qDebug() << Q_FUNC_INFO;
    if (!accountId.isValid()) {
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No account specified"));
        return false;
    }

    /*
    if (folderId.isValid()) {
        // Folders don't make sense for VK
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No account specified"));
        return false;
    }
    */

    // We cannot use minimum messages for vkontakte service.
    QMailMessageKey countKey(QMailMessageKey::parentAccountId(accountId));
    countKey &= ~QMailMessageKey::status(QMailMessage::Temporary);
    uint existing = QMailStore::instance()->countMessages(countKey);
    qDebug() << "existing =" << existing
             << "minimum =" << minimum;


    _service->_client.setOperation(QMailRetrievalAction::MetaData);
    _service->_client.newConnection();
    _unavailable = true;
    return true;

    Q_UNUSED(folderId)
    Q_UNUSED(sort)
}

bool VkService::Source::retrieveMessages(
                               const QMailMessageIdList &messageIds,
                               QMailRetrievalAction::RetrievalSpecification spec
                                         )
{
    //qDebug() << Q_FUNC_INFO;
    if (messageIds.isEmpty()) {
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No messages to retrieve"));
        return false;
    }

    if (spec == QMailRetrievalAction::Flags) {
        // Just report success
        // _service->updateStatus(""); // Why is it done?
        QTimer::singleShot(0, this, SLOT(retrievalCompleted()));
        return true;
    }

    // We also can't get MetaData & Content.

    SelectionMap selectionMap;
    foreach (const QMailMessageId& id, messageIds) {
        QMailMessageMetaData message(id);
        selectionMap.insert(message.serverUid(), id);
    }

    _service->_client.setOperation(spec);
    _service->_client.setSelectedMails(selectionMap);
    _service->_client.newConnection();
    _unavailable = true;
    return true;
}

bool VkService::Source::retrieveAll(const QMailAccountId &accountId)
{
    qDebug() << Q_FUNC_INFO;
    if (!accountId.isValid()) {
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No account specified"));
        return false;
    }

    _service->_client.setOperation(QMailRetrievalAction::MetaData);
    _service->_client.newConnection();
    _unavailable = true;
    return true;
}

bool VkService::Source::exportUpdates(const QMailAccountId &accountId)
{
    //qDebug() << Q_FUNC_INFO;
    if (!accountId.isValid()) {
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No account specified"));
        return false;
    }

    _service->_client.setOperation(VkClient::ExportUpdates);
    _service->_client.newConnection();

    return true;
}

bool VkService::Source::synchronize(const QMailAccountId &accountId)
{
    qDebug() << Q_FUNC_INFO;
    return retrieveAll(accountId);
}

bool VkService::Source::deleteMessages(const QMailMessageIdList &messageIds)
{
    if (messageIds.isEmpty()) {
        _service->errorOccurred(QMailServiceAction::Status::ErrInvalidData,
                                tr("No messages to delete"));
        return false;
    }

    /*
    // If we will provided delete later function
    QMailAccountConfiguration accountCfg(_service->accountId());
    VkConfiguration vkCfg(accountCfg);
    if (vkCfg.canDeleteMail())
    */

    // Delete the messages from the server
    SelectionMap selectionMap;
    foreach (const QMailMessageId& id, messageIds) {
        QMailMessageMetaData message(id);
        selectionMap.insert(message.serverUid(), id);
    }
    _deleting = true;
    _service->_client.setOperation(VkClient::DeleteMessage);
    _service->_client.setSelectedMails(selectionMap);
    _service->_client.newConnection();
    _unavailable = true;
    return true;

    // For delete later: Just delete the local copies
    //return QMailMessageSource::deleteMessages(messageIds);
}

void VkService::Source::messageActionCompleted(const QString &uid)
{
    if (_deleting) {
        QMailMessageMetaData metaData(uid, _service->accountId());
        if (metaData.id().isValid()) {
            QMailMessageIdList messageIds;
            messageIds.append(metaData.id());

            emit messagesDeleted(messageIds);
        }
    }
}

void VkService::Source::retrievalCompleted()
{
    _unavailable = false;

    if (_queuedMailCheckInProgress) {
        _queuedMailCheckInProgress = false;
        emit _service->availabilityChanged(true);
    }
    emit _service->actionCompleted(true);

    _deleting = false;
    if (_mailCheckQueued)
        queueMailCheck();
}

void VkService::Source::queueMailCheck()
{
    if (_unavailable) {
        _mailCheckQueued = true;
        return;
    }

    _mailCheckQueued = false;
    _queuedMailCheckInProgress = true;


    emit _service->availabilityChanged(false);
    synchronize(_service->accountId());
}

void VkService::Source::retrievalTerminated()
{
    _unavailable = false;
    if (_queuedMailCheckInProgress) {
        _queuedMailCheckInProgress = false;
        emit _service->availabilityChanged(true);
    }
    
    // Just give up if an error occurs
    _mailCheckQueued = false;
}

class VkService::Sink : public QMailMessageSink
{
    Q_OBJECT

public:
    Sink(VkService *service)
        : QMailMessageSink(service),
          _service(service),
          nothingSent(false)
    {
        connect(&_service->_client, SIGNAL(messageTransmitted(QMailMessageId)),
                this, SLOT(messageTransmitted(QMailMessageId)));
        connect(&_service->_client, SIGNAL(sendCompleted()),
                this, SLOT(sendCompleted()));
    }

public slots:
    virtual bool transmitMessages(const QMailMessageIdList &ids);

    void messageTransmitted(const QMailMessageId &id);
    void sendCompleted();

private:
    VkService *_service;
    bool nothingSent;
};

bool VkService::Sink::transmitMessages(const QMailMessageIdList &ids)
{
    bool messageQueued = false;
    QMailMessageIdList failedMessages;

    foreach (const QMailMessageId id, ids) {
        QMailMessage message(id);
        if (_service->_client.addMail(message) == QMailServiceAction::Status::ErrNoError) {
            messageQueued = true;
        } else {
            failedMessages << id;
        }
    }

    if (failedMessages.count()) {
        emit messagesFailedTransmission(failedMessages, QMailServiceAction::Status::ErrInvalidAddress);
    }

    if (messageQueued) {
        // At least one message could be queued for sending
        _service->_client.setOperation(VkClient::SendMessage);
        _service->_client.newConnection();
    } else {
        // No messages to send, so sending completed successfully
        qDebug() << "No messages to send, so sending completed successfully";
        nothingSent = true;
        QTimer::singleShot(0, this, SLOT(sendCompleted()));
    }

    return true;
}

void VkService::Sink::messageTransmitted(const QMailMessageId &id)
{
    qDebug() << "Id of transmited message: " << id.toULongLong();
    emit messagesTransmitted(QMailMessageIdList() << id);

    // Remove Transmitted because vkontakte does not support uids for sent
    if (!QMailStore::instance()->removeMessage(id))
        qWarning() << "Unable to remove message with ID:" << id.toULongLong();
}

void VkService::Sink::sendCompleted()
{
    emit _service->actionCompleted(true);

    if  (nothingSent) {
        // Sent Request was empty
        nothingSent = false;
    } else {
        // Retrieve send messages from server.
        _service->_client.newConnection();
    }
}

VkService::VkService(const QMailAccountId &accountId)
    : QMailMessageService(),
      _client(this),
      _source(new Source(this)),
      _sink(new Sink(this))
{
    //qDebug() << "VkService for id=" << (accountId);
    connect(&_client, SIGNAL(progressChanged(uint, uint)),
            this, SIGNAL(progressChanged(uint, uint)));
    connect(&_client, SIGNAL(errorOccurred(int, QString)),
            this, SLOT(errorOccurred(int, QString)));
    connect(&_client,
            SIGNAL(errorOccurred(QMailServiceAction::Status::ErrorCode,
                                 QString)),
            this, SLOT(errorOccurred(QMailServiceAction::Status::ErrorCode,
                                     QString)));
    connect(&_client, SIGNAL(updateStatus(QString)),
            this, SLOT(updateStatus(QString)));

    _client.setAccount(accountId);
    QMailAccountConfiguration accountCfg(accountId);
    VkConfiguration vkCfg(accountCfg);
    _source->setIntervalTimer(vkCfg.checkInterval());
}

VkService::~VkService()
{
    delete _source;
    delete _sink;
}

QString VkService::service() const
{
    return serviceKey;
}

QMailAccountId VkService::accountId() const
{
    return _client.account();
}

bool VkService::hasSource() const
{
    return true;
}

bool VkService::hasSink() const
{
    return true;
}

QMailMessageSource &VkService::source() const
{
    return *_source;
}

QMailMessageSink &VkService::sink() const
{
    return *_sink;
}

bool VkService::available() const
{
    return true;
}

bool VkService::cancelOperation(QMailServiceAction::Status::ErrorCode code, const QString &text)
{
    _client.cancelTransfer(code, text);
    _client.closeConnection();
    _source->retrievalTerminated();
    return true;
}

void VkService::errorOccurred(int code, const QString &text)
{
    updateStatus(code, text, _client.account());
    _source->retrievalTerminated();
    emit actionCompleted(false);
}

void VkService::errorOccurred(QMailServiceAction::Status::ErrorCode code,
                              const QString &text)
{
    updateStatus(code, text, _client.account());
    _source->retrievalTerminated();
    emit actionCompleted(false);
}

void VkService::updateStatus(const QString &text)
{
    updateStatus(QMailServiceAction::Status::ErrNoError,
                 text, _client.account());
}


class VkConfigurator : public QMailMessageServiceConfigurator
{
public:
    VkConfigurator();
    ~VkConfigurator();

    virtual QString service() const;
    virtual QString displayName() const;

#ifndef QMF_NO_MESSAGE_SERVICE_EDITOR
    virtual QMailMessageServiceEditor *createEditor(
            QMailMessageServiceFactory::ServiceType type);
#endif
};

VkConfigurator::VkConfigurator()
{
}

VkConfigurator::~VkConfigurator()
{
}

QString VkConfigurator::service() const
{
    return serviceKey;
}

QString VkConfigurator::displayName() const
{
    return QCoreApplication::instance()->translate("QMailMessageService", "VKontakte");
}

#ifndef QMF_NO_MESSAGE_SERVICE_EDITOR
QMailMessageServiceEditor *VkConfigurator::createEditor(
        QMailMessageServiceFactory::ServiceType type)
{
    // Return form for manage recieve settings
    if (type == QMailMessageServiceFactory::Source)
        return new VkSettingsSource;

    // Now, I don't know, how to use SourceAnd Sink
    if (type == QMailMessageServiceFactory::Sink)
        return new VkSettingsSink;

    return 0;
}
#endif

Q_EXPORT_PLUGIN2(vk,VkServicePlugin)

VkServicePlugin::VkServicePlugin()
    : QMailMessageServicePlugin()
{
}

QString VkServicePlugin::key() const
{
    return serviceKey;
}

bool VkServicePlugin::supports(
        QMailMessageServiceFactory::ServiceType type) const
{
    return (type == QMailMessageServiceFactory::Source)
            || (type == QMailMessageServiceFactory::Sink)
            ||  (type == QMailMessageServiceFactory::Any);
}

bool VkServicePlugin::supports(QMailMessage::MessageType type) const
{
    return (type == QMailMessage::Email);
}

QMailMessageService *VkServicePlugin::createService(const QMailAccountId &id)
{
    return new VkService(id);
}

QMailMessageServiceConfigurator *VkServicePlugin::createServiceConfigurator()
{
    return new VkConfigurator();
}

#include "vkservice.moc"
