/*******************************************************************************
**
** vktransport.cpp - transport module for QMF-Vkontakte plug-in
** Based on livkontakte API:  http://oss.fruct.org/wiki/Libvkontakte/xml-data
** This file is part of QMF-Vkontakte plug-in.
**
** Copyright (C) 2010 Pavel Shiryaev <shiryaev AT cs.karelia.ru>
** Copyright (C) 2010 - Nikolai Agafonov
**
** QMF-Vkontakte plug-in is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** QMF-Vkontakte plug-in is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with QMF-Vkontakte plug-in; if not, write to the Free Software
** Foundation, Inc., 51 Franklin St, Fifth Floor,
** Boston, MA  02110-1301  USA
**
*******************************************************************************/

#include <QDebug>
#include <QDateTime>
#include <qmailstore.h>
#include <qmailtimestamp.h>

#include "vktransport.h"

// Name of library that used for connection (without extension)
#define LIBVKONTAKTE "libvkontakte"

// Name of exported function
#define MODULE_INIT "msa_module_init"

#define VK_RESP_CODE "//string[@name='code']"
#define VK_RESP_TEXT "//string[@name='text']"
#define VK_FIRST_NAME "//string[@name='FirstName']"
#define VK_LAST_NAME "//string[@name='LastName']"
#define VK_STATUS "/string[@name='Status']"
#define VK_STRUCT "//array[@name='messagetList']/struct["
#define VK_ID_R "/@id"
#define VK_SENDER_NAME "/string[@name='SenderName']"
#define VK_SENDER_ID "/string[@name='SenderId']"
#define VK_RECIPIENT_NAME "/string[@name='RecipientName']"
#define VK_RECIPIENT_ID "/string[@name='RecipientId']"
#define VK_TIME "/string[@name='Time']"
#define VK_TEXT "/string[@name='Text']"
#define VK_IMG "//img[@name='Img']"
#define VK_FCSID "//string[@name='fcsid']"
#define VK_FCCODE "//string[@name='fccode']"
#define VK_IMG "//img[@name='Img']"
#define VK_MODULE_NAME "/string[@name='moduleName']"
#define VK_OLD_REQUEST "/string[@name='oldRequest']"
#define VK_FUNCTION "//Response/@function"
#define VK_FROM "//array/@from"
#define VK_TO "//array/@to"
#define VK_QUANTITY "//array/@quantity"

/*
// Deprecated libvkontakte interface (now unused)
namespace {
    int vkCurEvents(xmlChar * strData)
    {
        // Debug message for calling function
        qDebug() << "Called vkCurEvents()";

        // Add XML data to "request" string
        xmlChar *chData  = strData;

        if (chData != NULL)
        {
            qDebug() << "vkCurEvents : " << (char *) chData;
        }
        return 0;
    }
}
*/

VkTransport::VkTransport()
    : msa(0)
{
#ifdef VK_LOGING
    vkLog = fopen(QDir::tempPath().append(QDir::separator()).toAscii().append(VK_LOGING_FILE),"w");
#endif

    // Add dynamic library (libvkontakte.so)
    if ((msa = new QLibrary(LIBVKONTAKTE)) == 0)
        qFatal("Problems with initiation QLibrary object");

    msa->setLoadHints(QLibrary::ResolveAllSymbolsHint); // analog RTLD_LAZY
    // Check driver existance
    if (!msa->load())
        qFatal((QString("can't load driver library: ") + msa->errorString()).toUtf8());

    if ((driverModule = new struct msa_module) == 0)
        qFatal("can't init msa_module");

    // fill important properties
    driverModule->proxy = 0;
    driverModule->port = 0;

    // This is a magic stub for master branch of libvkontakte
    driverModule->id = (gchar*) "vkontakte";
    /*driverModule->set_events = vkCurEvents;*/
    // end magic stub

    gint (*driver_init)(msa_module *);

    *(void **) (&driver_init) = msa->resolve(MODULE_INIT);
    (*driver_init)(driverModule);

    if (driverModule == 0)
        qFatal((QString("can't resolve funtion of  library: ")+msa->errorString()).toUtf8());

    // Decoder of "russian" messages from libvkontakte
    decoder =  QTextCodec::codecForName("utf8");
}

VkTransport::~VkTransport()
{
    if (msa != 0) {
        if (msa->unload())
            qWarning((QString("can't unload  library: ")+msa->errorString()).toUtf8());
        delete msa;
        qDebug() << "vktransport deleted";
    }

}

void VkTransport::vkCreateRemoved()
{
    qDebug() << "VkTransport::vkCreateRemoved";
    // Find the locally-deleted UIDs for this account
    cannotPurgeMessageRemovalRecords = false;
    removed.clear();
    foreach (const QMailMessageRemovalRecord& r, QMailStore::instance()->messageRemovalRecords(id)) {
        qDebug() << "In removed:" << r.serverUid();
        removed.insert(r.serverUid());
    }
}

void VkTransport::vkCleareRemoved() {
    qDebug() << "VkTransport::vkCreateRemoved";
    if (!cannotPurgeMessageRemovalRecords)
        QMailStore::instance()->purgeMessageRemovalRecords(id);
    removed.clear();
}

void VkTransport::vkGetProfile()
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='profile' function='getProfileNoImage'>")
              + QString("<Params /></Request></Content></TransitData>");
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkGetProfile(QByteArray captchaCode, QByteArray captchaText)
{
    qDebug() << "Captcha code:" << captchaCode
             << "Captcha text:" << captchaText;
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content>")
              + QString("<Request class='systemMessages' function = 'captchaMessage'>")
              + QString("<Params>")
              + QString("<string name='moduleName'>vk</string>")
              + QString("<string name='code'>%1</string>").arg(QString(captchaCode))
              + QString("<string name='text'>%1</string>").arg(QString(captchaText))
              + QString("<img name='Img'></img>")
              + QString("<string name='oldRequest'>")
              + QString("<Request class='profile' function='getProfileNoImage'>")
              + QString("<Params>")
              + QString("<string name='fcsid'>%1</string>").arg(QString(captchaCode))
              + QString("<string name='fccode'>%1</string>").arg(QString(captchaText))
              + QString("</Params>")
              + QString("</Request>")
              + QString("</string></Params></Request></Content></TransitData>");
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkGetSend(int reqFrom = 0,
                             int reqTo = VK_MESSAGES_AT_ONCE)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetSend("<< reqFrom << ", " << reqTo <<");";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
       << "<TransitData id='1' type='data'>"
       << "<SourceID>" << CLIENT_NAME << "</SourceID>"
       << "<TargetID>" << this->driverModule->id << "</TargetID>"
       << "<Content><Request class='messages' function = 'updateOutboxMessages'>"
       << "<Params>"
       << "<number name='from'>" << reqFrom << "</number>"
       << "<number name='to'>" << reqTo << "</number>"
       << "</Params></Request></Content></TransitData>";
    ts.flush();
    vkSendRequest(reqStr);
}

void VkTransport::vkGetInbox(int reqFrom = 0,
                             int reqTo = VK_MESSAGES_AT_ONCE)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__
             << ": Called vkGetInbox("<< reqFrom << ", " << reqTo <<");";
    // Add XML data to "request" string
    QByteArray reqStr;
    QTextStream ts(&reqStr);
    ts << "<?xml version='1.0'?>"
       << "<TransitData id='1' type='data'>"
       << "<SourceID>" << CLIENT_NAME << "</SourceID>"
       << "<TargetID>" << this->driverModule->id << "</TargetID>"
       << "<Content><Request class='messages' function = 'updateInboxMessages'>"
       << "<Params>"
       << "<number name='from'>" << reqFrom << "</number>"
       << "<number name='to'>" << reqTo << "</number>"
       << "</Params></Request></Content></TransitData>";
    ts.flush();
    vkSendRequest(reqStr);
    //qDebug() << "vkGetInbox(): OK" << endl;
}

void VkTransport::vkGetSend() {
    // response only one message for checking of new outgoing messages.
    vkGetSend(0, VK_MIN_MESSAGES_AT_ONCE);
}

void VkTransport::vkGetInbox() {
    // response only one message for checking of new incoming messages.
    vkGetInbox(0, VK_MIN_MESSAGES_AT_ONCE);
}

/** vkInit()
  * @param strProxyHost
  * @param strProxyPort
  */
void VkTransport::vkInit(QString proxyHost, uint proxyPort)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkInit()";

    if (proxyHost.isEmpty()) {
        driverModule->proxy = (gchar*) NULL;
        driverModule->port = (gint) 0;
    } else {
        char *host = new char[proxyHost.toLocal8Bit().size()+1];
        strcpy(host,proxyHost.toLocal8Bit().data());
        driverModule->proxy = (gchar*) host;
        driverModule->port = (gint) proxyPort;
    }
    // Converting function's arguments

    // Send data to driver module
    qDebug() << "vkInit(): OK" << endl;
}

/** vkGetSettings()
  */
void VkTransport::vkGetSettings()
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkGetSettings()";

    QString reqStr = QString("<?xml version='1.0'?>")
                     + QString("<TransitData id='1' type='data'>")
                     + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
                     + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
                     + QString("<Content>")
                     + QString("<Request class='settings' function='getSettings'>")
                     + QString("<Params/>")
                     + QString("</Request></Content></TransitData>");
    Q_UNUSED(reqStr);
}

/** vkSetSettings()
  * @param strLogin
  * @param strPassword
  */
void VkTransport::vkSetSettings(QString strLogin, QString strPassword)
{
    // Debug message for calling function
    qDebug() << __FILE__<<":"<< __LINE__<<": Call vkSetSettings()";

    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='settings' function='setSettings'>")
              + QString("<Params>")
              + QString("<string name='login'>%1</string>").arg(strLogin)
              + QString("<string name='password'>%1</string>").arg(strPassword)
              + QString("<number name='autoupdate'>60</number>") // Deprecated
              + QString("<boolean name='active'>1</boolean>")    // Deprecated
              + QString("</Params></Request></Content></TransitData>");
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkSendMessage(QByteArray strRecipientId, QByteArray strText="")
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='messages' function='sendMessage'>")
              + QString("<Params id='vk%1'>").arg(QString(strRecipientId))
              + QString("<string name='text'>%1</string>").arg(QString(strText))
              + QString("</Params></Request></Content></TransitData>");
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkReadMessage(QString messageId)
{
    qDebug() << "init VkTransport::vkReadMessage";
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='messages' function='readMessage'>")
              + QString("<Params>")
              + QString("<string name='messageId'>vk%1</string>").arg(messageId)
              + QString("</Params></Request></Content></TransitData>");
    vkSendRequest(reqStr.toUtf8());
}

void VkTransport::vkDeleteMessage(QString messageId)
{
    QString reqStr = QString("<?xml version='1.0'?>")
              + QString("<TransitData id='1' type='data'>")
              + QString("<SourceID>%1</SourceID>").arg(CLIENT_NAME)
              + QString("<TargetID>%1</TargetID>").arg(this->driverModule->id)
              + QString("<Content><Request class='messages' function='deleteMessage'>")
              + QString("<Params>")
              + QString("<string name='messageId'>vk%1</string>").arg(messageId)
              + QString("</Params></Request></Content></TransitData>");
    vkSendRequest(reqStr.toUtf8());
}

// Deprecated
void VkTransport::parseOneNodeSet(xmlXPathObjectPtr &object,
                                  xmlChar *xpathRequest,
                                  xmlXPathContextPtr &content) {
    object = NULL;
    if ((object = xmlXPathEvalExpression(xpathRequest, content)) == NULL) {
        qDebug() << __FILE__<< ":" << "error in xmlXPathEvalExpression" << endl;
        return;
    }
    //qDebug() << "Xpath object type: " << object->type << (char *) object->stringval;
    if ((object->type != XPATH_NODESET)
        || ( xmlXPathNodeSetIsEmpty(object->nodesetval))
        || (xmlXPathNodeSetGetLength(object->nodesetval) != 1)
        ) {
        //xmlXPathFreeObject(object);
        qDebug() << __FILE__<< ":" << __LINE__
                << " incorrext response or XPath request type " << endl;
        emit errorOccurred(-1,"incorrext response or XPath request");
    }
}

char* VkTransport::parseString(QByteArray charRequest,
                                       xmlXPathContextPtr &content,
                                       bool important=true) {
    xmlChar *xrequest = xmlCharStrdup(charRequest.data());
    char* xresponse = parseString(xrequest, content, important);
    free(xrequest);
    return xresponse;
}

char* VkTransport::parseString(xmlChar *xpathRequest,
                                       xmlXPathContextPtr &content,
                                       bool important = true) {
    xmlXPathObjectPtr object = 0;
    if (!(object = xmlXPathEvalExpression(xpathRequest, content))) {
        qDebug() << __FILE__<< ":" << "error in xmlXPathEvalExpression";
        return 0;
    }
    if ((object->type != XPATH_NODESET)
        || ( xmlXPathNodeSetIsEmpty(object->nodesetval))
        || (xmlXPathNodeSetGetLength(object->nodesetval) != 1)
        || (!object->nodesetval->nodeTab[0])
        || (!object->nodesetval->nodeTab[0]->children)
        || (!object->nodesetval->nodeTab[0]->children->content)
        ){
        xmlXPathFreeObject(object);
        if (important) {
            qDebug() << "Requested: " << (char *) xpathRequest;
            qDebug() << __FILE__<< ":" << __LINE__
                    << " incorrect response or XPath request type";
            emit errorOccurred(-1,"incorrext response or XPath request");
        }
        return 0;
    }
    char *data = new char[strlen((char*) object->nodesetval->nodeTab[0]->
                                 children->content) + 1];
    if (!data) {
        qWarning() << "Not Enough Memory";
        return 0;
    }
    strcpy(data, (char*) object->nodesetval->nodeTab[0]->children->content);
    xmlXPathFreeObject(object);
    return data;
}

void VkTransport::vkClose()
{

}

void VkTransport::vkGetNewInbox()
{
}

void VkTransport::vkSendRequest(QByteArray xmlRequest)
{
#ifdef VK_LOGING
    qDebug() << "Generated request: " << xmlRequest;
#endif
    VkResponse* response = new VkResponse();
    response->text = NULL;
    xmlDocPtr reqXml = xmlParseDoc(xmlCharStrdup(xmlRequest.constData()));
#ifdef VK_LOGING
    xmlDocDump(vkLog, reqXml);
#endif
    xmlDocPtr respXml = NULL;
    // Process request
    if (driverModule != NULL) {
        qDebug() << "Sending: " << xmlRequest;
        try {
            driverModule->send(reqXml, &respXml, driverModule);
        } catch (...) {
            qDebug() << "Caught an exception.";
            emit errorOccurred(-1,tr("Caught an exception."));
        }
    } else {
        qDebug() << "driverModule == NULL" << endl;
    }

#ifdef VK_LOGING
    xmlDocDump(vkLog, respXml);
#endif
    // Parsing from DOM to XPath Content (Need free memory)
    xmlXPathContextPtr respContext = xmlXPathNewContext(respXml);
    if (respContext == NULL) {
        qDebug() << __FILE__ << ":" << "error in getting xmlXPathNewContext";
        return;
    }
    char xpathRequestBuffer[255]; // Buffer for generating xpath requests
    char *xpathRequestBufferPointer = xpathRequestBuffer;
    char *xpathRequestPattern = stpcpy (xpathRequestBuffer, VK_STRUCT);
    char *parsedString;

    // Getting type of response
    char *functionName  = parseString(VK_FUNCTION, respContext);
    if (functionName == NULL) {
        qDebug() << "functionName not found" << endl;
        return;
    }
    qDebug() << "Processing " << functionName << endl;

    bool error=false; // true if was error message
    bool sent=false;  // true if incoming sent messages (not inbox)

    // Info/Error message (Errors it problems with network or account)
    // Avoid duplication of code. Combine errors and info in one.
    if ((error = (strcmp(functionName,"errorMessage") == 0))
        || (strcmp(functionName,"infoMessage") == 0)) {

        // error == true if was error message
        if (error) {
            response->action = VkTransport::errorMessageAction;
            qDebug("Error message:");
        } else {
            response->action = VkTransport::infoMessageAction;
            qDebug("Info message:");
        }

        if ((parsedString = parseString(VK_RESP_CODE,respContext)) != 0) {
            qDebug() << "Code: " << parsedString;
            response->code = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            response->code = "-1";
            qWarning() << "Info/Error code not found.";
        }

        if ((parsedString = parseString(VK_RESP_TEXT,respContext)) != 0) {
            qDebug() << "Text: " << decoder->toUnicode(parsedString);
            response->text = new QString(parsedString);
            delete [] parsedString;
        } else {
            response->text = new QString("Some message");
            qWarning() << "Info/Error text not found.";
        }

        error = false;
        emit responseReceived(response);
        response = 0;
    } else
    // Captcha message
    if (strcmp(functionName,"captchaMessage") == 0) {
        /*
         * Here will be captcha message
         */
        QByteArray code;
        QByteArray img;
        //QByteArray old;

        if ((parsedString = parseString(VK_FCSID,respContext)) != NULL) {
            code = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "Captcha code not found.";
        }

        if ((parsedString = parseString(VK_IMG,respContext)) != NULL) {
            img = QByteArray(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "Captcha img not found.";
        }

        response->action = VkTransport::captchaMessageAction;
        response->code = code;
        response->captchaImg = img;
        emit responseReceived(response);
    } else
    // Profile message
    if ((strcmp(functionName,"updateProfile") == 0)
        || (strcmp(functionName,"getProfileNoImage") == 0)) {
        userName.clear();
        if ((parsedString = parseString(VK_FIRST_NAME,respContext)) != 0) {
            userName = decoder->toUnicode(parsedString);
            delete [] parsedString;
        } else {
            qWarning() << "FirstName not found.";
        }

        if ((parsedString = parseString(VK_LAST_NAME,respContext)) != 0) {
            if (!userName.isEmpty())
                userName.append(QChar(' '));
            userName.append(decoder->toUnicode(parsedString));
            delete [] parsedString;
        } else {
            qWarning() << "LastName not found.";
        }

        response->text = new QString(userName);
        response->action = VkTransport::updateProfileAction;
        emit responseReceived(response);
    } else
    // Inbox/outbox message
    if ((sent = (strcmp(functionName,"updateOutboxMessages") == 0))
        || (strcmp(functionName,"updateInboxMessages") == 0)) {

        response->action = sent
                           ? VkTransport::updateOutboxMessagesAction
                           : VkTransport::updateInboxMessagesAction;

        // Parse From String
        if ((parsedString = parseString(VK_FROM,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"From\" not found" << endl;
            return;
        }
        int inboxFrom = QString(parsedString).toInt();
        delete [] parsedString;

        // Parse To String
        if ((parsedString = parseString(VK_TO,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"To\" not found" << endl;
            return;
        }
        int inboxTo = QString(parsedString).toInt();
        delete [] parsedString;

        // Parse To String
        if ((parsedString = parseString(VK_QUANTITY,respContext)) == 0) {
            qDebug() << __FILE__<< ":" << "\"Quantity\" not found" << endl;
            return;
        }
        int inboxQuantity = QString(parsedString).toInt();
        delete [] parsedString;

        qDebug() << "From =" << inboxFrom
                 << "To ="   << inboxTo
                 << "Quantity ="   << inboxQuantity;

        // returned 0 messages
        if (inboxFrom >= inboxTo) {
            qDebug() << "Retrivied none messages." << endl;
            response->text = new QString("inboxFrom >= inboxTo");
        }

        QMailMessage *message;

        // Parse messages from response
        for (int index=1; index<=inboxTo-inboxFrom; index++) {

            // if receiving chanceled or duplicated messages
            if (response->text)
                break;
            message = new QMailMessage();
            // Message id
            QByteArray messageUid;
            QByteArray recipientId;
            QByteArray senderId;
            char indexString[20]; // Only for message index;
            sprintf(indexString,"%d]", index); // Bad hack with ]
            xpathRequestBufferPointer = stpcpy (xpathRequestPattern, indexString);
            char *xpathRequestBufferCurrentPointer = xpathRequestBufferPointer;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferPointer, VK_ID_R);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                qDebug() << "Message id " << index << " have uid = " << parsedString;
                messageUid = QByteArray(parsedString).remove(0,2);
                delete [] parsedString;
            } else {
                qDebug() << "Message uid not found. Message passed." << endl;
                continue;
            }

            // RecipientId
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_RECIPIENT_ID);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                recipientId = QByteArray(parsedString).remove(0,2);
                // Note: Using parsedString+2 more performance, but dangerously,
                // because response from lib can be less than 2 chars.
                delete [] parsedString;
            }

            // SenderId
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_SENDER_ID);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                senderId = QByteArray(parsedString).remove(0,2);
                delete [] parsedString;
            }

            if (sent)
                message->setServerUid(senderId+VK_UID_SEPARATOR+messageUid);
            else
                message->setServerUid(recipientId+VK_UID_SEPARATOR+messageUid);

            const QMailMessageKey serverUidKey
                    = QMailMessageKey::serverUid(message->serverUid());
            const QMailMessageKey parentAccountIdKey
                    = QMailMessageKey::parentAccountId(id);
            int matching = QMailStore::instance()->
                           countMessages(serverUidKey & parentAccountIdKey);

            // if we already retrivied this message, then end processing;
            if ((matching != 0) || (removed.contains(message->serverUid()))) {
                if (!matching) {
                    qDebug() << "found message in removed";
                    cannotPurgeMessageRemovalRecords = true;
                }
                response->text = new QString("Loaded exist message");
                break;
            }

            if (sent) {
                message->setStatus(QMailMessage::Sent | QMailMessage::Outgoing, true);
                message->setStatus(QMailMessage::Incoming | QMailMessage::New, false);
            } else {
                message->setStatus(QMailMessage::Incoming, true);
                message->setStatus(QMailMessage::New, true);
            }
            QByteArray senderName;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_SENDER_NAME);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, false)) != 0) {
                senderName = QByteArray(parsedString);
                delete [] parsedString;
            }

            QByteArray recipientName;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_RECIPIENT_NAME);
            if ((parsedString = parseString(xpathRequestBuffer, respContext, false)) != 0) {
                recipientName = QByteArray(parsedString);
                delete [] parsedString;
            }

            const QByteArray vkFakeAdress = QByteArray(VK_FAKE_ADDRESS);
            if (sent) {
                message->setFrom(QMailAddress(userName, senderId + vkFakeAdress));
                message->setTo(QMailAddress(decoder->toUnicode(recipientName), recipientId + vkFakeAdress));
            } else {
                message->setFrom(QMailAddress(decoder->toUnicode(senderName), senderId + vkFakeAdress));
                message->setTo(QMailAddress(userName, recipientId + vkFakeAdress));
            }

            // Time
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TIME);
            if ((parsedString = parseString(xpathRequestBuffer, respContext)) != 0) {
                QDateTime mdf;
                mdf.setTime_t(QString(parsedString).toUInt());
                // message->setReceivedDate(); Current time
                message->setDate(QMailTimeStamp(mdf));
                delete [] parsedString;
            }

            // Text
            QString messageBody;
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_TEXT);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                message->setBody(QMailMessageBody::fromData (
                                            decoder->toUnicode(parsedString),
                                            QMailMessageContentType("text/plain; charset=UTF-8"),
                                            QMailMessageBodyFwd::NoEncoding // working with 7&8 bit
                                        ));
                messageBody = message->body().data();
                //qDebug() << "Text:" << messageBody << endl;
                delete [] parsedString;
            }

            message->setSubject(vkSubjectFromBody(messageBody));

            // Status of message: 0 - new (unread), 1 - old (seen).
            xpathRequestBufferPointer = stpcpy (xpathRequestBufferCurrentPointer, VK_STATUS);
            if ((parsedString = parseString(xpathRequestBuffer,respContext)) != 0) {
                const bool readStatus = *parsedString!='0' || sent;
                message->setStatus(QMailMessage::Read, readStatus);
                message->setStatus(QMailMessage::ReadElsewhere, readStatus);
                delete [] parsedString;
            }

            emit pushNewMessage(message);
        }

        response->midFrom = QByteArray(QByteArray::number(inboxTo));
        response->midTo = QByteArray(QByteArray::number(inboxQuantity));

        if (inboxTo == inboxQuantity) {
            qDebug() << "Retrivied last message." << endl;
            response->text = new QString("inboxTo == inboxQuantity");
        }

        emit responseReceived(response);
    } else {
        qDebug() << "Impossible response: " << functionName << ". Passed.";
        delete response;
    }

    delete [] functionName;
    // Clear memory
    xmlXPathFreeContext(respContext);
    xmlFreeDoc(respXml);
}

QString VkTransport::vkUserName() const {
    return userName;
}

QString VkTransport::vkSubjectFromBody(QString messageBody) {
    // Choose subject for message as begin of body
    if (messageBody.isEmpty()) {
        return VK_DEFAULT_SUBJECT;
    } else if (messageBody.length() <= VK_SUBJECT_LENGTH) {
        return messageBody;
    } else {
        messageBody.truncate(VK_SUBJECT_LENGTH);
        messageBody.truncate(messageBody.lastIndexOf(QRegExp("\\s")));
        messageBody = messageBody.simplified();
        messageBody.append(VK_DEFAULT_SUBJECT);
        return messageBody;
    }
}

void VkTransport::vkSetId(QMailAccountId id) {
    this->id = QMailAccountId(id);
}

VkResponse::VkResponse()
    : text(0)
{
}

VkResponse::~VkResponse()
{
    delete text;
}
