/**********************************************************************************************
    Copyright (C) 2008 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

**********************************************************************************************/
#ifndef CTRACKDB_H
#define CTRACKDB_H

#include <QObject>
#include <QPointer>
#include <QMap>
#include <QSqlDatabase>


class CCanvas;
class IGps;
class CTrack;
class QPainter;
class QRectF;
class QColor;
class QPixmap;
struct STrackPoint;

class CTrackDB : public QObject
{
    Q_OBJECT;
    public:
        virtual ~CTrackDB();

        static CTrackDB& self(){return *m_self;}

        enum state_e
        {
             eRecord
            ,ePause
            ,eStop
        };

        state_e getState(){return state;}

        /// draw current track
        /**
            @param p the paint device
            @param rect the viewport in [m]
        */
        void draw(QPainter& p, const QRectF& rect);

        bool add(CTrack &trk, QString& errmsg);
        /// delete track by key
        bool del(quint32 trackid, QString& errmsg);
        /// delete all tracks
        bool delAll(QString& errmsg);
        /// get track by key
        bool get(const QString& key, CTrack& trk, QString& errmsg);
        /// get Track information from key
        bool getTrackInfo(QString key, CTrack &track, QString& errmsg);
        /// update Track information
        bool updateTrackInfo(CTrack &track, QString& errmsg);
        /// load track as secondary track
        bool show(const QString& key, QString& errmsg);

        /// get current recording interval setting
        int getInterval(){return intvl;}
        /// get current extended recording setting
        bool getExtended(){return ext;}
        /// set current recording interval
        void setInterval(int i);
        /// set current extended recording
        void setExtended(bool on);
        /// List all track recorded
        bool list(QStringList& keys, QStringList& names, QStringList& icons, QString& errmsg);

    signals:
        void sigChanged();
        void sigProgress(int p);

    public slots:
        /// start recording a new track
        void record();
        /// pause recording
        void pause();
        /// stop recording
        void stop();

    private slots:
        void slotNewGPS(IGps* dev);
        void slotNewGPSData(IGps& gps);

    private:
        friend class CCanvas;
        friend class CDlgTrack;
        CTrackDB(CCanvas * parent);

        void drawTrack(QPainter& p, const QRectF& rect, STrackPoint * track, quint32 idx, const QColor& c, const QPixmap& bullet);
        bool storeTrackPoint(STrackPoint& trkpt);

        void initDB();
        void migrateDB(QString &version);

        static CTrackDB * m_self;

        state_e state;

        /// the current GPS object
        QPointer<IGps> gps;
        /// azimuth[n - 1] used for track optimization
        float prevAzimuth;
        /// recording interval 1..5 sec or -1 for optimization
        int intvl;
        /// counter for interval recording
        int intvlcnt;
        /// record extended information
        bool ext;

        /// the database file path
        QString dbName;
        /// use this database object exclusively
        QSqlDatabase db;
        // track id for db
        int trackid;
        // Tracl color;
        int colorIdx;
        int colorIdx1;
        // Track distance
        QString computeTrackDistance(int trackid);
    };

#endif //CTRACKDB_H
