/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "keyboard.h"
#include "ui_keyboard.h"

#include <qimsysdebug.h>
#include <qimsysapplicationmanager.h>
#include <qimsyscommandlineargs.h>

#include <QDir>
#include <QTimer>
#include <QDesktopWidget>

#include <QDBusInterface>

#include <QGLWidget>

#include <QDeclarativeEngine>
#include <QDeclarativeContext>

using namespace N900::Keyboard;

class Keyboard::Private : private QObject
{
    Q_OBJECT
public:

    Private(QimsysAbstractPluginObject *object, Keyboard *parent);
    ~Private();

public slots:
    void setGeometry();

private slots:
    void init();

    void focusChanged(QWidget *previous, QWidget *current);

    void focusChanged(bool focus);
    void keyboardChanged(bool keyboard);
    void slideChanged();

private:
    void setupUi();
    void setVisible(bool visible);

private:
    Keyboard *q;
    Ui::Keyboard ui;
    QimsysAbstractPluginObject *plugin;

    QimsysApplicationManager applicationManager;
public:
    QWidget *lastFocusedWidget;
};

Keyboard::Private::Private(QimsysAbstractPluginObject *object, Keyboard *parent)
    : QObject(parent)
    , q(parent)
    , plugin(object)
    , lastFocusedWidget(0)
{
    qimsysDebugIn() << object << parent;
    // race condition?
//    metaObject()->invokeMethod(this, "init", Qt::QueuedConnection);
    QTimer::singleShot(100, this, SLOT(init()));
    qimsysDebugOut();
}

Keyboard::Private::~Private()
{
    qimsysDebugIn();
    qimsysDebugOut();
}

void Keyboard::Private::init()
{
    qimsysDebugIn();

    applicationManager.init();
    connect(&applicationManager, SIGNAL(focusChanged(bool)), this, SLOT(focusChanged(bool)));
    connect(&applicationManager, SIGNAL(keyboardChanged(bool)), this, SLOT(keyboardChanged(bool)));

    QDBusConnection::systemBus().connect(QLatin1String("org.freedesktop.Hal"), QLatin1String("/org/freedesktop/Hal/devices/platform_slide"), QLatin1String("org.freedesktop.Hal.Device"), QString("PropertyModified"), this, SLOT(slideChanged()));

    setupUi();

    qimsysDebugOut();
}

void Keyboard::Private::setupUi()
{
    qimsysDebugIn();

    q->setWindowFlags(Qt::Window | Qt::WindowStaysOnTopHint | Qt::X11BypassWindowManagerHint | Qt::FramelessWindowHint);
    connect(QApplication::instance(), SIGNAL(focusChanged(QWidget*, QWidget*)), this, SLOT(focusChanged(QWidget*,QWidget*)));
    q->setFocusPolicy(Qt::NoFocus);

    // ui part
    ui.setupUi(q);

    QVariant url = QimsysCommandlineArgs::instance()->value("Virtual Keyboard");
    if (url.isNull()) {
        ui.declarativeView->setSource(ui.declarativeView->resolveUrl("N900/Keyboard/main.qml"));
    } else {
        ui.declarativeView->setSource(url.toUrl());
    }

    qimsysDebugOut();
}

void Keyboard::Private::setVisible(bool visible)
{
    if (visible) {
        connect(QApplication::desktop(), SIGNAL(workAreaResized(int)), this, SLOT(setGeometry()));
        connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(setGeometry()));
        setGeometry();
    } else {
        disconnect(QApplication::desktop(), SIGNAL(workAreaResized(int)), this, SLOT(setGeometry()));
        disconnect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(setGeometry()));
    }
    q->setVisible(visible);
}

void Keyboard::Private::focusChanged(QWidget *previous, QWidget *current)
{
    qimsysDebugIn() << previous << current;
    if (current != 0 && !q->isAncestorOf(current)) {
        qimsysDebug() << previous << current;
        lastFocusedWidget = current;
    }
    qimsysDebugOut();
}

void Keyboard::Private::slideChanged()
{
    qimsysDebugIn();

    QDBusInterface propertyInterface(QLatin1String("org.freedesktop.Hal"), QLatin1String("/org/freedesktop/Hal/devices/platform_slide"), QLatin1String("org.freedesktop.Hal.Device"), QDBusConnection::systemBus());
    bool closed = propertyInterface.call(QLatin1String("GetProperty"), QLatin1String("button.state.value")).arguments().at(0).toBool();

    if (!applicationManager.focus())
        applicationManager.blockSignals(true);
    applicationManager.setKeyboard(closed);
    if (!applicationManager.focus())
        applicationManager.blockSignals(false);
    qimsysDebugOut();
}

void Keyboard::Private::setGeometry()
{
    qimsysDebugIn();
    QRect desktop = QApplication::desktop()->screenGeometry();
#ifdef QIMSYS_PLATFORM_MAEMO
    desktop.setRight(desktop.right() + 5);
    desktop.setTop(desktop.top() + desktop.height() / 2);
#else
    desktop.setWidth(desktop.width() / 2);
    desktop.moveLeft(desktop.width());
    desktop.setHeight(desktop.height() / 3);
    desktop.moveTop(desktop.height());
#endif
    qimsysDebug() << desktop;
    if (q->geometry() != desktop) {
        q->setGeometry(desktop);
        if (q->isVisible()) {
            q->hide();
            QMetaObject::invokeMethod(q, "show", Qt::QueuedConnection);
        }
    }
    qimsysDebugOut();
}

void Keyboard::Private::focusChanged(bool focus)
{
    qimsysDebugIn() << focus;
    setVisible(focus && applicationManager.keyboard());
    qimsysDebugOut();
}

void Keyboard::Private::keyboardChanged(bool keyboard)
{
    qimsysDebugIn() << keyboard;
    setVisible(keyboard);
    qimsysDebugOut();
}

Keyboard::Keyboard(QimsysAbstractPluginObject *plugin, QWidget *parent)
    : QWidget(parent)
{
    qimsysDebugIn();
    d = new Private(plugin, this);
    qimsysDebugOut();
}

Keyboard::~Keyboard()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

bool Keyboard::event(QEvent *e)
{
    switch(e->type()) {
    case QEvent::WindowActivate:
//        qimsysDebugIn();
        if (d->lastFocusedWidget)
            d->lastFocusedWidget->activateWindow();
//        qimsysDebugOut();
        break;
    default:
        break;
    }
    return QWidget::event(e);
}

#include "keyboard.moc"
