/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "inputmethod.h"
#include "global.h"
#include "keyactions.h"

#include <qimsysdebug.h>
#include <qimsysapplicationmanager.h>
#include <qimsysinputmethodmanager.h>
#include <qimsyskeymanager.h>
#include <qimsyspreeditmanager.h>
#include <qimsyscandidatemanager.h>
#include <qimsyspluginmanager.h>
#include <qimsysconverter.h>
#include <qimsysinterpreter.h>
#include <qimsysengine.h>

#include <QKeySequence>
#include <QPainter>
#include <QPixmap>
#include <QTimer>

using namespace Japanese;
using namespace Japanese::Standard;

class InputMethod::Private : private QObject
{
    Q_OBJECT
public:
    Private(InputMethod *parent);
    ~Private();

private slots:
    void init();
    void activeChanged(bool active);

    void stateChanged(uint state);
    void converterChanged(const QString &identifier);
    void interpreterChanged(const QString &identifier);
    void engineChanged(const QString &identifier);

    void keyPressed(const QString &text, int keycode, int modifiers, bool autoRepeat);
//    void keyReleased(const QString &text, int keycode, int modifiers, bool autoRepeat);

    void currentIndexChanged(int currentIndex);

    void resetIcon();
private:
    void updateIcon(const QIcon &overlay);

private:
    InputMethod *q;

    QimsysApplicationManager *applicationManager;
    QimsysInputMethodManager *inputMethodManager;
    QimsysKeyManager *keyManager;
    QimsysPreeditManager *preeditManager;
    QimsysCandidateManager *candidateManager;
    KeyActions *keyActions;

    QTimer resetIconTimer;
};

InputMethod::Private::Private(InputMethod *parent)
    : QObject(parent)
    , q(parent)
    , applicationManager(0)
    , inputMethodManager(0)
    , keyManager(0)
    , preeditManager(0)
    , candidateManager(0)
    , keyActions(0)
{
    qimsysDebugIn() << parent;
    init();
    qimsysDebugOut();
}

InputMethod::Private::~Private()
{
    qimsysDebugIn();
    qimsysDebugOut();
}

void InputMethod::Private::init()
{
    q->setIdentifier(QLatin1String("Japanese(Standard)"));
    q->setPriority(0x10);

    q->setLocale("ja_JP");
    q->setLanguage("Japanese");

    q->setIcon(QIcon(":/japanese/standard/resources/japanese.png"));
    q->setName(QT_TR_NOOP("Japanese(Standard)"));
    q->setAuthor(QT_TR_NOOP("Tasuku Suzuki"));
    q->setTranslator(QT_TR_NOOP("None"));
    q->setDescription(QT_TR_NOOP("Japanese language"));

    q->setGroups(QStringList() << QLatin1String("X11 Classic"));
    q->setCategoryType(MoreThanOne);
    q->setCategoryName(QT_TR_NOOP("Input/InputMethod"));

    connect(q, SIGNAL(activeChanged(bool)), this, SLOT(activeChanged(bool)));

    resetIconTimer.setInterval(2000);
    resetIconTimer.setSingleShot(true);
    connect(&resetIconTimer, SIGNAL(timeout()), this, SLOT(resetIcon()));
}

void InputMethod::Private::activeChanged(bool active)
{
    qimsysDebugIn();
    if (active) {
        if (!applicationManager) {
            applicationManager = new QimsysApplicationManager(this);
            applicationManager->init();
        }

        if (!inputMethodManager) {
            inputMethodManager = new QimsysInputMethodManager(this);
            inputMethodManager->init();
            inputMethodManager->setState(Direct);
            connect(inputMethodManager, SIGNAL(stateChanged(uint)), this, SLOT(stateChanged(uint)), Qt::QueuedConnection);
            connect(inputMethodManager, SIGNAL(converterChanged(QString)), this, SLOT(converterChanged(QString)), Qt::QueuedConnection);
            connect(inputMethodManager, SIGNAL(interpreterChanged(QString)), this, SLOT(interpreterChanged(QString)), Qt::QueuedConnection);
            connect(inputMethodManager, SIGNAL(engineChanged(QString)), this, SLOT(engineChanged(QString)), Qt::QueuedConnection);
        }
        QMetaObject::invokeMethod(this, "resetIcon", Qt::QueuedConnection);

        if (!keyManager) {
            keyManager = new QimsysKeyManager(this);
            keyManager->init();
            connect(keyManager, SIGNAL(keyPressed(QString,int,int,bool)), this, SLOT(keyPressed(QString,int,int,bool)));
        }

        if (!preeditManager) {
            preeditManager = new QimsysPreeditManager(this);
            preeditManager->init();
        }

        if (!candidateManager) {
            candidateManager = new QimsysCandidateManager(this);
            candidateManager->init();
            connect(candidateManager, SIGNAL(currentIndexChanged(int)), this, SLOT(currentIndexChanged(int)));
        }

        if (!keyActions) {
            keyActions = new KeyActions(this);
        }
    } else {
        if (applicationManager) {
            applicationManager->deleteLater();
            applicationManager = 0;
        }

        if (inputMethodManager) {
            disconnect(inputMethodManager, SIGNAL(stateChanged(uint)), this, SLOT(stateChanged(uint)));
            disconnect(inputMethodManager, SIGNAL(converterChanged(QString)), this, SLOT(converterChanged(QString)));
            disconnect(inputMethodManager, SIGNAL(interpreterChanged(QString)), this, SLOT(interpreterChanged(QString)));
            disconnect(inputMethodManager, SIGNAL(engineChanged(QString)), this, SLOT(engineChanged(QString)));
            inputMethodManager->deleteLater();
            inputMethodManager = 0;
        }

        if (keyManager) {
            disconnect(keyManager, SIGNAL(keyPressed(QString,int,int,bool)), this, SLOT(keyPressed(QString,int,int,bool)));
            keyManager->deleteLater();
            keyManager = 0;
        }

        if (preeditManager) {
            preeditManager->deleteLater();
            preeditManager = 0;
        }

        if (candidateManager) {
            disconnect(candidateManager, SIGNAL(currentIndexChanged(int)), this, SLOT(currentIndexChanged(int)));
            candidateManager->deleteLater();
            candidateManager = 0;
        }

        if (keyActions) {
            keyActions->deleteLater();
            keyActions = 0;
        }
    }
    qimsysDebugOut();
}

void InputMethod::Private::stateChanged(uint state)
{
    resetIcon();
    switch (state) {
    case Convert:
        candidateManager->setCurrentIndex(-1);
        break;
    default:
        break;
    }
}

void InputMethod::Private::converterChanged(const QString &identifier)
{
    qimsysDebugIn() << identifier;
    QList<QimsysConverter *> converters = QimsysPluginManager::objects<QimsysConverter>();
    foreach(QimsysConverter *converter, converters) {
        if (converter->identifier() == identifier) {
            updateIcon(converter->icon());
            break;
        }
    }
    if (resetIconTimer.isActive()) {
        resetIconTimer.stop();
    }
    resetIconTimer.start();
    qimsysDebugOut();
}

void InputMethod::Private::interpreterChanged(const QString &identifier)
{
    qimsysDebugIn() << identifier;
    QList<QimsysInterpreter *> interpreters = QimsysPluginManager::objects<QimsysInterpreter>();
    foreach(QimsysInterpreter *interpreter, interpreters) {
        if (interpreter->identifier() == identifier) {
            updateIcon(interpreter->icon());
            break;
        }
    }
    if (resetIconTimer.isActive()) {
        resetIconTimer.stop();
    }
    resetIconTimer.start();
    qimsysDebugOut();
}

void InputMethod::Private::engineChanged(const QString &identifier)
{
    qimsysDebugIn() << identifier;
    QList<QimsysEngine *> engines = QimsysPluginManager::objects<QimsysEngine>();
    foreach(QimsysEngine *engine, engines) {
        if (engine->identifier() == identifier) {
            updateIcon(engine->icon());
            break;
        }
    }
    if (resetIconTimer.isActive()) {
        resetIconTimer.stop();
    }
    resetIconTimer.start();
    qimsysDebugOut();
}

void InputMethod::Private::updateIcon(const QIcon &overlay)
{
    static qint64 cache = 0;
    if (cache == overlay.cacheKey()) return;
    qimsysDebugIn();
    cache = overlay.cacheKey();

    QIcon icon;
    foreach (const QSize &size, q->icon().availableSizes()) {
        QRect r;
        r.setSize(size);
        QPixmap pix = q->icon().pixmap(size);
        QPainter p(&pix);
        p.setOpacity(0.50);
        p.drawPixmap(r, overlay.pixmap(size));
        p.end();
        icon.addPixmap(pix);
    }

    applicationManager->setCurrentIcon(icon);
    qimsysDebugOut();
}

void InputMethod::Private::resetIcon()
{
    qimsysDebugIn();
    if (inputMethodManager) {
        switch (inputMethodManager->state()) {
        case Direct:
            updateIcon(QIcon());
            break;
        default:
            QList<QimsysConverter *> converters = QimsysPluginManager::objects<QimsysConverter>();
            foreach(QimsysConverter *converter, converters) {
                if (converter->identifier() == inputMethodManager->converter()) {
                    updateIcon(converter->icon());
                    break;
                }
            }
            break;
        }
    }
    qimsysDebugOut();
}

void InputMethod::Private::keyPressed(const QString &text, int keycode, int modifiers, bool autoRepeat)
{
    if (keyManager->isAccepted()) return;

//    qimsysDebugOn();
    qimsysDebugIn() << text << keycode << modifiers << autoRepeat;

    int key = keycode;
    if (modifiers & Qt::ControlModifier) key += Qt::CTRL;
    if (modifiers & Qt::AltModifier) key += Qt::ALT;
    if (modifiers & Qt::ShiftModifier) key += Qt::SHIFT;

    QKeySequence seq(key);
    if (keyActions->contains(seq)) {
        keyActions->trigger(seq);
        keyManager->accept();
        // reset
    } else if (!text.isEmpty()){
        const QChar &ch = text.at(0);
        qimsysDebug() << ch;
        uint state = inputMethodManager->state();
        if (ch.isPrint()) {
            switch (state) {
            case Direct:
                    break;
            case Convert:
            case Select:
                    inputMethodManager->execute(QLatin1String("Commit all"));
                    fallthrough;
            case Empty:
            case Input: {
                    // clear selection first
                    inputMethodManager->execute(QLatin1String("Clear Selection"));
                    inputMethodManager->setState(Input);

                    preeditManager->insert(ch);
                    keyManager->accept();
                }
            }
        } else {
            switch (state) {
            case Direct:
            case Empty:
                break;
            default:
                keyManager->accept();
                break;
            }
        }
    }

    qimsysDebugOut() << keyManager->isAccepted();
//    qimsysDebugOff();
}

void InputMethod::Private::currentIndexChanged(int currentIndex)
{
    if (currentIndex == -1) return;
    qimsysDebugIn() << currentIndex;

    QimsysPreeditItem item = preeditManager->item();
    int cursor = item.cursor;
    int index = 0;
    int pos = 0;
    for (int i = 0; i < item.to.length(); i++) {
        if (pos == cursor) {
            index = i;
            break;
        }
        pos += item.to.at(i).length();
    }
    qimsysDebug() << index;

    qimsysDebug() << candidateManager;
    qimsysDebug() << candidateManager->items();

    QString t = candidateManager->items().at(currentIndex).to;
    QString f = candidateManager->items().at(currentIndex).from;

    qimsysDebug() << item.to;

    if (item.to.isEmpty()) {
        item.to.append(t);
        item.from.append(f);
        item.rawString.append(f);
    } else {
        item.to.replace(index, t);
        item.from.replace(index, f);
    }
    item.selection = t.length();
    preeditManager->blockSignals(true);
    preeditManager->setItem(item);
    preeditManager->blockSignals(false);

    qimsysDebugOut();
}

InputMethod::InputMethod(QObject *parent)
    : QimsysInputMethod(parent)
{
    qimsysDebugIn() << parent;
    d = new Private(this);
    qimsysDebugOut();
}

InputMethod::~InputMethod()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

#include "inputmethod.moc"
