/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "simplelistwidget.h"
#include "ui_simplelistwidget.h"

#include <qimsysdebug.h>
#include <qimsysapplicationmanager.h>
#include <qimsyspreedit.h>
#include <translator.h>

#include <QKeyEvent>
#include <QDesktopWidget>

class SimpleListWidget::Private : private QObject
{
    Q_OBJECT
public:
    Private(SimpleListWidget *parent);
    ~Private();

    void setItems(const QStringList &list);
    void keyEvent(QKeyEvent *e);

public slots:
    void update();

private slots:
    void focusChanged(uint focus);
    void rectChanged(const QRect &rect);
    void fontChanged(const QFont &font);

private:
    QPoint pos(const QRect &rect) const;

private:
    SimpleListWidget *q;
public:
    Ui::SimpleListWidget ui;
    QimsysApplicationManager manager;
private:
    QimsysPreedit preedit;
};

SimpleListWidget::Private::Private(SimpleListWidget *parent)
    : QObject(parent)
    , q(parent)
{
    Translator::begin();
    ui.setupUi(q);
    Translator::ui(q);
    Translator::end();
    q->setWindowFlags(Qt::Window | Qt::WindowStaysOnTopHint | Qt::X11BypassWindowManagerHint | Qt::FramelessWindowHint);
    q->setFocusPolicy(Qt::NoFocus);
    q->setWindowOpacity(0.80);
    q->setLineWidth(1);
    q->setFrameShape(QFrame::Box);
    q->setFrameShadow(QFrame::Plain);

    QFontMetrics met(q->font());
    ui.label->setFixedWidth(met.width("WW"));
    for (int i = 0; i < 10; i++) {
        int k = (i + 1) % 10;
        QListWidgetItem *item = new QListWidgetItem(QString::number(k));
        item->setTextAlignment(Qt::AlignCenter);
        item->setData(Qt::UserRole, k + Qt::Key_0);
        ui.label->addItem(item);
    }

    manager.init();
    connect(&manager, SIGNAL(focusChanged(uint)), this, SLOT(focusChanged(uint)));

    preedit.init();
    connect(&preedit, SIGNAL(rectChanged(QRect)), this, SLOT(rectChanged(QRect)));
    connect(&preedit, SIGNAL(fontChanged(QFont)), this, SLOT(fontChanged(QFont)));

    connect(ui.listWidget, SIGNAL(currentRowChanged(int)), q, SIGNAL(currentIndexChanged(int)));

    metaObject()->invokeMethod(this, "update", Qt::QueuedConnection);
}

SimpleListWidget::Private::~Private()
{
}

void SimpleListWidget::Private::focusChanged(uint focus)
{
    if (!focus) {
        q->hide();
    }
}


void SimpleListWidget::Private::setItems(const QStringList &list)
{
    ui.listWidget->clear();

    QSize size(0, 0);
    QFontMetrics met(q->font());
    ui.label->setFixedWidth(met.width("WW"));
    foreach(const QString &str, list) {
        size.setWidth(qMax(size.width(), met.width(str)));
        size.setHeight(size.height() + met.height());
        ui.listWidget->addItem(str);
    }
    size.setWidth(ui.label->width() + size.width());
    if (list.count() > 10) {
        size.setWidth(size.width() + 30);   // TODO: margin should be calculated properly.
    } else {
        size.setWidth(size.width() + 10);   // TODO: margin should be calculated properly.
    }
    size.setHeight(qMin(size.height(), met.height()  *10));
    size.setHeight(size.height() + q->lineWidth()  *2);
    q->setFixedSize(size);
    update();
    QPalette palette = ui.label->palette();
    palette.setBrush(QPalette::Base, Qt::transparent);
    palette.setBrush(QPalette::Text, palette.windowText());
    ui.label->setPalette(palette);
    q->setVisible(!list.isEmpty() && manager.focus() > 0);
    q->setCurrentIndex(-1);
    ui.listWidget->scrollToTop();
}

void SimpleListWidget::setItems(const QStringList &list)
{
    d->setItems(list);
}

void SimpleListWidget::setCurrentIndex(int currentIndex)
{
    d->ui.listWidget->blockSignals(true);
    d->ui.listWidget->setCurrentRow(currentIndex);
    d->ui.listWidget->blockSignals(false);
    if (currentIndex < 0) {
        d->ui.label->setCurrentRow(currentIndex);
    } else {
        QRect r = d->ui.listWidget->visualItemRect(d->ui.listWidget->item(currentIndex));
        d->ui.label->setCurrentItem(d->ui.label->itemAt(0, r.center().y()));
    }
}

void SimpleListWidget::Private::update()
{
    rectChanged(preedit.rect());
}

void SimpleListWidget::Private::rectChanged(const QRect &rect)
{
    QPoint p = pos(rect);
    p.setX(p.x() - ui.label->width());
    q->move(p);
}

QPoint SimpleListWidget::Private::pos(const QRect &rect) const
{
    QRect desktop = QApplication::desktop()->availableGeometry();
    QPoint ret = rect.topLeft();
    if (rect.bottom() + q->height() < desktop.height()) {
        ret.setY(rect.bottom());
    } else if (rect.top() - q->height() > desktop.top()) {
        ret.setY(rect.top() - q->height());
    } else {
        ret.setY(rect.bottom() - q->height());
    }
    return ret;
}

void SimpleListWidget::Private::fontChanged(const QFont &font)
{
    q->setFont(font);
}

void SimpleListWidget::Private::keyEvent(QKeyEvent *e)
{
    if (!q->isVisible()) return;
    int key = e->key();
    if (key == Qt::Key_Control) {
        QPalette palette = ui.label->palette();
        if (e->type() == QEvent::KeyPress) {
            palette.setBrush(QPalette::Base, palette.highlight());
            palette.setBrush(QPalette::Text, palette.highlightedText());
        } else {
            palette.setBrush(QPalette::Base, Qt::transparent);
            palette.setBrush(QPalette::Text, palette.windowText());
        }
        ui.label->setPalette(palette);
        return;
    }
    if (e->type() != QEvent::KeyPress) return;
    if (!(e->modifiers() & Qt::ControlModifier)) return;
    for (int i = 0; i < ui.label->count(); i++) {
        if (e->key() == ui.label->item(i)->data(Qt::UserRole).toInt()) {
            QRect r = ui.label->visualItemRect(ui.label->item(i));
            ui.listWidget->setCurrentItem(ui.listWidget->itemAt(r.center()));
            e->accept();
            break;
        }
    }
}

SimpleListWidget::SimpleListWidget(QWidget *parent)
    : QFrame(parent)
{
    d = new Private(this);
}

SimpleListWidget::~SimpleListWidget()
{
    delete d;
}

void SimpleListWidget::keyEvent(QKeyEvent *e)
{
    d->keyEvent(e);
}

Qt::KeyboardModifiers SimpleListWidget::modifiers() const
{
    return Qt::NoModifier;
}

#include "simplelistwidget.moc"
