/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "n900.h"
#include "ui_n900.h"

#include <qimsysdebug.h>
#include <translator.h>
#include <setproperty.h>
#include <qbinding.h>

#include <qimsysapplicationmanager.h>
#include <qimsyspluginmanager.h>
#include <qimsysconverter.h>
#include <qimsyspreedit.h>
#include <qimsyskeymanager.h>
#include <qimsyscandidates.h>
#include <qimsysengine.h>


#include <QDesktopWidget>
#include <QSettings>
#include <QSignalMapper>

#include <QStateMachine>
#include <QSignalTransition>
#include <QParallelAnimationGroup>
#include <QSequentialAnimationGroup>
#include <QPropertyAnimation>
#include <QGraphicsOpacityEffect>
#include <QGraphicsBlurEffect>
#include <QEasingCurve>
#include <QTimer>
#include "qboolsignal.h"

class N900::Private : private QObject
{
    Q_OBJECT
public:

    Private(QimsysAbstractPluginObject *object, N900 *parent);
    ~Private();

public slots:
    void move();

private slots:
    void init();
    void focusChanged(QWidget *previous, QWidget *current);
    void stateChanged(N900::State state);
    void itemClicked(QListWidgetItem* item);

    void focusChanged(uint focus);
    void inputLanguageChanged(const QString &inputLanguage);
    void currentEngineChanged(const QString &currentEngine);
    void currentIconChanged(const QIcon &icon);

    void candidatesChanged(const QimsysConversionItemList &candidates);
    void candidateIndexChanged(int index);

    void rectChanged(const QRect &rect);

signals:
    void focusChanged(bool focus);

private:
    void setupUi();
    void setupState();
    void setMenu();

private:
    N900 *q;
    Ui::N900 ui;
    QStateMachine *machine;
    QimsysAbstractPluginObject *plugin;

    QimsysApplicationManager manager;
    QimsysPreedit preedit;
    QimsysCandidates candidates;

    int systemLevel;
    QList< QList<QAction*> > menuItems;

    QAction *langAction;
    QActionGroup *langGroup;
    QAction *engineAction;
    QActionGroup *engineGroup;

public:
    QWidget *lastFocusedWidget;
    N900::State state;
};

N900::Private::Private(QimsysAbstractPluginObject *object, N900 *parent)
    : QObject(parent)
    , q(parent)
    , machine(0)
    , plugin(object)
    , systemLevel(0)
    , langAction(0)
    , engineAction(0)
    , lastFocusedWidget(0)
    , state(NotAvailable)
{
    qimsysDebugIn() << object << parent;
    metaObject()->invokeMethod(this, "init", Qt::QueuedConnection);
    qimsysDebugOut();
}

N900::Private::~Private()
{
    qimsysDebugIn();
    qimsysDebugOut();
}

void N900::Private::init()
{
    qimsysDebugIn();

    // system
    manager.init();
    connect(&manager, SIGNAL(inputLanguageChanged(QString)), this, SLOT(inputLanguageChanged(QString)));
    connect(&manager, SIGNAL(focusChanged(uint)), this, SLOT(focusChanged(uint)));
    connect(&manager, SIGNAL(currentEngineChanged(QString)), this, SLOT(currentEngineChanged(QString)));
    connect(&manager, SIGNAL(currentIconChanged(QIcon)), this, SLOT(currentIconChanged(QIcon)));

    candidates.init();
    connect(&candidates, SIGNAL(currentIndexChanged(int)), this, SLOT(candidateIndexChanged(int)));
    connect(&candidates, SIGNAL(candidatesChanged(QimsysConversionItemList)), this, SLOT(candidatesChanged(QimsysConversionItemList)));

    preedit.init();
    connect(&preedit, SIGNAL(rectChanged(QRect)), this, SLOT(rectChanged(QRect)));

    connect(q, SIGNAL(stateChanged(N900::State)), this, SLOT(stateChanged(N900::State)));

    setupUi();
    setupState();
    inputLanguageChanged(manager.inputLanguage());
    currentEngineChanged(manager.currentEngine());

    qimsysDebugOut();
}

void N900::Private::setupUi()
{
    qimsysDebugIn();
    Translator::begin();
    ui.setupUi(q);
    Translator::ui(q);
    Translator::end();

    q->setWindowFlags(Qt::Window | Qt::WindowStaysOnTopHint | Qt::X11BypassWindowManagerHint | Qt::FramelessWindowHint);
    connect(QApplication::instance(), SIGNAL(focusChanged(QWidget*, QWidget*)), this, SLOT(focusChanged(QWidget*,QWidget*)));
    q->setFocusPolicy(Qt::NoFocus);

    QList<QAction *> systemMenu;
    QList<QAction *> subMenu;

    {
        langAction = new QAction(this);
        langGroup = new QActionGroup(this);

        QSignalMapper *signalMapper = new QSignalMapper(this);
        QList<QimsysConverter *> converters = QimsysPluginManager::objects<QimsysConverter>();
        foreach(QimsysConverter *converter, converters) {
            QAction *action = new QAction(langGroup);
            new QBinding(converter, "icon", action, "icon");
            new QBinding(converter, "name", action, "text");
            new QBinding(converter, "enabled", action, "visible");
            action->setData(converter->identifier());
            action->setParent(langAction);
            connect(action, SIGNAL(triggered()), signalMapper, SLOT(map()));
            signalMapper->setMapping(action, converter->identifier());
            subMenu.append(action);
        }
        menuItems.append(subMenu);
        systemMenu.append(langAction);
        connect(signalMapper, SIGNAL(mapped(QString)), &manager, SLOT(setInputLanguage(QString)));
    }

    {
        engineAction = new QAction(this);
        engineGroup = new QActionGroup(this);
        subMenu.clear();

        QSignalMapper *signalMapper = new QSignalMapper(this);
        QList<QimsysEngine *> engines = QimsysPluginManager::objects<QimsysEngine>();
        foreach (QimsysEngine *engine, engines) {
            QAction *action = new QAction(engineGroup);
            new QBinding(engine, "icon", action, "icon");
            new QBinding(engine, "name", action, "text");
            new QBinding(engine, "enabled", action, "visible");
            new QBinding(engine, "available", action, "enabled");
            action->setData(engine->identifier());
            action->setParent(engineAction);
            connect(action, SIGNAL(triggered()), signalMapper, SLOT(map()));
            signalMapper->setMapping(action, engine->identifier());
            subMenu.append(action);
        }
        menuItems.append(subMenu);
        systemMenu.append(engineAction);
        connect(signalMapper, SIGNAL(mapped(QString)), &manager, SLOT(setCurrentEngine(QString)));
    }

    menuItems.insert(0, systemMenu);

    // ui part
    connect(QApplication::desktop(), SIGNAL(workAreaResized(int)), this, SLOT(move()));
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(move()));
    move();

    connect(ui.listWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(itemClicked(QListWidgetItem*)));
    qimsysDebugOut();
}

void N900::Private::setupState()
{
    qimsysDebugIn();
    // state machine
    machine = new QStateMachine(this);

    {
        QState *invisibleState = new QState(machine);
        {
            invisibleState->assignProperty(q, "visible", false);
        }
        QState *visibleState = new QState(machine);
        {
            visibleState->assignProperty(q, "visible", true);
        }

        {
            QState *normalState = new QState(visibleState);
            {
                normalState->assignProperty(q, "state", Input);
            }

            QState *systemState = new QState(visibleState);
            {

                QState *menuState = new QState(systemState);
                {
                    menuState->assignProperty(q, "state", System);
                }

                systemState->setInitialState(menuState);
            }

            QBoolSignal *system = new QBoolSignal(ui.system, SIGNAL(toggled(bool)), false, this);

            normalState->addTransition(system, SIGNAL(on()), systemState);

            systemState->addTransition(system, SIGNAL(off()), normalState);

            visibleState->setInitialState(normalState);
        }

        QBoolSignal *focus = new QBoolSignal(this, SIGNAL(focusChanged(bool)), manager.focus() > 0, machine);
        QBoolSignal *composing = new QBoolSignal(&manager, SIGNAL(composingChanged(bool)), manager.composing(), machine);
        QBoolsSignal *combination = new QBoolsSignal(focus, composing, QBoolsSignal::And, machine);
        invisibleState->addTransition(combination, SIGNAL(on()), visibleState);
        visibleState->addTransition(combination, SIGNAL(off()), invisibleState);
        connect(visibleState, SIGNAL(entered()), this, SLOT(move()), Qt::QueuedConnection);
        machine->setInitialState(invisibleState);
    }

    machine->start();
    qimsysDebugOut();
}

void N900::Private::focusChanged(QWidget *previous, QWidget *current)
{
    qimsysDebugIn() << previous << current;
    if (current != 0 && !q->isAncestorOf(current)) {
        qimsysDebug() << previous << current;
        lastFocusedWidget = current;
    }
    qimsysDebugOut();
}

void N900::Private::move()
{
    qimsysDebugIn();
    QRect desktop = QApplication::desktop()->screenGeometry();
    QRect rect = preedit.rect();
    rect.setRect(rect.x() - 2, rect.y() - 2, rect.width() + 4, rect.height() + 4);
    QRect geometry(0, desktop.bottom() - q->sizeHint().height(), desktop.width(), q->sizeHint().height());
    if (geometry.intersects(rect)) {
        geometry.moveBottom(rect.top());
    }
    q->setGeometry(geometry);
    qimsysDebugOut();
}

void N900::Private::focusChanged(uint focus)
{
    qimsysDebugIn() << focus;
    emit focusChanged(focus > 0);
    qimsysDebugOut();
}

void N900::Private::inputLanguageChanged(const QString &inputLanguage)
{
    qimsysDebugIn() << inputLanguage;

    // 0 = system
    // 1 = input language
    // 2 ~ n - 1 = extensions
    // n = language
    while (menuItems.count() > 3) {
        delete menuItems[0].takeAt(1);
        QList<QAction*> subMenu = menuItems.takeAt(2);
    }

    foreach (QAction *lang, langGroup->actions()) {
        if (lang->data().toString() == inputLanguage) {
            lang->setChecked(true);
            langAction->setIcon(lang->icon());
            break;
        }
    }

    QList<QimsysConverter *> converters = QimsysPluginManager::objects<QimsysConverter>();
    foreach(QimsysConverter *converter, converters) {
        if (converter->identifier() == inputLanguage) {
            int i = 1;
            QList<QAction *>actions = converter->actions();
            foreach (QAction *action, actions) {
                menuItems[0].insert(i, action);
                QList<QAction*> subMenu;
                foreach(QAction *a, action->findChildren<QAction*>()) {
                    subMenu.append(a);
                }
                menuItems.insert(i + 1, subMenu);
                i++;
            }
            break;
        }
    }
    setMenu();
    qimsysDebugOut();
}

void N900::Private::currentEngineChanged(const QString &currentEngine)
{
    qimsysDebugIn() << currentEngine;
    foreach (QAction *action, engineGroup->actions()) {
        if (action->data().toString() == currentEngine) {
            action->setChecked(true);
            break;
        }
    }

    QList<QimsysEngine *> engines = QimsysPluginManager::objects<QimsysEngine>();
    foreach (QimsysEngine *engine, engines) {
        if (engine->identifier() == currentEngine) {
            engineAction->setIcon(engine->icon());
            engineAction->setText(engine->name());
            break;
        }
    }
    qimsysDebugOut();
}

void N900::Private::currentIconChanged(const QIcon &icon)
{
    qimsysDebugIn();
    switch (state) {
    case NotAvailable:
    case Input:
        ui.system->setIcon(icon);
        break;
    case System:
        break;
    }
    qimsysDebugOut();
}

void N900::Private::stateChanged(N900::State state)
{
    qimsysDebugIn() << state;
    switch (state) {
    case NotAvailable:
        break;
    case Input:
        ui.system->setIcon(manager.currentIcon());
        ui.listWidget->clear();
        foreach (const QimsysConversionItem &item, candidates.candidates()) {
            ui.listWidget->addItem(item.to);
        }
        break;
    case System:
        ui.system->setIcon(QIcon(":/icons/qimsys_32x32.png"));
        systemLevel = 0;
        setMenu();
        break;
    }
    qimsysDebugOut();
}

void N900::Private::setMenu()
{
    qimsysDebugIn();
    ui.listWidget->clear();
    QList<QAction *> actions = menuItems[systemLevel];
    foreach (QAction *action, actions) {
        if (systemLevel == 0) {
            qimsysDebug() << action->text() << action->icon().availableSizes();
            ui.listWidget->addItem(new QListWidgetItem(action->icon(), QString::null));
        } else {
            if (action->text() == QLatin1String("-")) continue;
            QRegExp mnemonic("\\(&[A-Za-z]\\)");
            QString text = action->text();
            text.remove(mnemonic);
            text.remove(QLatin1Char('&'));
            qimsysDebug() << action->text() << action->icon().availableSizes();
            QListWidgetItem *item = new QListWidgetItem(action->icon(), text);
            ui.listWidget->addItem(item);
            if (action->isChecked()) {
                ui.listWidget->setCurrentItem(item);
            }
        }
    }
    qimsysDebugOut();
}

void N900::Private::itemClicked(QListWidgetItem* item)
{
    qimsysDebugIn();
    switch (q->state()) {
    case Input:
        candidates.setCurrentIndex(ui.listWidget->currentRow());
        break;
    case System:
        if (systemLevel == 0) {
            int currentRow = ui.listWidget->currentRow();
            QAction *action = menuItems[systemLevel][currentRow];
            if (action->findChild<QAction*>()) {
                systemLevel = currentRow + 1;
                setMenu();
            } else {
                action->trigger();
            }
        } else {
            QAction *action = menuItems[systemLevel][ui.listWidget->currentRow()];
            action->trigger();
            systemLevel = 0;
            setMenu();
        }
        break;
    default:
        break;
    }
    qimsysDebugOut();
}

void N900::Private::candidatesChanged(const QimsysConversionItemList &candidates)
{
    qimsysDebugIn() << candidates;
    switch (q->state()) {
    case Input:
        ui.listWidget->clear();
        foreach (const QimsysConversionItem &candidate, candidates) {
            ui.listWidget->addItem(candidate.to);
        }
        ui.listWidget->setCurrentRow(this->candidates.currentIndex());
        break;
    default:
        break;
    }
    qimsysDebugOut();
}

void N900::Private::candidateIndexChanged(int index)
{
    qimsysDebugIn() << index;
    switch (q->state()) {
    case Input:
        if (ui.listWidget->count() <= index) {
            qimsysWarning() << ui.listWidget->count() << "<=" << index;
        } else {
            ui.listWidget->setCurrentRow(index);
        }
        break;
    default:
        break;
    }
    qimsysDebugOut();
}

void N900::Private::rectChanged(const QRect &rect)
{
    qimsysDebugIn() << rect;
    move();
    qimsysDebugOut();
}

N900::N900(QimsysAbstractPluginObject *plugin, QWidget *parent)
    : QWidget(parent)
{
    qimsysDebugIn();
    d = new Private(plugin, this);
    qimsysDebugOut();
}

N900::~N900()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

N900::State N900::state() const
{
    return d->state;
}

void N900::setState(N900::State state)
{
    if (d->state == state) return;
    qimsysDebugIn() << (int)d->state;
    d->state = state;
    emit stateChanged(state);
    qimsysDebugOut() << (int)state;
}

void N900::settingsUpdated()
{
//    d->settingsUpdated();
}

bool N900::event(QEvent *e)
{
    switch(e->type()) {
    case QEvent::WindowActivate:
        qimsysDebugIn();
        if (d->lastFocusedWidget)
            d->lastFocusedWidget->activateWindow();
        qimsysDebugOut();
        break;
    default:
        break;
    }
    return QWidget::event(e);
}

void N900::showEvent(QShowEvent *e)
{
    qimsysDebugIn();
    QWidget::showEvent(e);
    qimsysDebugOut();
}

void N900::hideEvent(QHideEvent *e)
{
    qimsysDebugIn();
    QWidget::hideEvent(e);
    qimsysDebugOut();
}

void N900::moveEvent(QMoveEvent *e)
{
    qimsysDebugIn();
    bool v = isVisible();
    if (v) {
        hide();
    }
    QWidget::moveEvent(e);
    if (v) {
        show();
    }
    qimsysDebugOut();
}

#include "n900.moc"
