/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "ximclient.h"
#include <QtPlugin>
#include <QWidget>

#include "qimsysdebug.h"
#include "qimsysapplicationmanager.h"
#include "qimsyskeymanager.h"
#include "qimsyspreedit.h"
#include "translator.h"

class XimClient::Private : private QObject
{
    Q_OBJECT
public:
    Private(XimClient *parent);
    ~Private();

    void setActive(bool active);

private slots:
    void sendPreeditString(const QimsysPreeditItemList &items);
    void sendCommitString(const QString &commitString, uint target);

private:
    XimClient *q;
public:
    QimsysApplicationManager manager;
    QimsysKeyManager keyManager;
    QimsysPreedit preedit;
};

XimClient::Private::Private(XimClient *parent)
    : QObject(parent)
    , q(parent)
{
    manager.init();
    keyManager.init();
    preedit.init();
    connect(&preedit, SIGNAL(itemsChanged(QimsysPreeditItemList)), this, SLOT(sendPreeditString(QimsysPreeditItemList)));
    connect(&preedit, SIGNAL(committed(QString, uint)), this, SLOT(sendCommitString(QString, uint)));
}

XimClient::Private::~Private()
{
}

void XimClient::Private::sendPreeditString(const QimsysPreeditItemList &items)
{
    if (!q->focusWindow() || (manager.focus() != q->focusWindow())) return;
    // TODO q->sendPreeditString( items )
    QString preeditString;
    int cursorPosition = 0;
    int selectionLength = 0;
    foreach(const QimsysPreeditItem &item, items) {
        if (item.cursor > 0) {
            cursorPosition = preeditString.length() + item.cursor;
        }
        if (item.selection != 0) {
            selectionLength = item.selection;
        }
        preeditString.append(item.text);
    }
    q->sendPreeditString(preeditString, cursorPosition, selectionLength);
}

void XimClient::Private::sendCommitString(const QString &commitString, uint target)
{
    if (q->focusWindow() != target) return;
    q->sendCommitString(commitString);
}

void XimClient::Private::setActive(bool active)
{
    if (active) {
        manager.setFocus(q->focusWindow());
    } else {
        if (manager.focus() == q->focusWindow()) {
            manager.setFocus(0);
        }
    }
}

XimClient::XimClient(QObject *parent)
    : QXimInputMethod(parent)
{
    d = new Private(this);
}

XimClient::~XimClient()
{
    delete d;
}

bool XimClient::filter(int unicode, int keycode, int modifiers, bool isPress, bool autoRepeat)
{
    qimsysDebugIn() << unicode << keycode << modifiers << isPress << autoRepeat;
    if (isPress)
        d->keyManager.keyPress(QChar(unicode), keycode, modifiers, autoRepeat);
    else
        d->keyManager.keyRelease(QChar(unicode), keycode, modifiers, autoRepeat);

    bool ret = d->keyManager.isAccepted();
    qimsysDebugOut() << ret;
    return ret;
}

WId XimClient::winId()
{
    static QWidget *dummy = 0;
    if (!dummy)
        dummy = new QWidget;
    return dummy->winId();
}

const char *XimClient::name() const
{
    return "qimsys";
}

const char *XimClient::locale() const
{
    return "ja_JP";
}

void XimClient::updateHandler(UpdateType type)
{
    switch (type) {
    case Update:
        break;
    case FocusIn:
        d->setActive(true);
        break;
    case FocusOut:
        d->setActive(false);
        break;
    default:
        break;
    }
}

class XimClientObject::Private : private QObject
{
    Q_OBJECT
public:
    Private(XimClientObject *parent);
    ~Private();

private slots:
    void enabledChanged(bool enabled);

private:
    XimClientObject *q;
    XimClient *client;
};

XimClientObject::Private::Private(XimClientObject *parent)
    : QObject(parent)
    , q(parent)
    , client(0)
{
    q->setGroups(QStringList() << QLatin1String("X11 Classic"));
    q->setCategoryType(Hidden);
    TR(q, "categoryName", this, QT_TR_NOOP("Input/Platform"));
    q->setIcon(QIcon(":/icons/xim.png"));
    TR(q, "name", this, QT_TR_NOOP("XIM client"));
    TR(q, "author", this, QT_TR_NOOP("Tasuku Suzuki"));
    TR(q, "translator", this, QT_TR_NOOP("None"));
    connect(q, SIGNAL(enabledChanged(bool)), this, SLOT(enabledChanged(bool)));
    enabledChanged(q->isEnabled());
}

XimClientObject::Private::~Private()
{
}

void XimClientObject::Private::enabledChanged(bool enabled)
{
    if (enabled && !client) {
        client = new XimClient(this);
    } else if (!enabled && client) {
        delete client;
        client = 0;
    }
}

XimClientObject::XimClientObject(QObject *parent)
    : QimsysAbstractPluginObject(parent)
{
    d = new Private(this);
}

XimClientObject::~XimClientObject()
{
    delete d;
}

XimClientPlugin::XimClientPlugin()
    : QimsysPlugin()
{
}

QimsysAbstractPluginObject *XimClientPlugin::createObject(QObject *parent)
{
    return new XimClientObject(parent);
}

Q_EXPORT_PLUGIN2(ximClientPlugin, XimClientPlugin)


#include "ximclient.moc"
