/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   qimsys                                                                  *
 *   Copyright (C) 2010 by Tasuku Suzuki <stasuku@gmail.com>                 *
 *                                                                           *
 *   This program is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Lesser Public License as          *
 *   published by the Free Software Foundation; either version 2 of the      *
 *   License, or (at your option) any later version.                         *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU Lesser General Public License for more details.                     *
 *                                                                           *
 *   You should have received a copy of the GNU Lesser General Public        *
 *   License along with this program; if not, write to the                   *
 *   Free Software Foundation, Inc.,                                         *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "qimsyspreedit.h"

#include "qimsysdebug.h"

class QimsysPreedit::Private : private QObject
{
    Q_OBJECT
public:
    Private(QimsysPreedit *parent);
    ~Private();

    bool init();

private:
    QimsysPreedit *q;

public:
    static QimsysPreedit *server;
    QimsysPreeditItemList items;
    QRect rect;
    QFont font;
    int cursorPosition;
    QString surroundingText;
    QString currentSelection;
    int maximumTextLength;
};

QimsysPreedit *QimsysPreedit::Private::server = 0;

QimsysPreedit::Private::Private(QimsysPreedit *parent)
    : QObject(parent)
    , q(parent)
    , items()
    , rect()
    , font()
{
    qimsysDebugIn() << parent;
    qRegisterMetaType<QimsysPreeditItem>("QimsysPreeditItem");
    qRegisterMetaTypeStreamOperators<QimsysPreeditItem>("QimsysPreeditItem");
    qRegisterMetaType< QimsysPreeditItemList >("QimsysPreeditItemList");
    qRegisterMetaTypeStreamOperators< QimsysPreeditItemList >("QimsysPreeditItemList");
#ifndef QIMSYS_NO_DBUS
    qDBusRegisterMetaType<QimsysPreeditItem>();
    qDBusRegisterMetaType< QimsysPreeditItemList >();
    qDBusRegisterMetaType<QFont>();
#endif
    switch (q->type()) {
    case Server:
        server = q;
        break;
    case Client:
        break;
    }
    qimsysDebugOut();
}

QimsysPreedit::Private::~Private()
{
    qimsysDebugIn();
    qimsysDebugOut();
}

bool QimsysPreedit::Private::init()
{
    qimsysDebugIn();
    bool ret = q->QimsysAbstractIpcObject::init();
    if (ret) {
        switch (q->type()) {
        case Server:
            break;
        case Client:
            connect(q->proxy(), SIGNAL(itemsChanged(QimsysPreeditItemList)), q, SIGNAL(itemsChanged(QimsysPreeditItemList)));
            connect(q->proxy(), SIGNAL(rectChanged(QRect)), q, SIGNAL(rectChanged(QRect)));
            connect(q->proxy(), SIGNAL(fontChanged(QFont)), q, SIGNAL(fontChanged(QFont)));
            connect(q->proxy(), SIGNAL(cursorPositionChanged(int)), q, SIGNAL(cursorPositionChanged(int)));
            connect(q->proxy(), SIGNAL(surroundingTextChanged(QString)), q, SIGNAL(surroundingTextChanged(QString)));
            connect(q->proxy(), SIGNAL(currentSelectionChanged(QString)), q, SIGNAL(currentSelectionChanged(QString)));
            connect(q->proxy(), SIGNAL(maximumTextLengthChanged(int)), q, SIGNAL(maximumTextLengthChanged(int)));
            connect(q->proxy(), SIGNAL(committed(QString, uint)), q, SIGNAL(committed(QString, uint)));
            break;
        }
    }
    qimsysDebugOut() << ret;
    return ret;
}

QimsysPreedit::QimsysPreedit(QObject *parent, Type type)
    : QimsysAbstractIpcObject(parent, type)
{
    qimsysDebugIn() << parent << type;
    d = new Private(this);
    qimsysDebugOut();
}

QimsysPreedit::~QimsysPreedit()
{
    qimsysDebugIn();
    delete d;
    qimsysDebugOut();
}

bool QimsysPreedit::init()
{
    qimsysDebugIn();
    bool ret = d->init();
    qimsysDebugOut() << ret;
    return ret;
}

QimsysAbstractIpcObject *QimsysPreedit::server() const
{
    return d->server;
}

getProp(QimsysPreedit, QimsysPreeditItemList, items)
setProp(QimsysPreedit, const QRect&, rect, setRect)

getProp(QimsysPreedit, QRect, rect)
setProp(QimsysPreedit, const QimsysPreeditItemList&, items, setItems)

getProp(QimsysPreedit, QFont, font)
setProp(QimsysPreedit, const QFont&, font, setFont)

getProp(QimsysPreedit, int, cursorPosition)
setProp(QimsysPreedit, int, cursorPosition, setCursorPosition)

getProp(QimsysPreedit, QString, surroundingText)
setProp(QimsysPreedit, const QString &, surroundingText, setSurroundingText)

getProp(QimsysPreedit, QString, currentSelection)
setProp(QimsysPreedit, const QString &, currentSelection, setCurrentSelection)

getProp(QimsysPreedit, int, maximumTextLength)
setProp(QimsysPreedit, int, maximumTextLength, setMaximumTextLength)

void QimsysPreedit::commit(uint target)
{
    qimsysDebugIn() << target;
    if (items().isEmpty()) return;
    switch (type()) {
    case Server:
        {
            QString preeditString;
            foreach(const QimsysPreeditItem &item, items()) {
                preeditString.append(item.text);
            }
            emit committed(preeditString, target);
        }
        break;
    case Client:
        if (d->server) {
            d->server->commit(target);
#ifndef QIMSYS_NO_DBUS
        } else {
            QDBusReply<void> reply = qobject_cast<QDBusAbstractInterface*>(proxy())->call(QDBus::BlockWithGui, "commit", qVariantFromValue(target));
            if (!reply.isValid()) {
                qimsysWarning() << reply.error();
            }
#endif // QIMSYS_NO_DBUS
        }
//   proxy()->metaObject()->invokeMethod( proxy(), "commit", Q_ARG( QString, preeditString() ) );
        break;
    }
    setItems(QimsysPreeditItemList());
    qimsysDebugOut();
}

#include "qimsyspreedit.moc"
