#include "youtube.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QRegExp>
#include <QSettings>
#include <QDomDocument>
#include <QDomElement>

YouTube::YouTube(QObject *parent) :
    ServicePlugin(parent)
{
    m_formatList << 37 << 22 << 35 << 34 << 18;
}

QRegExp YouTube::urlPattern() const {
    return QRegExp("(http://(www.|m.|)youtube.com/(v/|.+)(v=|list=|)|http://youtu.be/)", Qt::CaseInsensitive);
}

bool YouTube::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void YouTube::checkUrl(const QUrl &webUrl) {
    QString urlString = webUrl.toString();
    QString id(urlString.section(QRegExp("v=|list=|/"), -1).section(QRegExp("&|\\?"), 0, 0));
    QUrl url;

    if (urlString.contains("list=")) {
        // QUrl::hasQueryItem() does not work :/
        url.setUrl("https://gdata.youtube.com/feeds/api/playlists/" + id);
        url.addQueryItem("fields", "openSearch:totalResults,openSearch:startIndex,entry(content,media:group(media:title))");
        url.addQueryItem("max-results", "50");
    }
    else {
        url.setUrl("https://gdata.youtube.com/feeds/api/videos/" + id);
        url.addQueryItem("fields", "content,media:group(media:title)");
    }

    url.addQueryItem("v", "2.1");
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void YouTube::checkPlaylistVideoUrls(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void YouTube::checkUrlIsValid() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomNodeList entries = doc.elementsByTagName("entry");

    if (entries.isEmpty()) {
        emit urlChecked(false);
    }
    else {
        for (int i = 0; i < entries.count(); i++) {
            QDomNode entry = entries.at(i);
            QUrl url(entry.firstChildElement("content").attribute("src"));
            QString title = entry.firstChildElement("media:group").firstChildElement("media:title").text().trimmed();
            emit urlChecked((url.isValid()) && (!title.isEmpty()), url, this->serviceName(), title + ".mp4", i == (entries.count() - 1));
        }

        QDomElement resultsElement = doc.namedItem("feed").firstChildElement("openSearch:totalResults");
        QDomElement startElement = doc.namedItem("feed").firstChildElement("openSearch:startIndex");

        if ((!resultsElement.isNull()) && (!startElement.isNull())) {
            int totalResults = resultsElement.text().toInt();
            int startIndex = startElement.text().toInt();

            if (totalResults > (startIndex + entries.count())) {
                QString urlString = reply->request().url().toString();
                QUrl playlistUrl(urlString.section("&start-index=", 0, 0));
                playlistUrl.addQueryItem("start-index", QString::number(startIndex + entries.count()));
                this->checkPlaylistVideoUrls(playlistUrl);
            }
        }
    }

    reply->deleteLater();
}

void YouTube::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    QString id(webUrl.toString().section(QRegExp("v=|/"), -1).section(QRegExp("&|\\?"), 0, 0));
    QUrl url("https://www.youtube.com/watch");
    url.addQueryItem("v", id);
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
    url.addQueryItem("has_verified", "1");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "QDL (Qt)");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVideoPage()));
}

void YouTube::parseVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map\":")) {
        emit error(UnknownError);
    }
    else {
        response = response.section("url_encoded_fmt_stream_map\":", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&");
        int unescapes = 0;

        while ((response.contains('%')) && (unescapes < 10)) {
            response = QByteArray::fromPercentEncoding(response.toUtf8());
            unescapes++;
        }

        bool encryptedSignatures = !response.contains("sig=");

        if (encryptedSignatures) {
            response = response.replace(QRegExp("[&,\"]s="), "&signature=");
        }
        else {
            response = response.replace(QRegExp("[&,\"]sig="), "&signature=");
        }

        QStringList urlStrings = response.split("url=", QString::SkipEmptyParts);
        QStringList sigs = response.split("signature=");
        QStringList signatures;
        QMap<int, QString> formats;

        if (encryptedSignatures) {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << this->decryptSignature(sigs.at(i).section(QRegExp("(&|,)"), 0, 0));
            }
        }
        else {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << sigs.at(i).section(QRegExp("(&|,)"), 0, 0);
            }
        }

        foreach (QString urlString, urlStrings) {
            if (urlString.startsWith("http")) {
                QUrl url(urlString.section(QRegExp("[&,\"]itag="), 0, 1));
                int key = url.queryItemValue("itag").toInt();

                if (key > 0) {
                    if (!signatures.isEmpty()) {
                        url.removeQueryItem("signature");
                        url.addQueryItem("signature", signatures.takeFirst());
                    }

                    formats[key] = url.toString();
                }
            }
        }

        QString videoUrl;
        int format = QSettings("QDL", "QDL").value("YouTube/videoFormat", 18).toInt();
        int index = m_formatList.indexOf(format);

        while ((videoUrl.isEmpty()) && (index < m_formatList.size())) {
            format = m_formatList.at(index);
            videoUrl = formats.value(format, "");
            index++;

        }

        if (!videoUrl.startsWith("http")) {
            emit error(UnknownError);
        }
        else {
            QNetworkRequest request;
            request.setUrl(QUrl(videoUrl));
            emit downloadRequestReady(request);
        }
    }

    reply->deleteLater();
}

QString YouTube::decryptSignature(const QString &s) {
    QString signature;

    switch (s.size()) {
    case 88: {
        signature = s.at(48)
                + this->reverseString(s.mid(68, 14))
                + s.at(82)
                + this->reverseString(s.mid(63, 4))
                + s.at(85)
                + this->reverseString(s.mid(49, 13))
                + s.at(67)
                + this->reverseString(s.mid(13, 35))
                + s.at(3)
                + this->reverseString(s.mid(4, 8))
                + s.at(2)
                + s.at(12);
    }
        break;
    case 87: {
        signature = s.at(62)
                + this->reverseString(s.mid(63, 20))
                + s.at(83)
                + this->reverseString(s.mid(53, 9))
                + s.at(0)
                + this->reverseString(s.mid(3, 49));
    }
        break;
    case 86: {
        signature = s.mid(2, 61)
                + s.at(82)
                + s.mid(64, 18)
                + s.at(63);
    }
        break;
    case 85: {
        signature = s.at(76)
                + this->reverseString(s.mid(77, 6))
                + s.at(83)
                + this->reverseString(s.mid(61, 15))
                + s.at(0)
                + this->reverseString(s.mid(51, 9))
                + s.at(1)
                + this->reverseString(s.mid(3, 47));
    }
        break;
    case 84: {
        signature = this->reverseString(s.mid(37, 47))
                + s.at(2)
                + this->reverseString(s.mid(27, 9))
                + s.at(3)
                + this->reverseString(s.mid(4, 22))
                + s.at(26);
    }
        break;
    case 83: {
        signature = s.at(6)
                + s.mid(3, 3)
                + s.at(33)
                + s.mid(7, 17)
                + s.at(0)
                + s.mid(25, 8)
                + s.at(53)
                + s.mid(34, 19)
                + s.at(24)
                + s.mid(54);
    }
        break;
    case 82: {
        signature = s.at(36)
                + this->reverseString(s.mid(68, 13))
                + s.at(81)
                + this->reverseString(s.mid(41, 26))
                + s.at(33)
                + this->reverseString(s.mid(37, 3))
                + s.at(40)
                + s.at(35)
                + s.at(0)
                + s.at(67)
                + this->reverseString(s.mid(1, 32))
                + s.at(34);
    }
        break;
    case 81: {
        signature = s.at(6)
                + s.mid(3, 3)
                + s.at(33)
                + s.mid(7, 17)
                + s.at(0)
                + s.mid(25, 8)
                + s.at(2)
                + s.mid(34, 19)
                + s.at(24)
                + s.mid(54, 27);
    }
        break;
    default:
        break;
    }

    return signature;
}

QString YouTube::reverseString(const QString &string) {
    QString reverse;

    for (int i = string.size() - 1; i >= 0; i--) {
        reverse.append(string.at(i));
    }

    return reverse;
}

Q_EXPORT_PLUGIN2(youtube, YouTube)
