#include "fourshared.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QRegExp>
#include <QTimer>

using namespace QtJson;

FourShared::FourShared(QObject *parent) :
    ServicePlugin(parent),
    m_waitTimer(new QTimer(this)),
    m_waitTime(0),
    m_wait(0)
{
    this->connect(m_waitTimer, SIGNAL(timeout()), this, SLOT(updateWaitTime()));
}

QRegExp FourShared::urlPattern() const {
    return QRegExp("http(s|)://(www.|)4shared.com/\\w+/\\w+", Qt::CaseInsensitive);
}

bool FourShared::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void FourShared::login(const QString &username, const QString &password) {
    QString data = QString("login=%1&password=%2").arg(username).arg(password);
    QUrl url("https://www.4shared.com/login");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkLogin()));
}

void FourShared::checkLogin() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    emit loggedIn((statusCode == 302) || (statusCode == 200));

    reply->deleteLater();
}

void FourShared::checkUrl(const QUrl &webUrl) {
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void FourShared::checkUrlIsValid() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();
    QRegExp re("http://\\w+.4shared.com/download/\\w+/.+\\?tsid=[^'\"]+");

    if ((!redirect.isEmpty()) && (re.indexIn(redirect) == -1)) {
        this->checkUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());

        if (response.contains(QRegExp("file link that you requested is not valid|enter a password to access"))) {
            emit urlChecked(false);
        }
        else {
            QString fileName = response.section("<meta name=\"title\" content=\"", 1, 1).section('"', 0, 0).trimmed();

            if (fileName.isEmpty()) {
                emit urlChecked(false);
            }
            else {
                emit urlChecked(true, reply->request().url(), this->serviceName(), fileName);
            }
        }
    }

    reply->deleteLater();
}

void FourShared::getDownloadUrl(const QUrl &webUrl) {
    if (!m_downloadUrl.isEmpty()) {
        emit downloadRequestReady(QNetworkRequest(m_downloadUrl));
        m_downloadUrl.clear();
    }
    else {
        emit statusChanged(Connecting);
        QNetworkRequest request(webUrl);
        request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
        QNetworkReply *reply = this->networkAccessManager()->get(request);
        this->connect(reply, SIGNAL(finished()), this, SLOT(onWebPageDownloaded()));
    }
}

void FourShared::onWebPageDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://\\w+.4shared.com/download/\\w+/.+\\?tsid=[^'\"]+");
    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (re.indexIn(redirect) == 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else if (!redirect.isEmpty()) {
        this->getDownloadUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());

        if (re.indexIn(response) >= 0) {
            m_downloadUrl = QUrl(re.cap());
            m_wait = response.section("secondsLeft\" value=\"", 1, 1).section('"', 0, 0).toInt();
            QString fileId = response.section("fileId\" value=\"", 1, 1).section('"', 0, 0);

            if (fileId.isEmpty()) {
                emit error(UnknownError);
            }
            else {
                this->getDownloadLimitInfo(fileId);
            }
        }
        else {
            QString urlString = reply->request().url().toString();

            if (!urlString.contains("4shared.com/get/")) {
                this->getDownloadUrl(QUrl(urlString.replace(QRegExp("4shared.com/\\w+/"), "4shared.com/get/")));
            }
            else {
                emit error(UnknownError);
            }
        }
    }

    reply->deleteLater();
}

void FourShared::getDownloadLimitInfo(const QString &fileId) {
    QUrl url("http://www.4shared.com/web/d2/getFreeDownloadLimitInfo");
    url.addQueryItem("fileId", fileId);
    QNetworkRequest request(url);
    request.setRawHeader("Accept", "application/json, text/javascript, */*; q=0.01");
    request.setRawHeader("X-Requested-With", "XMLHttpRequest");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDownloadLimitInfo()));
}

void FourShared::checkDownloadLimitInfo() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());
    QVariantMap map = Json::parse(response).toMap();

    if (map.value("status") == "ok") {
        this->startWait(m_wait * 1000);

        if (m_wait > 30) {
            this->connect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
        }
        else {
            this->connect(this, SIGNAL(waitFinished()), this, SLOT(getDownloadLink()));
        }
    }
    else {
        QVariantMap traffic = map.value("traffic").toMap();

        if (traffic.value("exceeded") != "none") {
            emit error(TrafficExceeded);
        }
        else {
            emit error(UnknownError);
        }
    }

    reply->deleteLater();
}

void FourShared::startWait(int msecs) {
    if (msecs > 30000) {
        emit statusChanged(LongWait);
    }
    else {
        emit statusChanged(ShortWait);
    }

    emit waiting(msecs);
    m_waitTime = msecs;
    m_waitTimer->start(1000);
}

void FourShared::updateWaitTime() {
    m_waitTime -= m_waitTimer->interval();
    emit waiting(m_waitTime);

    if (m_waitTime <= 0) {
        m_waitTimer->stop();
        emit waitFinished();
    }
}

void FourShared::onWaitFinished() {
    emit statusChanged(Ready);
    this->disconnect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
}

void FourShared::getDownloadLink() {
    QNetworkRequest request(m_downloadUrl);
    QNetworkReply *reply = this->networkAccessManager()->head(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDownloadLink()));
    this->disconnect(this, SIGNAL(waitFinished()), this, SLOT(getDownloadLink()));
}

void FourShared::checkDownloadLink() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.hasQueryItem("cau2")) {
        emit error(NotFound);
    }
    else {
        emit downloadRequestReady(QNetworkRequest(m_downloadUrl));
    }

    reply->deleteLater();
}

Q_EXPORT_PLUGIN2(fourshared, FourShared)
