#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import ecore

__all__ = ("KineticMouse",)

class KineticMouse(object):
    """Moves things based on mouse events.

    The object should be feed with calls to mouse_down(), mouse_up() and
    mouse_move(), it will call move_offset() in reaction.
    """
    accel_constant = 0.7

    def __init__(self, move_offset):
        """Kinetics that call move_offset with the amount to move.

        Signature: C{function(offset) -> bool}

        C{move_offset()} will be called with the required offset and should
        return True when it was successful or False when it was not possible,
        in this case any animation will be stopped.
        """
        self.time_dif_constant = ecore.animator_frametime_get()
        self.last_value = self.value = None
        self.last_time = self.time = None
        self.animation = None
        self.move_offset = move_offset

    def mouse_down(self, value):
        """Feed mouse down event at given position (value).

        @return: False if it stopped an animation or True otherwise.
        """
        r = True
        if self.animation is not None:
            self.animation.stop()
            self.animation = None
            r = False

        self.last_value = self.value = value
        self.last_time = self.time = ecore.time_get()
        return r

    def mouse_up(self, value):
        "Feed mouse up event at given position (value)."
        if self.last_value is None or self.animation:
            return

        t = ecore.time_get()
        dt = t - self.last_time
        dv = value - self.last_value

        self._kinetic_start(dv / dt)
        self.mouse_cancel()

    def mouse_move(self, value):
        "Feed mouse move event at given position (value)."
        if self.last_value is None:
            return

        dv = value - self.value
        old_dv = self.value - self.last_value

        t = ecore.time_get()
        dt = t - self.time
        if dt < self.time_dif_constant and dv * old_dv > 0:
            return

        self.last_value = self.value
        self.last_time = self.time
        self.value = value
        self.time  = t
        self.move_offset(int(dv))

    def mouse_cancel(self):
        "Request kinetic to cancel analysis and stop move on drag."
        self.last_value = self.value = None
        self.last_time = self.time = None

    def kinetic_stop(self):
        "Stop running kinetic animation."
        if self.animation:
            self.animation.stop()
        self.animation = None
        self.anim_speed = None
        self.anim_t0 = None

    def _kinetic_start(self, speed):
        self.anim_speed = speed
        self.anim_t0 = ecore.time_get()
        self.anim_accel = -self.anim_speed * self.accel_constant
        self.animation = ecore.animator_add(self._animator)

    def _animator(self):
        t_now = ecore.time_get()
        dt = t_now - self.anim_t0
        speed = self.anim_speed + self.anim_accel * dt
        value = self.anim_speed * dt + self.anim_accel * dt * dt / 2
        if self.anim_accel * speed > 0:
            # just to notify animation finish
            self.move_offset(0)
            self.kinetic_stop()
            return False
        else:
            r = self.move_offset(int(value))
            self.anim_speed = speed
            self.anim_t0 = t_now
            if not r:
                self.animation = None
            return r
