#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import threading
import logging
import traceback
from dispatcher import send_to_thread_sync, send_to_thread_async

from model import ModelFolder
from threads import ThreadsManager


log = logging.getLogger("terra.core.threaded_model")


## NOTE: The following functions to return decorators are highly dependent on
##       ThreadedModelFolder code so be careful when changing things here!!

def send_to_terra_thread_sync():
    return send_to_thread_sync("_terra_thr_name", "_terra_disp")

def send_to_terra_thread_async():
    return send_to_thread_async("_terra_thr_name", "_terra_disp")


class ThreadedModelFolder(ModelFolder):
    def __init__(self, name, parent=None, children=None):
        ModelFolder.__init__(self, name, parent, children)

        self._die = threading.Event()
        self._thr_mger = ThreadsManager()
        self._thr = None
        self._terra_thr_name = self._thr_mger.thr_name
        self._terra_disp = self._thr_mger.disp
        self.is_daemon = False

    def _check_die(self):
        return self._die.isSet()

    must_die = property(_check_die)

    ##
    ## Methods executed in Terra's thread
    ##
    def _terminate(self):
        self._die.set()
        self._thr.join(0.3)
        if self._thr.isAlive():
            # FIXME: do something here?
            log.debug_warning("thread %r of threaded folder %r didn't "
                              "terminate!", self._thr.getName(), self.name)

    @send_to_terra_thread_async()
    def _finished(self):
        self._terminate()
        if not self.is_loaded:
            self.inform_loaded()

    @send_to_terra_thread_async()
    def _cleanup(self):
        self._terminate()
        del self.children[:]
        self.children.callback_changed = None
        if not self.is_loaded:
            self.inform_loaded()

    @send_to_terra_thread_async()
    def append(self, item):
        ModelFolder.append(self, item)

    @send_to_terra_thread_async()
    def extend(self, items):
        ModelFolder.extend(self, items)

    @send_to_terra_thread_async()
    def insert(self, index, item):
        ModelFolder.insert(self, index, item)

    @send_to_terra_thread_async()
    def remove(self, item):
        ModelFolder.remove(self, item)

    @send_to_terra_thread_async()
    def inform_loaded(self):
        ModelFolder.inform_loaded(self)

    def load(self):
        if self._load_count < 0:
            log.debug_error("trying to load model folder %r with "
                            "weird load count: %d", self, self._load_count)
            traceback.print_stack()
            self._load_count = 0
            return

        self._load_count += 1
        if self._load_count > 1: # self.is_loaded
            return

        log.debug("loading %r ...", self)

        # if needed, try to collect last thread
        if self._thr is not None and \
           self._thr.isAlive():
            self._thr.join(0.1)

        self.children.callback_changed = self.notify_model_changed

        self.is_loading = True
        self._die.clear()

        # start plugin's thread to load items
        self._thr = self._thr_mger.next_worker_thread(self._run)
        self._thr.setDaemon(self.is_daemon)
        self._thr.start()

    def _do_unload(self):
        log.debug("unloading %r ...", self)

        self.children.callback_changed = None
        self._terminate()
        self.is_loading = False
        self.do_unload()

    def unload(self):
        if self._load_count <= 0: # not self.is_loaded
            log.debug_error("trying to unload model folder %r with "
                            "weird load count: %d", self, self._load_count)
            traceback.print_stack()
            self._load_count = 0
            return

        self._load_count -= 1
        if self._load_count > 0:
            return

        if self._hold_count > 0:
            log.debug_warning("still held, not unloading %r (load count %d, "
                              "hold count %d) ...", self, self._load_count,
                              self._hold_count)
            return

        self._do_unload()

    ##
    ## Methods executed in plugin's thread
    ##
    def do_load(self):
        raise NotImplementedError("must be implemented by subclasses")

    def _run(self):
        try:
            self.do_load()
        except:
            log.debug_error("error when loading threaded folder %r from "
                            "thread %r, finishing it ...", self.name,
                            self._thr.getName(), exc_info=True)
            self._cleanup()
        else:
            if not self.must_die:
                self._finished()
