#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
import cPickle

from manager import Manager
from cli_utils import process_cmdline


class PluginPrefs(dict):
    """Plugin preferences.

    This reads and writes a dictionary with all preferences for a
    specific plugin.
    """
    def __init__(self, plugin_name):
        dict.__init__(self)
        cfg = Manager().terra_config
        self.prefs_dir = os.path.join(os.path.expanduser("~"),
                                      cfg.app_user_dir, "prefs")
        self.prefs_filename = os.path.join(self.prefs_dir, plugin_name)
        self._load()

    def _load(self):
        if not os.path.exists(self.prefs_dir):
            os.makedirs(self.prefs_dir, 0700)

        if not os.path.exists(self.prefs_filename):
            return

        prefs = cPickle.load(open(self.prefs_filename, "rb"))
        if type(prefs) is not dict:
            raise TypeError("preference file %r should have a dict" %
                            self.prefs_filename)

        self.update(prefs)

    def reload(self):
        self._load()

    def save(self):
        prefs = {}
        prefs.update(self)
        cPickle.dump(prefs, open(self.prefs_filename, "wb"),
                     cPickle.HIGHEST_PROTOCOL)


def _process_cmdline():
    parser, opts, args = process_cmdline()
    mger = Manager(opts.config_file)

    return mger, args, parser


def get_prefs():
    from pprint import pprint

    mger, args, parser = _process_cmdline()
    if len(args) != 1:
        parser.error("expected plugin name but received %r" % args)

    prefs = PluginPrefs(args[0])
    pprint(prefs)


def set_prefs():
    from pprint import pprint

    mger, args, parser = _process_cmdline()
    if len(args) != 3:
        parser.error("expected plugin name, key and value "
                     "but received %r" % args)

    prefs = PluginPrefs(args[0])
    prefs[args[1]] = eval(args[2]) # FIXME
    prefs.save()
    pprint(prefs)
