/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#ifndef __TYPECONVERTER_H__
#define __TYPECONVERTER_H__

#include <QtCore/QString>
#include <QtCore/QPair>

#include <boost_headers.hpp>
#include <type_manager.hpp>
#include <pyside_global.hpp>
#include <parent_policy.hpp>

namespace PySide
{

PYSIDE_LOCAL void register_qstring_converter();
PYSIDE_LOCAL void register_qbool_converter();
PYSIDE_LOCAL void register_qbytearray_converter();

/************************* QPair<X, Y> *************************/

template<typename X, typename Y>
struct PYSIDE_LOCAL QPair2Py
{
    static PyObject*
    convert(const QPair<X, Y>& pair)
    {
        return boost::python::incref(boost::python::make_tuple(pair.first, pair.second).ptr());
    }
};

template<typename X, typename Y>
struct PYSIDE_LOCAL Py2QPair
{
    static void*
    convertible(PyObject* py_obj)
    {
        if (!PyTuple_Check(py_obj) || PyTuple_Size(py_obj) != 2)
            return 0;
        return py_obj;
    }

    static void
    construct(PyObject* py_obj,
              boost::python::converter::rvalue_from_python_stage1_data* data)
    {
        typedef boost::python::converter::rvalue_from_python_storage<QPair<X, Y> > storage_t;
        storage_t* the_storage = reinterpret_cast<storage_t*>( data );

        X x = boost::python::extract<X>(PyTuple_GetItem(py_obj, 0));
        Y y = boost::python::extract<Y>(PyTuple_GetItem(py_obj, 1));

        void* memory_chunk = the_storage->storage.bytes;
        new (memory_chunk) QPair<X, Y>(x, y);

        data->convertible = memory_chunk;
    }
};

template<typename T> PYSIDE_LOCAL
void
register_qpair_converter(const char* type_name) {
    typedef typename T::first_type X;
    typedef typename T::second_type Y;

    if (!type_manager::instance().register_converter(type_name))
        return;
    boost::python::to_python_converter<QPair<X, Y>, QPair2Py<X, Y> >();
    boost::python::converter::registry::push_back(Py2QPair<X, Y>::convertible,
                                                  Py2QPair<X, Y>::construct,
                                                  boost::python::type_id<QPair<X, Y> >());
}

/************************* QList<T>/QVector<T> ***************************/

template<typename Container, typename T = typename Container::value_type>
struct PYSIDE_LOCAL Container2Py
{
    static PyObject*
    convert(const Container& lst)
    {
        boost::python::list result;
        foreach (T t, lst) {
            result.append(t);
        }
        return boost::python::incref(result.ptr());
    }
};

// specialization to Container<T*>, etc
template<typename Container, typename T>
struct PYSIDE_LOCAL Container2Py<Container, T*>
{
    static PyObject*
    convert(const Container& lst)
    {
        PyObject* result = PyList_New(0);
        foreach (T* t, lst)
        {
            PyList_Append(result,
                          boost::python::incref(boost::python::object(PySide::ptr(t)).ptr()));
        }
        Py_INCREF(result);
        return result;
    }
};

// specialization to Container<const char*>, etc
template<typename Container>
struct PYSIDE_LOCAL Container2Py<Container, const char*>
{
    static PyObject*
    convert(const Container& lst)
    {
        PyObject* result = PyList_New(0);
        foreach (const char* t, lst)
        {
            PyList_Append(result,
                          boost::python::incref(boost::python::object(std::string(t)).ptr()));
        }
        Py_INCREF(result);
        return result;
    }
};


template<typename Container, typename T = typename Container::value_type>
struct PYSIDE_LOCAL Py2Container
{
    static void*
    convertible(PyObject* py_obj)
    {
        if (!PyList_Check(py_obj))
            return 0;

        // check if we have a list full of T's
        const Py_ssize_t n = PyList_Size(py_obj);
        for (int i = 0; i < n; ++i) {
            if (!boost::python::extract<T>(PyList_GetItem(py_obj, i)).check())
                return 0;
        }

        return py_obj;
    }

    static void
    construct(PyObject* py_obj, boost::python::converter::rvalue_from_python_stage1_data* data)
    {
        typedef boost::python::converter::rvalue_from_python_storage<Container > storage_t;

        storage_t* the_storage = reinterpret_cast<storage_t*>( data );
        void* memory_chunk = the_storage->storage.bytes;
        Container* container = new (memory_chunk) Container();

        const Py_ssize_t n = PyList_Size(py_obj);
        for (int i = 0; i < n; ++i) {
            container->append(boost::python::extract<T>(PyList_GetItem(py_obj, i)));
        }
        data->convertible = memory_chunk;
    }
};

template<typename Container> PYSIDE_LOCAL
void
register_container_converter(const char* type_name) {
    type_manager& tm = type_manager::instance();
    if (!tm.register_converter(type_name))
        return;
    boost::python::to_python_converter<Container, Container2Py<Container> >();
    boost::python::converter::registry::push_back (&Py2Container<Container>::convertible,
                                                   &Py2Container<Container>::construct,
                                                   boost::python::type_id<Container >());
    tm.register_container_type<Container>(type_name);
}

/************************* QHash<T>/QMap<T> ****************************/

template<typename Dict>
struct PYSIDE_LOCAL Dict2Py
{
    static PyObject*
    convert(const Dict& d)
    {
        // TODO: do as in List2Py::convert
        boost::python::dict result;
        typename Dict::const_iterator it = d.begin();
        for (; it != d.end(); ++it)
            result[it.key()] = it.value();
        return boost::python::incref(result.ptr());
    }
};

template<typename Dict,
         typename Key = typename Dict::key_type,
         typename Value = typename Dict::mapped_type>
struct PYSIDE_LOCAL Py2Dict
{
    static void*
    convertible(PyObject* py_obj)
    {
        using namespace boost::python;

        if (!PyDict_Check(py_obj))
            return 0;

        PyObject* key;
        PyObject* value;
        Py_ssize_t pos = 0;

        while (PyDict_Next(py_obj, &pos, &key, &value)) {
            if (!extract<Key>(key).check() || !extract<Value>(value).check())
                return 0;
        }
        return py_obj;
    }

    static void
    construct(PyObject* py_obj, boost::python::converter::rvalue_from_python_stage1_data* data)
    {
        using namespace boost::python;
        typedef boost::python::converter::rvalue_from_python_storage<Dict > storage_t;

        storage_t* the_storage = reinterpret_cast<storage_t*>( data );
        void* memory_chunk = the_storage->storage.bytes;
        Dict* dict = new (memory_chunk) Dict();

        PyObject* key;
        PyObject* value;
        Py_ssize_t pos = 0;

        while (PyDict_Next(py_obj, &pos, &key, &value)) {
            (*dict)[extract<Key>(key)] = extract<Value>(value);
        }
        data->convertible = memory_chunk;
    }
};

template<typename Dict> PYSIDE_LOCAL
void
register_dict_converter(const char* type_name) {
    if (!type_manager::instance().register_converter(type_name))
        return;
    boost::python::to_python_converter<Dict, Dict2Py<Dict> >();
    boost::python::converter::registry::push_back(&Py2Dict<Dict>::convertible,
                                                  &Py2Dict<Dict>::construct,
                                                  boost::python::type_id<Dict>());
}



/************************* QMultiMap<T> ****************************/

template<typename MultiMap>
struct PYSIDE_LOCAL MultiMap2Py
{
    static PyObject*
    convert(const MultiMap& d)
    {
        boost::python::dict result;
        typename MultiMap::const_iterator it = d.begin();
        for (; it != d.end(); ++it) {
            QList<typename MultiMap::mapped_type> values = d.values(it.key());
            boost::python::list py_values;

            for (typename MultiMap::size_type i = 0, max = values.size(); i < max; i++) {
                qDebug() << "Add item: " << it.key() << "/" << values.at(i);
                py_values.append(values.at(i));
            }

            result[it.key()] = py_values;
        }
        return boost::python::incref(result.ptr());
    }
};

template<typename MultiMap,
         typename Key = typename MultiMap::key_type,
         typename Value = typename MultiMap::mapped_type>
struct PYSIDE_LOCAL Py2MultiMap
{
    static void*
    convertible(PyObject* py_obj)
    {
        using namespace boost::python;

        qDebug() << "WILL CHECK IF IS CONVERTIBLE" << __PRETTY_FUNCTION__;

        if (!PyDict_Check(py_obj))
            return 0;

        PyObject* key;
        PyObject* value;
        Py_ssize_t pos = 0;

        while (PyDict_Next(py_obj, &pos, &key, &value)) {
            if (!extract<Key>(key).check() || !PyList_Check(value))
                return 0;

            for (Py_ssize_t i=0; i < PyList_Size(value); i++) {
                PyObject *item = PyList_GetItem(value, i);
                if (!extract<Value>(item).check())
                    return 0;
            }
        }
        return py_obj;
    }

    static void
    construct(PyObject* py_obj,
              boost::python::converter::rvalue_from_python_stage1_data* data)
    {
        using namespace boost::python;
        typedef boost::python::converter::rvalue_from_python_storage< MultiMap > storage_t;

        qDebug() << "WILL CONSTRUCT A MULTIMAP" << __PRETTY_FUNCTION__;
        storage_t* the_storage = reinterpret_cast<storage_t*>( data );
        void* memory_chunk = the_storage->storage.bytes;
        MultiMap* dict = new (memory_chunk) MultiMap();

        PyObject* key;
        PyObject* value;
        Py_ssize_t pos = 0;

        while (PyDict_Next(py_obj, &pos, &key, &value)) {
            Key current_key = extract<Key>(key);
            for (Py_ssize_t i=0, max = PyList_Size(value); i < max; i++) {
                (*dict).insert(current_key, extract<Value>(PyList_GetItem(value, i)));
            }
        }
        data->convertible = memory_chunk;
    }
};

template<typename MultiMap> PYSIDE_LOCAL
void
register_multimap_converter(const char* type_name)
{
    if (!type_manager::instance().register_converter(type_name))
        return;
    boost::python::to_python_converter<MultiMap, MultiMap2Py<MultiMap> >();
    boost::python::converter::registry::push_back(&Py2MultiMap<MultiMap>::convertible,
                                                  &Py2MultiMap<MultiMap>::construct,
                                                  boost::python::type_id<MultiMap>());
}

} // namespace PySide

#endif

