/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __PARENT_POLICY_HPP__
#define __PARENT_POLICY_HPP__

#include <boost_headers.hpp>
#include <cstddef>
#include <string>

#include "qptr.hpp"

namespace PySide
{

/**
* Call Police used to associate returned python object with c++ pointer
*
* @T is the c++ object type
* @BasePolicy_ can be used to concatenate new policies
**/
template <
    class T
    , class BasePolicy_ = boost::python::default_call_policies>
struct PYSIDE_LOCAL register_wrapper_object : BasePolicy_
{
    template <class ArgumentPackage>
    static PyObject*
    postcall(ArgumentPackage const& args_, PyObject* result)
    {
        result = BasePolicy_::postcall(args_, result);
        if (result) {
            //only register python object in qptr_list
            PyObject* py_self =
                    boost::python::detail::get_prev<1>::execute(args_, result);
            qptr<T> ptr_parent(py_self);
        }
        return result;
    }
};

/**
 * Call Police used to associate child and parent objects
 * This function create a life link between parent and child objects
 *
 * @parent is the arg index of parent object
 * @child is the arg index of child object
 * @T_PARENT is the c++ parent type
 * @T_CHILD is the c++ child type
 * @BasePolicy_ can be used to concatenate new policies
 **/
template <
    std::size_t parent
  , std::size_t child
  , class T_PARENT
  , class T_CHILD
  , class BasePolicy_ = boost::python::default_call_policies
  , class T = void>
struct PYSIDE_LOCAL parent_policy_add : BasePolicy_
{
    BOOST_STATIC_ASSERT(parent != child);

    template <class ArgumentPackage>
    static PyObject*
    postcall(ArgumentPackage const& args_, PyObject* result)
    {
        unsigned arity_ = boost::python::detail::arity(args_);
        if ((std::max)(parent, child) > arity_) {
            //PyErr_SetString(PyExc_IndexError,
            //                "PySide::parent_policy_add: argument index out of range");
            return BasePolicy_::postcall(args_, result);
        }

        result = BasePolicy_::postcall(args_, result);
        if (result) {
            PyObject* py_parent = Py_None;
            if (parent <= arity_)
                py_parent = boost::python::detail::get_prev<parent>::execute(args_, result);

            PyObject* py_child = boost::python::detail::get_prev<child>::execute(args_, result);
            if (py_child == Py_None) {
                return result;
            } else if (py_parent == Py_None) {
                qptr<T_CHILD> ptr_child(py_child);
                ptr_child.remove_parent();
            } else {
                qptr<T_PARENT> ptr_parent(py_parent);
                qptr<T_CHILD> ptr_child(py_child);

                if (ptr_parent.is_wrapper()) {
                    ptr_parent.add_child(ptr_child);
                } else {
                    ptr_child.add_cpp_ref();
                }
            }
        }
        return result;
    }

};

template <  std::size_t parent
            , std::size_t child
            , class T_PARENT
            , class BasePolicy_
            , class T>
struct PYSIDE_LOCAL parent_policy_add<parent, child, T_PARENT, QList<T*>, BasePolicy_ > : BasePolicy_
{
    template <class ArgumentPackage>
    static PyObject*
    postcall(ArgumentPackage const& args_, PyObject* result)
    {
        unsigned arity_ = boost::python::detail::arity(args_);
        if ((std::max)(parent, child) > arity_) {
            return BasePolicy_::postcall(args_, result);
        }

        result = BasePolicy_::postcall(args_, result);
        if (result) {
            PyObject* py_parent = Py_None;
            if (parent <= arity_)
                py_parent = boost::python::detail::get_prev<parent>::execute(args_, result);

            PyObject* py_child = boost::python::detail::get_prev<child>::execute(args_, result);
            if (py_child == Py_None || !PyList_Check(py_child)) {
                return result;
            } else if (py_parent == Py_None) {
                Py_ssize_t max = PyList_Size(py_child);
                for (Py_ssize_t i = 0; i < max; ++i) {
                    qptr<T> ptr_child(PyList_GetItem(py_child, i));
                    ptr_child.remove_parent();
                }
            } else {
                Py_ssize_t max = PyList_Size(py_child);
                qptr<T_PARENT> ptr_parent(py_parent);
                for (Py_ssize_t i = 0; i < max; ++i) {
                    qptr<T> ptr_child(PyList_GetItem(py_child, i));

                    if (ptr_parent.is_wrapper()) {
                       ptr_parent.add_child(ptr_child);
                    } else {
                        ptr_child.add_cpp_ref();
                    }
                }
            }
        }
        return result;
    }
};

/**
 * Call Police used to take or relase ownership from a PyObject
 *
 * @arg_index is the arg index of parent object
 * @release_ownership is a boolean value to specify if PyObject will release or not the c++ pointer ownership
 * @T is the c++ arg type
 * @BasePolicy_ can be used to concatenate new policies
 **/
template <
    std::size_t arg_index
  , bool release_ownership
  , class T
  , class BasePolicy_ = boost::python::default_call_policies>
struct PYSIDE_LOCAL transfer_ownership : BasePolicy_
{
    template <class ArgumentPackage>
    static PyObject*
    postcall(ArgumentPackage const& args_, PyObject* result)
    {
        unsigned arity_ = boost::python::detail::arity(args_);
        if (arg_index > arity_) {
            PyErr_SetString(PyExc_IndexError,
                            "PySide::transfer_ownership: argument index out of range");
            return 0;
        }

        result = BasePolicy_::postcall(args_, result);
        if (result) {
            PyObject* py_arg = boost::python::detail::get_prev<arg_index>::execute(args_, result);

            qptr<T> ptr_child(py_arg);
            if (release_ownership)
                ptr_child.release_ownership();
            else
                ptr_child.acquire_ownership();
        }

        return result;
    }
};


/**
 * Call Police used to remove association between child and parent objects
 * This function remove the link create in previous call policy
 *
 * @parent is the arg index of parent object
 * @child is the arg index of child object
 * @T_PARENT is the c++ parent type
 * @T_CHILD is the c++ child type
 * @BasePolicy_ can be used to concatenate new policies
 **/
template <
    std::size_t parent
  , std::size_t child
  , class T_PARENT
  , class T_CHILD
  , class BasePolicy_ = boost::python::default_call_policies
  , class T = void>
struct PYSIDE_LOCAL parent_policy_remove : BasePolicy_
{
    BOOST_STATIC_ASSERT(parent != child);

    template <class ArgumentPackage>
    static PyObject*
    postcall(ArgumentPackage const& args_, PyObject* result)
    {
        unsigned arity_ = boost::python::detail::arity(args_);
        if ((std::max)(parent, child) > arity_) {
            PyErr_SetString(PyExc_IndexError,
                            "PyQt::parent_policy_remove: argument index out of range");
            return 0;
        }

        result = BasePolicy_::postcall(args_, result);
        if (result) {
            PyObject* py_parent = 0;
            if (parent <= arity_)
                py_parent = boost::python::detail::get_prev<parent>::execute(args_, result);

            PyObject* py_child = boost::python::detail::get_prev<child>::execute(args_, result);

            qptr<T_PARENT> ptr_parent(py_parent);
            qptr<T_CHILD> ptr_child(py_child);

            if (ptr_parent.is_wrapper()) {
                ptr_child.remove_parent();
            } else {
                ptr_child.remove_cpp_ref();
            }
        }

        return result;
    }
};

template <  std::size_t parent
            , std::size_t child
            , class T_PARENT
            , class BasePolicy_
            , class T>
struct PYSIDE_LOCAL parent_policy_remove<parent, child, T_PARENT, QList<T*>, BasePolicy_> : BasePolicy_
{
    BOOST_STATIC_ASSERT(parent != child);

    template <class ArgumentPackage>
    static PyObject*
    postcall(ArgumentPackage const& args_, PyObject* result)
    {
        unsigned arity_ = boost::python::detail::arity(args_);
        if ((std::max)(parent, child) > arity_) {
            PyErr_SetString(PyExc_IndexError,
                            "PyQt::parent_policy_remove: argument index out of range");
                            return 0;
        }

        result = BasePolicy_::postcall(args_, result);
        if (result) {
            PyObject* py_parent = 0;
            if (parent <= arity_)
                py_parent = boost::python::detail::get_prev<parent>::execute(args_, result);

            PyObject* py_child = boost::python::detail::get_prev<child>::execute(args_, result);
            if (PyList_Check(py_child)) {
                qptr<T_PARENT> ptr_parent(py_parent);
                uint max = PyList_Size(py_child);
                for (Py_ssize_t i = 0; i < max; ++i) {
                    qptr<T> ptr_child(PyList_GetItem(py_child, i));

                    if (ptr_parent.is_wrapper()) {
                        ptr_child.remove_parent();
                    } else {
                        ptr_child.remove_cpp_ref();
                    }
                }
            }
        }
        return result;
    }
};

/**
 * Used to create a PyObject from a C++ Pointer
 * In this struct a PyObject will be created if this not registered otherwise will
 * create one and register this for future access and avoid return different
 * object for the same c++ pointer
 **/
template<bool cpp_ownership>
struct PYSIDE_LOCAL make_ptr_reference_holder
{
    template <class T>
    static PyObject*
    execute(T* p)
    {
        if(p == 0)
        {
            return boost::python::incref(Py_None);
        }

        //Verify if already exists python object
        qptr<T> ptr(const_cast<T*>(p));
        PyObject *ret = ptr.get_pyobject();
        if (ret)
            return boost::python::incref(ret);

        ptr.set_pyobject(execute_impl(p));

        //only the first object set the ownership
        if (cpp_ownership) {
            ptr.release_ownership();
        }

        return ptr.get_pyobject();
    }

    template <class T>
    static PyObject*
    execute_impl(T* p)
    {
        typedef qptr<T> smart_pointer;
        typedef boost::python::objects::pointer_holder<smart_pointer, T> holder_t;

        smart_pointer ptr(const_cast<T*>(p));
        PyObject *ret = boost::python::objects::make_ptr_instance<T, holder_t>::execute(ptr);
        ptr.set_pyobject(ret);
        return ret;
    }
};


/**
 * This return policy is used in default return policy
 * In this policy the c++ object will be maped to a PyObject using the previous
 * routine "make_ptr_reference_holder"
 *
 * @cpp_ownership is used to say if the C++ is responsable to delete the poiter
 * or not
 **/
template <bool cpp_ownership = false>
struct PYSIDE_LOCAL return_ptr_object
{
    template <class T>
    struct apply
    {
        typedef typename boost::mpl::if_c<
            boost::is_pointer<T>::value
            , boost::python::to_python_indirect<T, make_ptr_reference_holder<cpp_ownership> >
            , boost::python::detail::reference_existing_object_requires_a_pointer_or_reference_return_type<T>
        >::type type;
    };
};

struct PYSIDE_LOCAL reference_ptr_object
{
    template <class T>
    struct apply
    {
        BOOST_STATIC_CONSTANT(
            bool, ok = boost::is_pointer<T>::value || boost::is_reference<T>::value);

        typedef typename boost::mpl::if_c<
            ok
            , boost::python::to_python_indirect<T, make_ptr_reference_holder<false> >
            , boost::python::detail::reference_existing_object_requires_a_pointer_or_reference_return_type<T>
        >::type type;
    };
};


/**
 * Used when the object returned by the function is child of any function arg
 *
 * @parent_arg is the  index of parent argument
 * @ChildType is the C++ type of child object used to extract child from Python Object
 * @BasePolicy_ used to concatenate new policies
 **/
template < std::size_t parent_arg
          , std::size_t child_arg
          , class T_PARENT
          , class T_CHILD
          , class BasePolicy_ = boost::python::default_call_policies
          , class T = void >
struct PYSIDE_LOCAL return_object
    : parent_policy_add<parent_arg, child_arg, T_PARENT, T_CHILD, BasePolicy_, T>
{
 private:
    BOOST_STATIC_CONSTANT(bool, legal = parent_arg > 0);
 public:
    typedef typename boost::mpl::if_c<
        legal
        , reference_ptr_object
        , boost::python::detail::return_internal_reference_owner_arg_must_be_greater_than_zero<parent_arg>
    >::type result_converter;
};

template < std::size_t parent_arg
         , std::size_t child_arg
         , class T_PARENT
         , class BasePolicy_
         , class T >
struct PYSIDE_LOCAL return_object<parent_arg, child_arg, T_PARENT, QList<T>, BasePolicy_>
: parent_policy_add<parent_arg, child_arg, T_PARENT, QList<T>, BasePolicy_, T>
{
};

/**
 *
 **/
template<class T>
inline boost::python::handle<>
ptr(T* data, bool take_ownership = false)
{
    if (data == 0)
        return boost::python::handle<>(boost::python::incref(Py_None));

    PyObject *ret;

    if (take_ownership) {
        boost::python::to_python_indirect<T, make_ptr_reference_holder<false> >
                convert;
        ret = convert(data);
    } else {
        boost::python::to_python_indirect<T, make_ptr_reference_holder<true> >
                convert;
        ret = convert(data);
    }
    return boost::python::handle<>(ret);
}

} //namespace PySide

#endif

