/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <iostream>
#include <typeinfo>

#include <QtCore/QDebug>

#include "type_manager.hpp"
#include "type_converter.hpp"
#include "type_details.hpp"

using namespace boost;

namespace PySide
{

type_manager::type_manager()
{
    //Register native types
    register_native_type<int>("int");
    register_native_type<double>("double");
    register_native_type<float>("float");
    register_native_type<qreal>("qreal");
    register_native_type<qint64>("qint64");
    register_native_type<bool>("bool");

    //Register converter types
    register_qstring_converter();
    register_qbool_converter();
    register_qbytearray_converter();
}

type_manager&
type_manager::instance()
{
    static type_manager instance;
    return instance;
}

type_manager::~type_manager()
{
    foreach(type_details *det, m_type_map.values())
        delete det;

    m_type_map.clear();
}

python::object
type_manager::to_python(const QString &name, void *data)
{
    if (m_type_map.contains(name)) {
        return m_type_map[name]->to_python(data);
    } else {
        PyErr_Format(PyExc_TypeError, "Can't convert %s to python! Ahh!!", name.toLocal8Bit().data());
        throw python::error_already_set();
    }
}

QGenericArgument
type_manager::to_cpp(const QString &name, const python::object &obj)
{
    if (m_type_map.contains(name)) {
        return m_type_map[name]->to_cpp(obj);
    }
    PyErr_BadArgument();
    throw python::error_already_set();
}

void
type_manager::delete_data(QGenericArgument &arg)
{
    QString name(arg.name());
    if (m_type_map.contains(name))
        m_type_map[name]->delete_data(arg);
}

} // namespace PySide

