/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __QCONTAINER_INDEXING_SUIT_HPP__
#define __QCONTAINER_INDEXING_SUIT_HPP__

#include <boost_headers.hpp>

namespace PySide
{

// Forward declaration
template <class Container, bool NoProxy, class DerivedPolicies>
class PYSIDE_LOCAL qcontainer_indexing_suite;

template <class Container, bool NoProxy>
class PYSIDE_LOCAL final_qcontainer_derived_policies
    : public qcontainer_indexing_suite<Container, NoProxy,
                    final_qcontainer_derived_policies<Container, NoProxy> >
{};

template <class Container,
          bool NoProxy = false,
          class DerivedPolicies = final_qcontainer_derived_policies<Container,
                                                                    NoProxy> >
class PYSIDE_LOCAL qcontainer_indexing_suite
    : public boost::python::indexing_suite <Container,
                                            DerivedPolicies,
                                            NoProxy>
{
public:
    typedef typename Container::value_type data_type;
    typedef typename Container::value_type key_type;
    typedef typename Container::size_type index_type;
    typedef typename Container::size_type size_type;
    typedef typename Container::difference_type difference_type;

    static typename boost::mpl::if_<boost::is_class<data_type>,
                                    data_type&,
                                    data_type>::type
    get_item(Container& container, index_type i)
    {
        return container[i];
    }

    static boost::python::object
    get_slice(Container& container, index_type from, index_type to)
    {
        if (from > to)
            return boost::python::object(Container());

        return boost::python::object(container.mid(from, to-from));
    }

    static void
    set_item(Container& container, index_type i, data_type const& v)
    {
        container[i] = v;
    }

    static void
    set_slice(Container& container, index_type from,
              index_type to, data_type const& v)
    {
        if (from > to)
            return;

        container.erase(container.begin()+from, container.end()+to);
        container.insert(container.begin()+from, v);
    }

    template <class Iter>
    static void
    set_slice(Container& container, index_type from,
              index_type to, Iter first, Iter last)
    {
        if (from > to) {
            std::copy(first, last, container.begin()+from);
        } else {
            container.erase(container.begin()+from, container.end()+to);
            std::copy(first, last, container.begin()+from);
        }
    }

    static void
    delete_item(Container& container, index_type i)
    {
        container.erase(container.begin()+i);
    }

    static void
    delete_slice(Container& container, index_type from, index_type to)
    {
        if (from > to)
            return;

        container.erase(container.begin()+from, container.begin()+to);
    }

    static size_t
    size(Container& container)
    {
        return container.size();
    }

    static bool
    contains(Container& container, key_type const& key)
    {
        return std::find(container.begin(), container.end(), key)
               != container.end();
    }

    static index_type
    convert_index(Container& container, PyObject* i_)
    {
        boost::python::extract<long> i(i_);
        if (i.check()) {
            long index = i();
            if (index < 0)
                index += DerivedPolicies::size(container);

            if (index >= long(container.size()) || index < 0) {
                PyErr_SetString(PyExc_IndexError, "Index out of range");
                boost::python::throw_error_already_set();
            }
            return index;
        }

        PyErr_SetString(PyExc_TypeError, "Invalid index type");
        boost::python::throw_error_already_set();
        return index_type();
    }

    static index_type
    get_min_index(Container& /*container*/)
    {
        return 0;
    }

    static index_type
    get_max_index(Container& container)
    {
        return container.size();
    }

    static bool
    compare_index(Container& /*container*/, index_type a, index_type b)
    {
        return a < b;
    }

    static void
    append(Container& container, data_type const& v)
    {
        container.push_back(v);
    }
};

} // namespace PySide

#endif

